/*  update.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 1x1 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseSymmetricUpdate1 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00 ;
double   *base0, *colj0, *rowi0, *temp0 ;
int      first, ichv0, ii, jj, kk, mm, ipivot, offset ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_denseSymmetricUpdate1()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n chvJ = %d, nDI = %d, ncolI = %d, npivot = %d"
        "\n iifirst = %d, iilast = %d",
        chvJ->id, nDI, ncolI, npivot, iifirst, iilast) ;
#endif
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, nDI) ;
temp0 = DV_entries(tempDV) ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n npivot = %d, pivotsizes = %p", npivot, pivotsizes) ;
#endif
if ( pivotsizes != NULL ) {
   for ( ipivot = first = offset = 0 ; ipivot < npivot ; ipivot++ ) {
      offset += first * pivotsizes[ipivot] ;
      first  += pivotsizes[ipivot] ;
   }
} else {
   offset = (nDI*(nDI-1))/2 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n nDI = %d, offset = %d", nDI, offset) ;
#endif
}
offset += (iifirst - nDI) * nDI ;
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
#endif
/*
   -------------------
   perform the updates
   -------------------
*/
rowi0 = upperent + offset ;
for ( ii = iifirst ; ii <= iilast ; ii++ ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
/*
   --------------------------------------
   store U(:,ii)^T * D in the temp vector
   --------------------------------------
*/
   if ( pivotsizes == NULL ) {
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         temp0[kk] = rowi0[kk] * diagent[kk] ;
      }
   } else {
      for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            temp0[kk] = rowi0[kk] * diagent[mm] ;
            kk++, mm++ ;
         } else {
            temp0[kk]   = rowi0[kk]   * diagent[mm] 
                        + rowi0[kk+1] * diagent[mm+1] ;
            temp0[kk+1] = rowi0[kk]   * diagent[mm+1] 
                        + rowi0[kk+1] * diagent[mm+2] ;
            kk += 2, mm += 3 ;
         }
      }
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
#endif
/*
   ------------------------------------------------------
   loop over the rows and compute U^T(:,ii) * D * U(:,jj)
   ------------------------------------------------------
*/
   colj0 = rowi0 ;
   for ( jj = ii ; jj < ncolI ; jj++ ) {
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n sum00 = %20.12e adding into location %d", 
              sum00, &base0[locind[jj]] - DChv_entries(chvJ)) ;
#endif
      base0[locind[jj]] -= sum00 ;
      colj0 += nDI ;
   }
   rowi0 += nDI ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 2x2 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseSymmetricUpdate2 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00, sum01, sum10, sum11 ;
double   *base0, *base1, *colj0, *colj1, 
         *rowi0, *rowi1, *temp0, *temp1 ;
int      first, ichv0, ichv1, ii, jj, jloc0, jloc1, 
         kk, mm, ipivot, offset ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_denseSymmetricUpdate2()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, 2*nDI) ;
temp0 = DV_entries(tempDV) ;
temp1 = temp0 + nDI ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n npivot = %d, pivotsizes = %p", npivot, pivotsizes) ;
#endif
if ( pivotsizes != NULL ) {
   for ( ipivot = first = offset = 0 ; ipivot < npivot ; ipivot++ ) {
      offset += first * pivotsizes[ipivot] ;
      first  += pivotsizes[ipivot] ;
   }
} else {
   offset = (nDI*(nDI-1))/2 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n nDI = %d, offset = %d", nDI, offset) ;
#endif
}
offset += (iifirst - nDI) * nDI ;
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
#endif
/*
   -------------------
   perform the updates
   -------------------
*/
rowi0 = upperent + offset ;
for ( ii = iifirst ; ii <= iilast - 1 ; ii += 2 ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
   ichv1 = locind[ii + 1] ;
   base1 = DChv_diagLocation(chvJ, ichv1) - ichv1 ;
   rowi1 = rowi0 + nDI ;
/*
   -------------------------------------------
   store U(:,ii:ii+1)^T * D in the temp vector
   -------------------------------------------
*/
   if ( pivotsizes == NULL ) {
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         temp0[kk] = rowi0[kk] * diagent[kk] ;
         temp1[kk] = rowi1[kk] * diagent[kk] ;
      }
   } else {
      for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            temp0[kk] = rowi0[kk] * diagent[mm] ;
            temp1[kk] = rowi1[kk] * diagent[mm] ;
            kk++, mm++ ;
         } else {
            temp0[kk]   = rowi0[kk]   * diagent[mm] 
                        + rowi0[kk+1] * diagent[mm+1] ;
            temp0[kk+1] = rowi0[kk]   * diagent[mm+1] 
                        + rowi0[kk+1] * diagent[mm+2] ;
            temp1[kk]   = rowi1[kk]   * diagent[mm] 
                        + rowi1[kk+1] * diagent[mm+1] ;
            temp1[kk+1] = rowi1[kk]   * diagent[mm+1] 
                        + rowi1[kk+1] * diagent[mm+2] ;
            kk += 2, mm += 3 ;
         }
      }
   }
/*
   ----------------------------------------------------------------
   loop over the rows and compute U^T(:,ii:ii+1) * D * U(:,jj:jj+1)
   ----------------------------------------------------------------
*/
   colj0 = rowi0 ;
   sum00 = 0.0 ;
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      sum00 += temp0[kk] * colj0[kk] ;
   }
   base0[locind[ii]] -= sum00 ;
   colj0 += nDI ;
   for ( jj = ii + 1 ; jj < ncolI - 1 ; jj += 2 ) {
      colj1 = colj0 + nDI ;
      sum00 = sum01 = sum10 = sum11 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum11 += temp1[kk] * colj1[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      base1[jloc0] -= sum10 ;
      base1[jloc1] -= sum11 ;
      colj0 = colj1 + nDI ;
   }
   if ( jj == ncolI - 1 ) {
      sum00 = sum10 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
      }
      jloc0 = locind[jj] ;
      base0[jloc0] -= sum00 ;
      base1[jloc0] -= sum10 ;
   }
   rowi0 = rowi1 + nDI ;
}
if ( ii == iilast ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
/*
   --------------------------------------
   store U(:,ii)^T * D in the temp vector
   --------------------------------------
*/
   if ( pivotsizes == NULL ) {
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         temp0[kk] = rowi0[kk] * diagent[kk] ;
      }
   } else {
      for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            temp0[kk] = rowi0[kk] * diagent[mm] ;
            kk++, mm++ ;
         } else {
            temp0[kk]   = rowi0[kk]   * diagent[mm] 
                        + rowi0[kk+1] * diagent[mm+1] ;
            temp0[kk+1] = rowi0[kk]   * diagent[mm+1] 
                        + rowi0[kk+1] * diagent[mm+2] ;
            kk += 2, mm += 3 ;
         }
      }
   }
/*
   ------------------------------------------------------
   loop over the rows and compute U^T(:,ii) * D * U(:,jj)
   ------------------------------------------------------
*/
   colj0 = rowi0 ;
   for ( jj = ii ; jj < ncolI - 1 ; jj += 2 ) {
      colj1 = colj0 + nDI ;
      sum00 = sum01 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      colj0 = colj1 + nDI ;
   }
   if ( jj == ncolI - 1 ) {
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
      }
      jloc0 = locind[jj] ;
      base0[jloc0] -= sum00 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 3x3 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseSymmetricUpdate3 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00, sum01, sum02, sum10, sum11, sum12, sum20, sum21, sum22 ;
double   *base0, *base1, *base2, *colj0, *colj1, *colj2,
         *rowi0, *rowi1, *rowi2, *temp0, *temp1, *temp2 ;
int      first, ichv0, ichv1, ichv2, ii, jj, jloc0, jloc1, jloc2, 
         kk, mm, ipivot, offset ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_denseSymmetricUpdate3()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, 3*nDI) ;
temp0 = DV_entries(tempDV) ;
temp1 = temp0 + nDI ;
temp2 = temp1 + nDI ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
#if MYDEBUG > 0
fprintf(stdout, "\n npivot = %d, pivotsizes = %p", npivot, pivotsizes) ;
#endif
if ( pivotsizes != NULL ) {
   for ( ipivot = first = offset = 0 ; ipivot < npivot ; ipivot++ ) {
      offset += first * pivotsizes[ipivot] ;
      first  += pivotsizes[ipivot] ;
   }
} else {
   offset = (nDI*(nDI-1))/2 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n nDI = %d, offset = %d", nDI, offset) ;
#endif
}
offset += (iifirst - nDI) * nDI ;
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
#endif
/*
   -------------------
   perform the updates
   -------------------
*/
rowi0 = upperent + offset ;
for ( ii = iifirst ; ii <= iilast - 2 ; ii += 3 ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
   ichv1 = locind[ii + 1] ;
   base1 = DChv_diagLocation(chvJ, ichv1) - ichv1 ;
   ichv2 = locind[ii + 2] ;
   base2 = DChv_diagLocation(chvJ, ichv2) - ichv2 ;
   rowi1 = rowi0 + nDI ;
   rowi2 = rowi1 + nDI ;
/*
   -------------------------------------------
   store U(:,ii:ii+2)^T * D in the temp vector
   -------------------------------------------
*/
   if ( pivotsizes == NULL ) {
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         temp0[kk] = rowi0[kk] * diagent[kk] ;
         temp1[kk] = rowi1[kk] * diagent[kk] ;
         temp2[kk] = rowi2[kk] * diagent[kk] ;
      }
   } else {
      for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            temp0[kk] = rowi0[kk] * diagent[mm] ;
            temp1[kk] = rowi1[kk] * diagent[mm] ;
            temp2[kk] = rowi2[kk] * diagent[mm] ;
            kk++, mm++ ;
         } else {
            temp0[kk]   = rowi0[kk]   * diagent[mm] 
                        + rowi0[kk+1] * diagent[mm+1] ;
            temp0[kk+1] = rowi0[kk]   * diagent[mm+1] 
                        + rowi0[kk+1] * diagent[mm+2] ;
            temp1[kk]   = rowi1[kk]   * diagent[mm] 
                        + rowi1[kk+1] * diagent[mm+1] ;
            temp1[kk+1] = rowi1[kk]   * diagent[mm+1] 
                        + rowi1[kk+1] * diagent[mm+2] ;
            temp2[kk]   = rowi2[kk]   * diagent[mm] 
                        + rowi2[kk+1] * diagent[mm+1] ;
            temp2[kk+1] = rowi2[kk]   * diagent[mm+1] 
                        + rowi2[kk+1] * diagent[mm+2] ;
            kk += 2, mm += 3 ;
         }
      }
   }
/*
   ----------------------------------------------------------------
   loop over the rows and compute U^T(:,ii:ii+2) * D * U(:,jj:jj+2)
   ----------------------------------------------------------------
*/
   colj0 = rowi0 ;
   colj1 = colj0 + nDI ;
   sum00 = sum01 = sum11 = 0.0 ;
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      sum00 += temp0[kk] * colj0[kk] ;
      sum01 += temp0[kk] * colj1[kk] ;
      sum11 += temp1[kk] * colj1[kk] ;
   }
   base0[locind[ii]]   -= sum00 ;
   base0[locind[ii+1]] -= sum01 ;
   base1[locind[ii+1]] -= sum11 ;
   colj0 = colj1 + nDI ;
   for ( jj = ii + 2 ; jj < ncolI - 2 ; jj += 3 ) {
      colj1 = colj0 + nDI ;
      colj2 = colj1 + nDI ;
      sum00 = sum01 = sum02 = sum10 = sum11 = sum12 
            = sum20 = sum21 = sum22 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum02 += temp0[kk] * colj2[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum11 += temp1[kk] * colj1[kk] ;
         sum12 += temp1[kk] * colj2[kk] ;
         sum20 += temp2[kk] * colj0[kk] ;
         sum21 += temp2[kk] * colj1[kk] ;
         sum22 += temp2[kk] * colj2[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      jloc2 = locind[jj+2] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      base0[jloc2] -= sum02 ;
      base1[jloc0] -= sum10 ;
      base1[jloc1] -= sum11 ;
      base1[jloc2] -= sum12 ;
      base2[jloc0] -= sum20 ;
      base2[jloc1] -= sum21 ;
      base2[jloc2] -= sum22 ;
      colj0 = colj2 + nDI ;
   }
   if ( jj == ncolI - 2 ) {
      colj1 = colj0 + nDI ;
      sum00 = sum01 = sum10 = sum11 = sum20 = sum21 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum11 += temp1[kk] * colj1[kk] ;
         sum20 += temp2[kk] * colj0[kk] ;
         sum21 += temp2[kk] * colj1[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      base1[jloc0] -= sum10 ;
      base1[jloc1] -= sum11 ;
      base2[jloc0] -= sum20 ;
      base2[jloc1] -= sum21 ;
   } else if ( jj == ncolI - 1 ) {
      sum00 = sum10 = sum20 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum20 += temp2[kk] * colj0[kk] ;
      }
      jloc0 = locind[jj] ;
      base0[jloc0] -= sum00 ;
      base1[jloc0] -= sum10 ;
      base2[jloc0] -= sum20 ;
   }
   rowi0 = rowi2 + nDI ;
}
if ( ii == iilast - 1 ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
   ichv1 = locind[ii + 1] ;
   base1 = DChv_diagLocation(chvJ, ichv1) - ichv1 ;
   rowi1 = rowi0 + nDI ;
/*
   -------------------------------------------
   store U(:,ii:ii+1)^T * D in the temp vector
   -------------------------------------------
*/
   if ( pivotsizes == NULL ) {
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         temp0[kk] = rowi0[kk] * diagent[kk] ;
         temp1[kk] = rowi1[kk] * diagent[kk] ;
      }
   } else {
      for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            temp0[kk] = rowi0[kk] * diagent[mm] ;
            temp1[kk] = rowi1[kk] * diagent[mm] ;
            kk++, mm++ ;
         } else {
            temp0[kk]   = rowi0[kk]   * diagent[mm] 
                        + rowi0[kk+1] * diagent[mm+1] ;
            temp0[kk+1] = rowi0[kk]   * diagent[mm+1] 
                        + rowi0[kk+1] * diagent[mm+2] ;
            temp1[kk]   = rowi1[kk]   * diagent[mm] 
                        + rowi1[kk+1] * diagent[mm+1] ;
            temp1[kk+1] = rowi1[kk]   * diagent[mm+1] 
                        + rowi1[kk+1] * diagent[mm+2] ;
            kk += 2, mm += 3 ;
         }
      }
   }
/*
   ----------------------------------------------------------------
   loop over the rows and compute U^T(:,ii:ii+2) * D * U(:,jj:jj+2)
   ----------------------------------------------------------------
*/
   colj0 = rowi0 ;
   colj1 = colj0 + nDI ;
   sum00 = sum01 = sum11 = 0.0 ;
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      sum00 += temp0[kk] * colj0[kk] ;
      sum01 += temp0[kk] * colj1[kk] ;
      sum11 += temp1[kk] * colj1[kk] ;
   }
   base0[locind[ii]]   -= sum00 ;
   base0[locind[ii+1]] -= sum01 ;
   base1[locind[ii+1]] -= sum11 ;
   colj0 = colj1 + nDI ;
   for ( jj = ii + 2 ; jj < ncolI - 2 ; jj += 3 ) {
      colj1 = colj0 + nDI ;
      colj2 = colj1 + nDI ;
      sum00 = sum01 = sum02 = sum10 = sum11 = sum12 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum02 += temp0[kk] * colj2[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum11 += temp1[kk] * colj1[kk] ;
         sum12 += temp1[kk] * colj2[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      jloc2 = locind[jj+2] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      base0[jloc2] -= sum02 ;
      base1[jloc0] -= sum10 ;
      base1[jloc1] -= sum11 ;
      base1[jloc2] -= sum12 ;
      colj0 = colj2 + nDI ;
   }
   if ( jj == ncolI - 2 ) {
      colj1 = colj0 + nDI ;
      sum00 = sum01 = sum10 = sum11 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum11 += temp1[kk] * colj1[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      base1[jloc0] -= sum10 ;
      base1[jloc1] -= sum11 ;
   } else if ( jj == ncolI - 1 ) {
      sum00 = sum10 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
      }
      jloc0 = locind[jj] ;
      base0[jloc0] -= sum00 ;
      base1[jloc0] -= sum10 ;
   }
} else if ( ii == iilast ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
/*
   -----------------------------------------
   store U(:,ii:ii)^T * D in the temp vector
   -----------------------------------------
*/
   if ( pivotsizes == NULL ) {
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         temp0[kk] = rowi0[kk] * diagent[kk] ;
      }
   } else {
      for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
         if ( pivotsizes[ipivot] == 1 ) {
            temp0[kk] = rowi0[kk] * diagent[mm] ;
            kk++, mm++ ;
         } else {
            temp0[kk]   = rowi0[kk]   * diagent[mm] 
                        + rowi0[kk+1] * diagent[mm+1] ;
            temp0[kk+1] = rowi0[kk]   * diagent[mm+1] 
                        + rowi0[kk+1] * diagent[mm+2] ;
            kk += 2, mm += 3 ;
         }
      }
   }
/*
   ----------------------------------------------------------------
   loop over the rows and compute U^T(:,ii:ii+2) * D * U(:,jj:jj+2)
   ----------------------------------------------------------------
*/
   colj0 = rowi0 ;
   for ( jj = ii ; jj < ncolI - 2 ; jj += 3 ) {
      colj1 = colj0 + nDI ;
      colj2 = colj1 + nDI ;
      sum00 = sum01 = sum02 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum02 += temp0[kk] * colj2[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      jloc2 = locind[jj+2] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
      base0[jloc2] -= sum02 ;
      colj0 = colj2 + nDI ;
   }
   if ( jj == ncolI - 2 ) {
      colj1 = colj0 + nDI ;
      sum00 = sum01 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
      }
      jloc0 = locind[jj] ;
      jloc1 = locind[jj+1] ;
      base0[jloc0] -= sum00 ;
      base0[jloc1] -= sum01 ;
   } else if ( jj == ncolI - 1 ) {
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
      }
      jloc0 = locind[jj] ;
      base0[jloc0] -= sum00 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------
   update a DChv object using a sparse front

   created -- 97may15, cca
   -----------------------------------------
*/
void
DChv_sparseSymmetricUpdate (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00, t1, t2 ;
double   *base0, *colj0, *rowi0, *temp0 ;
int      ichv0, ii, jj, kk, mm, ipivot, offset, usizei, usizej ;
int      *indi0, *indj0 ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_sparseSymmetricUpdate()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, nDI) ;
temp0 = DV_entries(tempDV) ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
for ( jj = offset = 0 ; jj < iifirst ; jj++ ) {
   offset += sizesU[jj] ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
#endif
/*
   -------------------
   perform the updates
   -------------------
*/
rowi0 = upperent + offset ;
indi0 = upperind + offset ;
for ( ii = iifirst ; ii <= iilast ; ii++ ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n ii = %d, usizei = %d", ii, sizesU[ii]) ;
#endif
   if ( (usizei = sizesU[ii]) > 0 ) {
      ichv0 = locind[ii] ;
      base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
#if MYDEBUG > 0
      fprintf(stdout, "\n ichv0 = %d, base offset = %d", 
              ichv0, base0 - DChv_entries(chvJ)) ;
#endif
/*
      --------------------------------------
      store U(:,ii)^T * D in the temp vector
      --------------------------------------
*/
      DVzero(nDI, temp0) ;
      for ( kk = 0 ; kk < usizei ; kk++ ) {
         temp0[indi0[kk]] = rowi0[kk] ;
      }
      if ( pivotsizes == NULL ) {
         for ( jj = 0 ; jj < nDI ; jj++ ) {
            temp0[jj] = temp0[jj] * diagent[jj] ;
         }
      } else {
         for ( ipivot = mm = kk = 0 ; ipivot < npivot ; ipivot++ ) {
            if ( pivotsizes[ipivot] == 1 ) {
               temp0[kk] = temp0[kk] * diagent[mm] ;
               kk++, mm++ ;
            } else {
               t1 = temp0[kk] ;
               t2 = temp0[kk+1] ;
               temp0[kk]   = t1 * diagent[mm]   + t2 * diagent[mm+1] ;
               temp0[kk+1] = t1 * diagent[mm+1] + t2 * diagent[mm+2] ;
               kk += 2, mm += 3 ;
            }
         }
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n temp0") ;
      DVfprintf(stdout, nDI, temp0) ;
#endif
/*
      ------------------------------------------------------
      loop over the rows and compute U^T(:,ii) * D * U(:,jj)
      ------------------------------------------------------
*/
      colj0 = rowi0 ;
      indj0 = indi0 ;
      for ( jj = ii ; jj < ncolI ; jj++ ) {
         if ( (usizej = sizesU[jj]) > 0 ) {
            sum00 = 0.0 ;
            for ( kk = 0 ; kk < usizej ; kk++ ) {
               sum00 += temp0[indj0[kk]] * colj0[kk] ;
            }
#if MYDEBUG > 0
            fprintf(stdout, "\n jj = %d, sum = %20.12e", jj, sum00)
;
#endif
            base0[locind[jj]] -= sum00 ;
            colj0 += usizej ;
            indj0 += usizej ;
         }
      }
      rowi0 += usizei ;
      indi0 += usizei ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 1x1 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseNonsymmetricUpdate1 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   double   diagent[],
   double   lowerent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00 ;
double   *base0, *colj0, *rowi0, *temp0 ;
int      coloffset, ichv0, ii, jj, kk, offset, rowoffset ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_denseNonsymmetricUpdate1()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, nDI) ;
temp0 = DV_entries(tempDV) ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
offset = (nDI*(nDI-1))/2 + (iifirst - nDI)*nDI ;
#if MYDEBUG > 0
fprintf(stdout, "\n offset = %d", offset) ;
#endif
/*
   -----------------------------------------------------
   perform the updates to the diagonal and upper entries
   -----------------------------------------------------
*/
rowoffset = offset ;
coloffset = offset ;
rowi0 = lowerent + rowoffset ;
for ( ii = iifirst ; ii <= iilast ; ii++ ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n ii = %d, ichv0 = %d, base = %d",
           ii, ichv0, base0 - DChv_entries(chvJ)) ;
#endif
/*
   ------------------------------------
   store L(ii,:) * D in the temp vector
   ------------------------------------
*/
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      temp0[kk] = rowi0[kk] * diagent[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n ii = %d, rowi0 = %d", ii, rowi0 - lowerent) ;
#endif
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
#endif
/*
   ------------------------------------------------------
   loop over the columns and compute L(ii,:) * D * U(:,jj)
   ------------------------------------------------------
*/
   colj0 = upperent + coloffset ;
   for ( jj = ii ; jj < ncolI ; jj++ ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, colj0 = %d", jj, colj0 - upperent) ;
#endif
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[locind[jj]] - DChv_entries(chvJ)) ;
#endif
      base0[locind[jj]] -= sum00 ;
      colj0 += nDI ;
   }
   rowi0 += nDI ;
   coloffset += nDI ;
}
/*
   ----------------------------------------
   perform the updates to the lower entries
   ----------------------------------------
*/
coloffset = offset ;
rowoffset = offset + nDI ;
colj0 = upperent + coloffset ;
#if MYDEBUG > 0
fprintf(stdout, "\n rowoffset = %d, coloffset = %d",
        rowoffset, coloffset) ;
#endif
for ( jj = iifirst ; jj <= iilast ; jj++ ) {
   ichv0 = locind[jj] ;
   base0 = DChv_diagLocation(chvJ, ichv0) + ichv0 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n jj = %d, ichv0 = %d, base = %d",
           jj, ichv0, base0 - DChv_entries(chvJ)) ;
#endif
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, colj0 = %d", jj, colj0 - upperent) ;
#endif
/*
   ------------------------------------
   store D * U(:,ii) in the temp vector
   ------------------------------------
*/
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      temp0[kk] = diagent[kk] * colj0[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
#endif
/*
   ------------------------------------------------------
   loop over the rows and compute L(ii,:) * D * U(:,jj)
   ------------------------------------------------------
*/
   rowi0 = lowerent + rowoffset ;
   for ( ii = jj + 1 ; ii < ncolI ; ii++ ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n ii = %d, rowi0 = %d", ii, rowi0 - lowerent) ;
#endif
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += rowi0[kk] * temp0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[-locind[ii]] - DChv_entries(chvJ)) ;
#endif
      base0[-locind[ii]] -= sum00 ;
      rowi0 += nDI ;
   }
   rowoffset += nDI ;
   colj0 += nDI ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 2x2 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseNonsymmetricUpdate2 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   double   diagent[],
   double   lowerent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00, sum01, sum10, sum11 ;
double   *base0, *base1, *colj0, *colj1, 
         *rowi0, *rowi1, *temp0, *temp1 ;
int      coloffset, ichv0, ichv1, ii, jj, kk, offset, rowoffset ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_denseNonsymmetricUpdate2()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, 2*nDI) ;
temp0 = DV_entries(tempDV) ;
temp1 = temp0 + nDI ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
offset = (nDI*(nDI-1))/2 + (iifirst - nDI)*nDI ;
#if MYDEBUG > 0
fprintf(stdout, "\n iifirst = %d, iilast = %d, offset = %d", 
        iifirst, iilast, offset) ;
fflush(stdout) ;
#endif
/*
   -----------------------------------------------------
   perform the updates to the diagonal and upper entries
   -----------------------------------------------------
*/
rowoffset = offset ;
coloffset = offset ;
rowi0     = lowerent + rowoffset ;
for ( ii = iifirst ; ii <= iilast - 1 ; ii += 2 ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
   ichv1 = locind[ii+1] ;
   base1 = DChv_diagLocation(chvJ, ichv1) - ichv1 ;
   rowi1 = rowi0 + nDI ;
#if MYDEBUG > 0
   fprintf(stdout, "\n ii = %2d, ichv0 = %d, base0 = %d",
           ii, ichv0, base0 - DChv_entries(chvJ)) ;
   fprintf(stdout, "\n          ichv1 = %d, base1 = %d",
           ichv1, base1 - DChv_entries(chvJ)) ;
   fprintf(stdout, "\n ii = %d, rowi0 = %d, rowi1 = %d", 
           ii, rowi0 - lowerent, rowi1 - lowerent) ;
#endif
/*
   ------------------------------------------
   store L(ii:ii+1,:) * D in the temp vectors
   ------------------------------------------
*/
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      temp0[kk] = rowi0[kk] * diagent[kk] ;
      temp1[kk] = rowi1[kk] * diagent[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
   fprintf(stdout, "\n temp1") ;
   DVfprintf(stdout, nDI, temp1) ;
#endif
/*
   -----------------------------------------------------------------
   loop over the columns and compute L(ii:ii+1,:) * D * U(:,jj:jj+1)
   -----------------------------------------------------------------
*/
   colj0 = upperent + coloffset ;
#if MYDEBUG > 0
   fprintf(stdout, "\n jj = %d, colj0 = %d", ii, colj0 - upperent) ;
#endif
   sum00 = 0.0 ;
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      sum00 += temp0[kk] * colj0[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
           sum00, ii, ii, 
           &base0[locind[ii]] - DChv_entries(chvJ)) ;
#endif
   base0[locind[ii]] -= sum00 ;
   colj0 += nDI ;
   for ( jj = ii + 1 ; jj < ncolI - 1 ; jj += 2 ) {
      colj1 = colj0 + nDI ;
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, colj0 = %d, colj1 = %d", 
              jj, colj0 - upperent, colj1 - upperent) ;
#endif
      sum00 = sum01 = sum10 = sum11 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
         sum11 += temp1[kk] * colj1[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[locind[jj]] - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum01, ii, jj+1, 
              &base0[locind[jj+1]] - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum10, ii+1, jj, 
              &base1[locind[jj]] - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum11, ii+1, jj+1, 
              &base1[locind[jj+1]] - DChv_entries(chvJ)) ;
#endif
      base0[locind[jj]]   -= sum00 ;
      base0[locind[jj+1]] -= sum01 ;
      base1[locind[jj]]   -= sum10 ;
      base1[locind[jj+1]] -= sum11 ;
      colj0 = colj1 + nDI ;
   }
   if ( jj == ncolI - 1 ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, colj0 = %d", jj, colj0 - upperent) ;
#endif
      sum00 = sum10 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum10 += temp1[kk] * colj0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[locind[jj]] - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum10, ii+1, jj, 
              &base1[locind[jj]] - DChv_entries(chvJ)) ;
#endif
      base0[locind[jj]]   -= sum00 ;
      base1[locind[jj]]   -= sum10 ;
   }
   rowi0 = rowi1 + nDI ;
   coloffset += 2*nDI ;
}
if ( ii == iilast ) {
   ichv0 = locind[ii] ;
   base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n ii = %2d, ichv0 = %d, base0 = %d",
           ii, ichv0, base0 - DChv_entries(chvJ)) ;
   fprintf(stdout, "\n ii = %d, rowi0 = %d", ii, rowi0 - lowerent) ;
#endif
/*
   -------------------------------------
   store L(ii,:) * D in the temp vectors
   -------------------------------------
*/
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      temp0[kk] = rowi0[kk] * diagent[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
#endif
/*
   ------------------------------------------------------------
   loop over the columns and compute L(ii,:) * D * U(:,jj:jj+1)
   ------------------------------------------------------------
*/
   colj0 = upperent + coloffset ;
#if MYDEBUG > 0
   fprintf(stdout, "\n jj = %d, colj0 = %d", ii, colj0 - upperent) ;
#endif
   sum00 = 0.0 ;
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      sum00 += temp0[kk] * colj0[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
           sum00, ii, ii, 
           &base0[locind[ii]] - DChv_entries(chvJ)) ;
#endif
   base0[locind[ii]] -= sum00 ;
   colj0 += nDI ;
   for ( jj = ii + 1 ; jj < ncolI - 1 ; jj += 2 ) {
      colj1 = colj0 + nDI ;
      sum00 = sum01 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
         sum01 += temp0[kk] * colj1[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, colj0 = %d, colj1 = %d",
              jj, colj0 - upperent, colj1 - upperent) ;
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[locind[jj]] - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum01, ii, jj+1, 
              &base0[locind[jj+1]] - DChv_entries(chvJ)) ;
#endif
      base0[locind[jj]]   -= sum00 ;
      base0[locind[jj+1]] -= sum01 ;
      colj0 = colj1 + nDI ;
   }
   if ( jj == ncolI - 1 ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, colj0 = %d", jj, colj0 - upperent) ;
#endif
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += temp0[kk] * colj0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[locind[jj]] - DChv_entries(chvJ)) ;
#endif
      base0[locind[jj]]   -= sum00 ;
   }
}
/*
   ----------------------------------------
   perform the updates to the lower entries
   ----------------------------------------
*/
coloffset = offset ;
rowoffset = offset + nDI ;
colj0     = upperent + coloffset ;
#if MYDEBUG > 0
fprintf(stdout, "\n rowoffset = %d, coloffset = %d",
        rowoffset, coloffset) ;
#endif
for ( jj = iifirst ; jj <= iilast - 1 ; jj += 2 ) {
   ichv0 = locind[jj] ;
   base0 = DChv_diagLocation(chvJ, ichv0) + ichv0 ;
   ichv1 = locind[jj+1] ;
   base1 = DChv_diagLocation(chvJ, ichv1) + ichv1 ;
   colj1 = colj0 + nDI ;
#if MYDEBUG > 0
   fprintf(stdout, "\n jj = %d, ichv0 = %d, base = %d",
           jj, ichv0, base0 - DChv_entries(chvJ)) ;
   fprintf(stdout, "\n          ichv1 = %d, base = %d",
           ichv1, base1 - DChv_entries(chvJ)) ;
   fprintf(stdout, "\n colj0 = %d, colj1 = %d",
           colj0 - upperent, colj1 - upperent) ;
#endif
/*
   ------------------------------------
   store D * U(:,ii) in the temp vector
   ------------------------------------
*/
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      temp0[kk] = diagent[kk] * colj0[kk] ;
      temp1[kk] = diagent[kk] * colj1[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
#endif
/*
   ------------------------------------------------------
   loop over the rows and compute L(ii,:) * D * U(:,jj)
   ------------------------------------------------------
*/
   rowi0 = lowerent + rowoffset ;
#if MYDEBUG > 0
   fprintf(stdout, "\n initial rowi0 = %d", rowi0 - lowerent) ;
#endif
#if MYDEBUG > 0
   fprintf(stdout, "\n initial rowi0") ;
   DVfprintf(stdout, nDI, rowi0) ;
#endif
   sum00 = 0.0 ;
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      sum00 += rowi0[kk] * temp0[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n ii = %d, sum00 = %20.12e", ii, sum00) ;
   fprintf(stdout, ", adding to location %d",
           &base0[-locind[jj+1]] - DChv_entries(chvJ)) ;
#endif
   base0[-locind[jj+1]] -= sum00 ;
   rowi0 += nDI ;
   for ( ii = jj + 2 ; ii < ncolI - 1 ; ii += 2 ) {
      rowi1 = rowi0 + nDI ;
#if MYDEBUG > 0
      fprintf(stdout, "\n ii = %d, rowi0 = %d, rowi1 = %d",
              ii, rowi0 - lowerent, rowi1 - lowerent) ;
#endif
      sum00 = sum01 = sum10 = sum11 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += rowi0[kk] * temp0[kk] ;
         sum01 += rowi0[kk] * temp1[kk] ;
         sum10 += rowi1[kk] * temp0[kk] ;
         sum11 += rowi1[kk] * temp1[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n adding %20.12e to (%d,%d), location %d"
              "\n adding %20.12e to (%d,%d), location %d"
              "\n adding %20.12e to (%d,%d), location %d"
              "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[-locind[ii]] - DChv_entries(chvJ),
              sum01, ii, jj, 
              &base1[-locind[ii]] - DChv_entries(chvJ),
              sum10, ii+1, jj, 
              &base0[-locind[ii+1]] - DChv_entries(chvJ),
              sum11, ii+1, jj, 
              &base1[-locind[ii+1]] - DChv_entries(chvJ)) ;
#endif
      base0[-locind[ii]]   -= sum00 ;
      base1[-locind[ii]]   -= sum01 ;
      base0[-locind[ii+1]] -= sum10 ;
      base1[-locind[ii+1]] -= sum11 ;
      rowi0 = rowi1 + nDI ;
   }
   if ( ii == ncolI - 1 ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n ii = %d, rowi0 = %d", ii, rowi0 - lowerent) ;
#endif
      sum00 = sum01 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += rowi0[kk] * temp0[kk] ;
         sum01 += rowi0[kk] * temp1[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n adding %20.12e to (%d,%d), location %d"
              "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[-locind[ii]] - DChv_entries(chvJ),
              sum01, ii, jj, 
              &base1[-locind[ii]] - DChv_entries(chvJ)) ;
#endif
      base0[-locind[ii]]   -= sum00 ;
      base1[-locind[ii]]   -= sum01 ;
   }
   rowoffset += 2*nDI ;
   colj0 = colj1 + nDI ;
}
if ( jj == iilast ) {
   ichv0 = locind[jj] ;
   base0 = DChv_diagLocation(chvJ, ichv0) + ichv0 ;
#if MYDEBUG > 0
   fprintf(stdout, "\n jj = %d, ichv0 = %d, base = %d",
           jj, ichv0, base0 - DChv_entries(chvJ)) ;
   fprintf(stdout, "\n colj0 = %d", colj0 - upperent) ;
#endif
/*
   ------------------------------------
   store D * U(:,ii) in the temp vector
   ------------------------------------
*/
   for ( kk = 0 ; kk < nDI ; kk++ ) {
      temp0[kk] = diagent[kk] * colj0[kk] ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n temp0") ;
   DVfprintf(stdout, nDI, temp0) ;
#endif
/*
   ------------------------------------------------------
   loop over the rows and compute L(ii,:) * D * U(:,jj)
   ------------------------------------------------------
*/
   rowi0 = lowerent + rowoffset ;
   for ( ii = jj + 1 ; ii < ncolI - 1 ; ii += 2 ) {
      rowi1 = rowi0 + nDI ;
#if MYDEBUG > 0
      fprintf(stdout, "\n ii = %d, rowi0 = %d, rowi1 = %d",
              ii, rowi0 - lowerent, rowi1 - lowerent) ;
#endif
      sum00 = sum10 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += rowi0[kk] * temp0[kk] ;
         sum10 += rowi1[kk] * temp0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n adding %20.12e to (%d,%d), location %d"
              "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[-locind[ii]] - DChv_entries(chvJ),
              sum10, ii+1, jj, 
              &base0[-locind[ii+1]] - DChv_entries(chvJ)) ;
#endif
      base0[-locind[ii]]   -= sum00 ;
      base0[-locind[ii+1]] -= sum10 ;
      rowi0 = rowi1 + nDI ;
   }
   if ( ii == ncolI - 1 ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n ii = %d, rowi0 = %d", ii, rowi0 - lowerent) ;
#endif
      sum00 = 0.0 ;
      for ( kk = 0 ; kk < nDI ; kk++ ) {
         sum00 += rowi0[kk] * temp0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n adding %20.12e to (%d,%d), location %d",
              sum00, ii, jj, 
              &base0[-locind[ii]] - DChv_entries(chvJ)) ;
#endif
      base0[-locind[ii]]   -= sum00 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------
   update a DChv object using a sparse front

   created -- 97may15, cca
   -----------------------------------------
*/
void
DChv_sparseNonsymmetricUpdate (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   double   diagent[],
   int      sizesL[],
   int      lowerind[],
   double   lowerent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) {
double   sum00 ;
double   *base0, *colj0, *rowi0, *temp0 ;
int      coloffset, ichv0, ierr, ii, jj, kk, lsizei, 
         offsetL, offsetU, rowoffset, usizej ;
int      *indi0, *indj0 ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_denseNonsymmetricUpdate3()"
           "\n chvJ is NULL\n") ;
   exit(-1) ;
}
/*
   -----------------------
   set the working storage
   -----------------------
*/
DV_setSize(tempDV, nDI) ;
temp0 = DV_entries(tempDV) ;
/*
   --------------------------------------
   get offset to the start of the entries
   --------------------------------------
*/
for ( ii = offsetL = offsetU = 0 ; ii < iifirst ; ii++ ) { 
   offsetL += sizesL[ii] ;
   offsetU += sizesU[ii] ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n offsetL = %d, offsetU = %d", 
        offsetL, offsetU) ;
#endif
/*
   -----------------------------------------------------
   perform the updates to the diagonal and upper entries
   -----------------------------------------------------
*/
rowi0 = lowerent + offsetL ;
indi0 = lowerind + offsetL ;
coloffset = offsetU ;
for ( ii = iifirst ; ii <= iilast ; ii++ ) {
   if ( (lsizei = sizesL[ii]) > 0 ) {
      ichv0 = locind[ii] ;
      base0 = DChv_diagLocation(chvJ, ichv0) - ichv0 ;
#if MYDEBUG > 0
      fprintf(stdout, "\n ii = %d, ichv0 = %d, base = %d",
              ii, ichv0, base0 - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n lsizei = %d, indi0", lsizei) ;
      IVfp80(stdout, lsizei, indi0, 80, &ierr) ;
      fprintf(stdout, "\n rowi0") ;
      DVfprintf(stdout, lsizei, rowi0) ;
#endif
/*
      ------------------------------------
      store L(ii,:) * D in the temp vector
      ------------------------------------
*/
      DVzero(nDI, temp0) ;
      for ( kk = 0 ; kk < lsizei ; kk++ ) {
         temp0[indi0[kk]] = rowi0[kk] * diagent[indi0[kk]] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n temp0") ;
      DVfprintf(stdout, nDI, temp0) ;
#endif
/*
      ------------------------------------------------------
      loop over the columns and compute L(ii,:) * D * U(:,jj)
      ------------------------------------------------------
*/
      colj0 = upperent + coloffset ;
      indj0 = upperind + coloffset ;
      for ( jj = ii ; jj < ncolI ; jj++ ) {
         if ( (usizej = sizesU[jj]) > 0 ) {
#if MYDEBUG > 0
            fprintf(stdout, "\n usizej = %d, indj0", usizej) ;
            IVfp80(stdout, usizej, indj0, 80, &ierr) ;
            fprintf(stdout, "\n colj0") ;
            DVfprintf(stdout, usizej, colj0) ;
#endif
            sum00 = 0.0 ;
            for ( kk = 0 ; kk < usizej ; kk++ ) {
               sum00 += temp0[indj0[kk]] * colj0[kk] ;
            }
#if MYDEBUG > 0
            fprintf(stdout, "\n jj = %d, sum00 = %20.12e", jj, sum00) ;
            fprintf(stdout, ", adding to location %d",
                    base0 - DChv_entries(chvJ) + locind[jj]) ;
#endif
            base0[locind[jj]] -= sum00 ;
            colj0 += usizej ;
            indj0 += usizej ;
         }
      }
      rowi0 += lsizei ;
      indi0 += lsizei ;
   }
   coloffset += sizesU[ii] ;
}
/*
   ----------------------------------------
   perform the updates to the lower entries
   ----------------------------------------
*/
colj0 = upperent + offsetU ;
indj0 = upperind + offsetU ;
rowoffset = offsetL ;
for ( jj = iifirst ; jj <= iilast ; jj++ ) {
   rowoffset += sizesL[jj] ;
   if ( (usizej = sizesU[jj]) > 0 ) {
      ichv0 = locind[jj] ;
      base0 = DChv_diagLocation(chvJ, ichv0) + ichv0 ;
#if MYDEBUG > 0
      fprintf(stdout, "\n jj = %d, ichv0 = %d, base = %d",
              jj, ichv0, base0 - DChv_entries(chvJ)) ;
      fprintf(stdout, "\n usizej = %d, indj0", usizej) ;
      IVfp80(stdout, usizej, indj0, 80, &ierr) ;
      fprintf(stdout, "\n colj0") ;
      DVfprintf(stdout, usizej, colj0) ;
#endif
/*
      ------------------------------------
      store D * U(:,ii) in the temp vector
      ------------------------------------
*/
      DVzero(nDI, temp0) ;
      for ( kk = 0 ; kk < usizej ; kk++ ) {
         temp0[indj0[kk]] = diagent[indj0[kk]] * colj0[kk] ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n temp0") ;
      DVfprintf(stdout, nDI, temp0) ;
#endif
/*
      ------------------------------------------------------
      loop over the rows and compute L(ii,:) * D * U(:,jj)
      ------------------------------------------------------
*/
      rowi0 = lowerent + rowoffset ;
      indi0 = lowerind + rowoffset ;
      for ( ii = jj + 1 ; ii < ncolI ; ii++ ) {
         if ( (lsizei = sizesL[ii]) > 0 ) {
#if MYDEBUG > 0
            fprintf(stdout, "\n lsizei = %d, indi0", lsizei) ;
            IVfp80(stdout, lsizei, indi0, 80, &ierr) ;
            fprintf(stdout, "\n indi0") ;
            DVfprintf(stdout, lsizei, rowi0) ;
#endif
            sum00 = 0.0 ;
            for ( kk = 0 ; kk < lsizei ; kk++ ) {
               sum00 += rowi0[kk] * temp0[indi0[kk]] ;
            }
#if MYDEBUG > 0
            fprintf(stdout, "\n ii = %d, sum00 = %20.12e", ii, sum00) ;
            fprintf(stdout, ", adding to location %d",
                    base0 - DChv_entries(chvJ) - locind[ii]) ;
#endif
            base0[-locind[ii]] -= sum00 ;
            rowi0 += lsizei ;
            indi0 += lsizei ;
         }
      }
      colj0 += usizej ;
      indj0 += usizej ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
