/*  testPivot.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------
   return value --
      1 if pivot passes the stability test
      0 if pivot fails the stability test

   created -- 97apr26, cca
   ---------------------------------------
*/
int
DChv_testPivot (
   DChv     *chv,
   int      irow,
   int      jcol,
   double   tau 
) {
double   ark, arr, ars, ask, ass, cutoff, val ;
double   *entries ;
int      ii, itmp, ioff, jj, joff, nD, nL, nU, off, stride, success ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || irow < 0 || jcol < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_testPivot(%p,%d,%d,%f)"
           "\n bad input\n", chv, irow, jcol, tau) ;
   exit(-1) ;
}
nD      = chv->nD      ;
nL      = chv->nL      ;
nU      = chv->nU      ;
#if MYDEBUG > 0
fprintf(stdout, 
        "\n inside DChv_testPivot, id = %d, nD = %d, nL = %d, nU = %d",
        chv->id, nD, nL, nU) ;
fflush(stdout) ;
#endif
entries = chv->entries ;
success = 1 ;
if ( chv->symflag == 0 ) {
#if MYDEBUG > 0
   fprintf(stdout, 
           "\n symmetric chevron, irow = %d, jcol = %d, tau = %f",
           irow, jcol, tau) ;
   fflush(stdout) ;
#endif
/*
   --------------------
   chevron is symmetric
   --------------------
*/
   if ( irow == jcol ) {
/*
      --------------
      diagonal pivot
      --------------
*/
      arr     = fabs(entries[irow*(nD + nU) - (irow*(irow-1))/2]) ;
      cutoff  = tau * arr ;
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n pivot is a(%d,%d) = %20.12e, cutoff = %12.4e",
              irow, jcol, arr, cutoff) ;
#endif
      off     = irow ;
      stride  = nD + nU - 1 ;
      for ( ii = 0 ; ii < irow ; ii++ ) {
         ark = fabs(entries[off]) ;
#if MYDEBUG > 0
         fprintf(stdout, "\n a(%d,%d) = %20.12e", ii, irow, ark) ;
#endif
         if ( ark > cutoff ) {
#if MYDEBUG > 0
            fprintf(stdout, ", pivot fails") ;
#endif
            success = 0 ;
            break ;
         }
         off += stride ;
         stride-- ;
      }
      if ( success == 1 ) {
         off++ ;
         for ( ii = irow + 1 ; ii < nD + nU ; ii++ ) {
            ark = fabs(entries[off]) ;
#if MYDEBUG > 0
         fprintf(stdout, "\n a(%d,%d) = %20.12e", irow, ii, ark) ;
#endif
            if ( ark > cutoff ) {
#if MYDEBUG > 0
               fprintf(stdout, ", pivot fails") ;
#endif
               success = 0 ;
               break ;
            }
            off++ ;
         }
      }
   } else {
/*
      -----------
      2 x 2 pivot
      -----------
*/
      if ( irow > jcol ) {
         itmp = irow ;
         irow = jcol ;
         jcol = itmp ;
      }
/*
      arr = fabs(entries[irow*(nD + nU) - (irow*(irow-1))/2]) ;
      ars = fabs(entries[irow*(nD + nU) - (irow*(irow-1))/2 
                         + jcol - irow]) ;
      ass = fabs(entries[jcol*(nD + nU) - (jcol*(jcol-1))/2]) ;
*/
      arr = entries[irow*(nD + nU) - (irow*(irow-1))/2] ;
      ars = entries[irow*(nD + nU) - (irow*(irow-1))/2 + jcol - irow] ;
      ass = entries[jcol*(nD + nU) - (jcol*(jcol-1))/2] ;
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n irow = %d, jcol = %d"
              "\n 2 x 2 pivot [ %20.12e %20.12e ]"
              "\n             [ %20.12e %20.12e ]",
              irow, jcol, arr, ars, ars, ass) ;
#endif
      if ( arr*ass - ars*ars == 0.0 ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n pivot is singular") ;
#endif
         success = 0 ;
         return(0) ;
      }
      if ( fabs(arr) >= fabs(ass) ) {
         cutoff  = tau * fabs(arr*ass-ars*ars)/sqrt(ars*ars+arr*arr) ;
      } else {
         cutoff  = tau * fabs(arr*ass-ars*ars)/sqrt(ars*ars+ass*ass) ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n cutoff = %12.4e", cutoff) ;
#endif
      ioff    = irow ;
      joff    = jcol ;
      stride  = nD + nU - 1 ;
      for ( ii = 0 ; ii < irow ; ii++ ) {
         ark = entries[ioff] ;
         ask = entries[joff] ;
#if MYDEBUG > 0
         fprintf(stdout, 
     "\n ioff = %d, a(%d,%d) = %12.4e, joff = %d, a(%d,%d) = %12.4e", 
     ioff, ii, irow, ark, joff, ii, jcol, ask) ;
#endif
         if ( sqrt(ark*ark + ask*ask) > cutoff ) {
#if MYDEBUG > 0
            fprintf(stdout, ", pivot fails") ;
#endif
            success = 0 ;
            break ;
         }
         ioff += stride ;
         joff += stride ;
         stride-- ;
      }
      if ( success == 1 ) {
         ioff++ ; joff += stride ; stride-- ;
         for ( ii = irow + 1 ; ii < jcol ; ii++ ) {
            ark = entries[ioff] ;
            ask = entries[joff] ;
#if MYDEBUG > 0
         fprintf(stdout, 
     "\n ioff = %d, a(%d,%d) = %12.4e, joff = %d, a(%d,%d) = %12.4e", 
     ioff, irow, ii, ark, joff, ii, jcol, ask) ;
#endif
            if ( sqrt(ark*ark + ask*ask) > cutoff ) {
#if MYDEBUG > 0
               fprintf(stdout, ", pivot fails") ;
#endif
               success = 0 ;
               break ;
            }
            ioff++ ;
            joff += stride ;
            stride-- ;
         }
      }
      if ( success == 1 ) {
         ioff++ ; joff++ ;
         for ( ii = jcol + 1 ; ii < nD + nU ; ii++ ) {
            ark = entries[ioff] ;
            ask = entries[joff] ;
#if MYDEBUG > 0
         fprintf(stdout, 
     "\n ioff = %d, a(%d,%d) = %12.4e, joff = %d, a(%d,%d) = %12.4e", 
     ioff, irow, ii, ark, joff, jcol, ii, ask) ;
#endif
            if ( sqrt(ark*ark + ask*ask) > cutoff ) {
#if MYDEBUG > 0
               fprintf(stdout, ", pivot fails") ;
#endif
               success = 0 ;
               break ;
            }
            ioff++ ;
            joff++ ;
         }
      }
   }
} else {
/*
   ------------------------------------
   chevron is nonsymmetric, 1 x 1 pivot
   ------------------------------------
*/
   ii = (irow <= jcol) ? irow : jcol ;
   off = jcol - irow ;
   arr = fabs(entries[(2*ii+1)*nD + (ii+1)*nL + ii*nU 
                      - ii*ii - ii - 1 + off]) ;
   cutoff  = tau * arr ;
#if MYDEBUG > 0
   fprintf(stdout, 
      "\n tau = %.3g, nonsymmetric chevron\n"
      "\n pivot is a(%d,%d) = %20.12e, cutoff = %12.4e",
      tau, irow, jcol, arr, cutoff) ;
   DChv_writeForHumanEye(chv, stdout) ;
#endif
/*
   ----------------------------
   check out entries in the row
   ----------------------------
*/
   off = nD + nL - 1 - irow ;
   stride = 2*nD + nL + nU - 1 ;
   for ( jj = 0 ; jj < irow ; jj++ ) {
      val = fabs(entries[off]) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n off = %d, a(%d,%d) = %20.12e", 
              off, irow, jj, val) ;
#endif
      if ( val > cutoff ) {
#if MYDEBUG > 0
         fprintf(stdout, ", pivot fails") ;
#endif
         success = 0 ;
         break ;
      }
      off += stride ;
      stride -= 2 ;
   }
   if ( success == 1 ) {
      for ( jj = irow ; jj < nD + nU ; jj++ ) {
         val = fabs(entries[off]) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n off = %d, a(%d,%d) = %20.12e", 
              off, irow, jj, val) ;
#endif
         if ( val > cutoff ) {
#if MYDEBUG > 0
            fprintf(stdout, ", pivot fails") ;
#endif
            success = 0 ;
            break ;
         }
         off++ ;
      }
   }
   if ( success == 1 ) {
/*
      -------------------------------
      check out entries in the column
      -------------------------------
*/
      off    = nD + nL + jcol - 1 ;
      stride = 2*nD + nL + nU - 3 ;
      for ( ii = 0 ; ii < jcol ; ii++ ) {
         val = fabs(entries[off]) ;
#if MYDEBUG > 0
         fprintf(stdout, "\n off = %d, a(%d,%d) = %20.12e", 
                 off, ii, jcol, val) ;
#endif
         if ( val > cutoff ) {
#if MYDEBUG > 0
            fprintf(stdout, ", pivot fails") ;
#endif
            success = 0 ;
            break ;
         }
         off += stride ;
         stride -= 2 ;
      }
   }
   if ( success == 1 ) {
      for ( ii = jcol ; ii < nD + nL ; ii++ ) {
         val = fabs(entries[off]) ;
#if MYDEBUG > 0
         fprintf(stdout, "\n off = %d, a(%d,%d) = %20.12e", 
                 off, ii, jcol, val) ;
#endif
         if ( val > cutoff ) {
#if MYDEBUG > 0
            fprintf(stdout, ", pivot fails") ;
#endif
            success = 0 ;
            break ;
         }
         off-- ;
      }
   }
}
return(success) ; }

/*--------------------------------------------------------------------*/
