/*  swap.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   -----------------------
   swap rows irow and jrow

   created -- 97apr24, cca
   -----------------------
*/
void
DChv_swapRows (
   DChv   *chv,
   int    irow,
   int    jrow
) {
double   dtmp ;
double   *entries ;
int      ierr, ii, ioff, itmp, joff, nD, nL, nrow, nU, stride ;
int      *rowind ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || irow < 0 || jrow < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_swapRows(%p,%d,%d)"
           "\n bad input\n", chv, irow, jrow) ;
   exit(-1) ;
}
#if MYDEBUG > 0
fprintf(stdout,"\n DChv_swapRows(%p,%d,%d)", chv, irow, jrow) ;
#endif
if ( irow == jrow ) {
   return ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
if ( irow >= nD || jrow >= nD ) {
   fprintf(stderr, "\n fatal error in DChv_swapRows(%p,%d,%d)"
           "\n rows must be less than nD = %d", chv, irow, irow, nD) ;
   exit(-1) ;
}
entries = DChv_entries(chv) ;
if ( entries == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_swapRows(%p,%d,%d)"
           "\n bad input, entries = %p, nD = %d\n", 
           chv, irow, jrow, entries, nD) ;
   exit(-1) ;
}
if ( chv->symflag == 0 ) {
/*
   ----------------------------------
   call method for symmetric chevrons
   ----------------------------------
*/
   DChv_swapRowsAndColumns(chv, irow, jrow) ;
} else {
/*
   ------------------------
   swap the two row indices
   ------------------------
*/
   DChv_rowIndices(chv, &nrow, &rowind) ;
#if MYDEBUG > 0
   fprintf(stdout,
           "\n before: rowind = %p, rowind[%d] = %d, rowind[%d] = %d",
           rowind, irow, rowind[irow], jrow, rowind[jrow]) ;
   IVfp80(stdout, nrow, rowind, 80, &ierr) ;
#endif
   itmp         = rowind[irow] ;
   rowind[irow] = rowind[jrow] ;
   rowind[jrow] = itmp         ;
#if MYDEBUG > 0
   fprintf(stdout,
           "\n after: rowind = %p, rowind[%d] = %d, rowind[%d] = %d",
           rowind, irow, rowind[irow], jrow, rowind[jrow]) ;
   IVfp80(stdout, nrow, rowind, 80, &ierr) ;
#endif
/*
   --------------------------------------------------
   set irow = min(irow, jrow), jrow = max(irow, jrow)
   --------------------------------------------------
*/
   if ( irow > jrow ) {
      itmp = irow ;
      irow = jrow ;
      jrow = itmp ;
   }
/*
   --------------------------------
   swap the entries in the two rows
   --------------------------------
*/
   ioff   = nD + nL - 1 - irow ;
   joff   = nD + nL - 1 - jrow ;
   stride = 2*nD + nL + nU - 1 ;
   for ( ii = 0 ; ii < irow ; ii++ ) {
      dtmp           = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += stride, joff += stride, stride -= 2 ;
   }
   for ( ii = irow ; ii < jrow ; ii++ ) {
      dtmp           = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += 1, joff += stride, stride -= 2 ;
   }
   for ( ii = jrow ; ii < nD + nU ; ii++ ) {
      dtmp           = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += 1, joff += 1 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------
   swap columns icol and jcol

   created -- 97apr24, cca
   --------------------------
*/
void
DChv_swapColumns (
   DChv   *chv,
   int      icol,
   int      jcol
) {
double   dtmp ;
double   *entries ;
int      ii, ioff, itmp, joff, ncol, nD, nL, nU, stride ;
int      *colind ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || icol < 0 || jcol < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_swapColumns(%p,%d,%d)"
           "\n bad input\n", chv, icol, jcol) ;
   exit(-1) ;
}
if ( icol == jcol ) {
   return ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
entries = DChv_entries(chv) ;
if ( entries == NULL || icol >= nD || jcol >= nD ) {
   fprintf(stderr, "\n fatal error in DChv_swapColumns(%p,%d,%d)"
           "\n bad input, entries = %p, nD = %d\n", 
           chv, icol, jcol, entries, nD) ;
   exit(-1) ;
}
if ( chv->symflag == 0 ) {
/*
   ----------------------------------
   call method for symmetric chevrons
   ----------------------------------
*/
   DChv_swapRowsAndColumns(chv, icol, jcol) ;
} else {
/*
   ---------------------------
   swap the two column indices
   ---------------------------
*/
   DChv_columnIndices(chv, &ncol, &colind) ;
   itmp         = colind[icol] ;
   colind[icol] = colind[jcol] ;
   colind[jcol] = itmp         ;
/*
   --------------------------------------------------
   set icol = min(icol, jcol), jcol = max(icol, jcol)
   --------------------------------------------------
*/
   if ( icol > jcol ) {
      itmp = icol ;
      icol = jcol ;
      jcol = itmp ;
   }
/*
   -----------------------------------
   swap the entries in the two columns
   -----------------------------------
*/
   ioff   = nD + nL - 1 + icol ;
   joff   = nD + nL - 1 + jcol ;
   stride = 2*nD + nL + nU - 3 ;
   for ( ii = 0 ; ii < icol ; ii++ ) {
      dtmp          = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += stride, joff += stride, stride -= 2 ;
   }
   for ( ii = icol ; ii < jcol ; ii++ ) {
      dtmp          = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff -= 1, joff += stride, stride -= 2 ;
   }
   for ( ii = jcol ; ii < nD + nL ; ii++ ) {
      dtmp          = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff -= 1, joff -= 1 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------
   swap rows and columns ii and jj

   created -- 97apr24, cca
   -------------------------------
*/
void
DChv_swapRowsAndColumns (
   DChv   *chv,
   int    ii,
   int    jj
) {
double   dtmp ;
double   *entries ;
int      iiloc, ioff, itmp, jjloc, joff, kk, ncol, nD, nL, nU, stride ;
int      *colind ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || ii < 0 || jj < 0 ) {
   fprintf(stderr, 
           "\n fatal error in DChv_swapRowsAndColumns(%p,%d,%d)"
           "\n bad input\n", chv, ii, jj) ;
   exit(-1) ;
}
if ( ii == jj ) {
   return ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
entries = DChv_entries(chv) ;
if ( entries == NULL || ii >= nD || jj >= nD ) {
   fprintf(stderr, 
           "\n fatal error in DChv_swapRowsAndColumns(%p,%d,%d)"
           "\n bad input, entries = %p, nD = %d\n", 
           chv, ii, jj, entries, nD) ;
   exit(-1) ;
}
if ( chv->symflag == 1 ) {
/*
   --------------------------------------
   call methods for nonsymmetric chevrons
   --------------------------------------
*/
   DChv_swapRows(chv, ii, jj) ;
   DChv_swapColumns(chv, ii, jj) ;
} else {
/*
   ---------------------------
   swap the two column indices
   ---------------------------
*/
   DChv_columnIndices(chv, &ncol, &colind) ;
   itmp       = colind[ii] ;
   colind[ii] = colind[jj] ;
   colind[jj] = itmp       ;
/*
   --------------------------------------
   set ii = min(ii, jj), jj = max(ii, jj)
   --------------------------------------
*/
   if ( ii > jj ) {
      itmp = ii   ;
      ii   = jj   ;
      jj   = itmp ;
   }
/*
   --------------------------------------------
   swap the entries in the two rows and columns
   --------------------------------------------
*/
   ioff   = ii ;
   joff   = jj ;
   stride = nD + nU - 1 ;
   for ( kk = 0 ; kk < ii ; kk++ ) {
#if MYDEBUG > 0
      fprintf(stdout, 
         "\n\n first step, kk = %d, ioff = %d, joff = %d, stride = %d", 
         kk, ioff, joff, stride) ;
#endif
      dtmp           = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += stride, joff += stride, stride -= 1 ;
   }
   iiloc = ioff ;
   ioff++, joff += stride, stride -= 1 ;
   for ( kk = ii + 1 ; kk < jj ; kk++ ) {
#if MYDEBUG > 0
      fprintf(stdout, 
        "\n\n second step, kk = %d, ioff = %d, joff = %d, stride = %d", 
        kk, ioff, joff, stride) ;
#endif
      dtmp           = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += 1, joff += stride, stride -= 1 ;
   }
   jjloc = joff ;
   dtmp            = entries[iiloc] ;
   entries[iiloc] = entries[jjloc] ;
   entries[jjloc] = dtmp            ;
   ioff++, joff++ ;
   for ( kk = jj + 1 ; kk < nD + nU ; kk++ ) {
#if MYDEBUG > 0
      fprintf(stdout, 
        "\n\n third step, kk = %d, ioff = %d, joff = %d, stride = %d", 
        kk, ioff, joff, stride) ;
#endif
      dtmp           = entries[ioff] ;
      entries[ioff] = entries[joff] ;
      entries[joff] = dtmp          ;
      ioff += 1, joff += 1 ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
