/*  instance.c  */

#include "../DChv.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   return the id of the chevron

   created -- 97apr19, cca
   ----------------------------
*/
int
DChv_id (
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_id(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return(chv->id) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------
   return the symmetry flag of the chevron
   return value = 0 --> chevron has
   symmetric structure and symmetric entries,
   return value = 1 --> chevron is nonsymmetric
   symmetric structure and nonsymmetric entries,
   return value = 2 --> chevron is nonsymmetric
   nonsymmetric structure and nonsymmetric entries,

   created -- 97apr19, cca
   ------------------------------------------------
*/
int
DChv_symmetryFlag (
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_symmetryFlag(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return(chv->symflag) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   return 1 if the chv is symmetric, 0 otherwise

   created -- 97apr19, cca
   -----------------------------------------------
*/
int
DChv_isSymmetric (
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_isSymmetric(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return(chv->symflag == 0) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------
   return 1 if the chv is symmetric structure

   created -- 97apr19, cca
   -----------------------------------------
*/
int
DChv_hasSymmetricStructure (
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_hasNonsymmetricStructure(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return(chv->symflag < 2) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------
   return 1 if the chv is nonsymmetric, 0 otherwise

   created -- 97apr19, cca
   --------------------------------------------------
*/
int
DChv_isNonsymmetric (
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_isNonsymmetric(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return(chv->symflag > 0) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------
   fill *pnD with nD, *pnL with nL, and *pnU with nU.

   created -- 97apr19, cca
   --------------------------------------------------
*/
void
DChv_dimensions (
   DChv   *chv,
   int    *pnD,
   int    *pnL,
   int    *pnU
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || pnD == NULL || pnL == NULL || pnU == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_dimensions(%p,%p,%p,%p)"
           "\n bad input\n", chv, pnD, pnL, pnU) ;
   exit(-1) ;
}
*pnD = chv->nD ;
*pnL = chv->nL ;
*pnU = chv->nU ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   fill *pnrow with nD + nL, *prowind with rowind

   created -- 97apr19, cca
   ----------------------------------------------
*/
void
DChv_rowIndices (
   DChv   *chv,
   int    *pnrow,
   int    **prowind
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || pnrow == NULL || prowind == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_rowIndices(%p,%p,%p)"
           "\n bad input\n", chv, pnrow, prowind) ;
   exit(-1) ;
}
if ( chv->symflag < 2 ) {
   *pnrow   = chv->nD + chv->nU ;
   *prowind = chv->colind ;
} else {
   *pnrow   = chv->nD + chv->nL ;
   *prowind = chv->rowind ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   fill *pncol with nD + nU, *pcolind with colind

   created -- 97apr19, cca
   ----------------------------------------------
*/
void
DChv_columnIndices (
   DChv   *chv,
   int    *pncol,
   int    **pcolind
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || pncol == NULL || pcolind == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_columnIndices(%p,%p,%p)"
           "\n bad input\n", chv, pncol, pcolind) ;
   exit(-1) ;
}
*pncol   = chv->nD + chv->nU ;
*pcolind = chv->colind ;

return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------
   fill *pentries with a pointer to the entries

   created -- 97apr19, cca
   --------------------------------------------
*/
double *
DChv_entries(
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_entries(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return(chv->entries) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------
   return the location of the diagonal entry
   for the ichv'th chevron

   created -- 97apr19, cca
   -----------------------------------------
*/
double *
DChv_diagLocation(
   DChv   *chv,
   int    ichv
) {
double   *diag ;
/*
   ---------------
   check the input
   ---------------
*/
if (  chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_diagLocation(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
if (  ichv < 0 || ichv > chv->nD ) {
   fprintf(stderr, "\n fatal error in DChv_diagLocation(%p)"
           "\n ichv = %d, nD = %d\n", chv, ichv, chv->nD) ;
   exit(-1) ;
}
if ( chv->entries == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_diagLocation(%p)"
           "\n chv->entries is NULL\n", chv) ;
   exit(-1) ;
}
switch ( chv->symflag ) {
case 0 :
   diag = chv->entries + ichv*(chv->nD + chv->nU) - (ichv*(ichv-1))/2 ;
   break ;
case 1 :
case 2 :
   diag = chv->entries + (2*ichv+1)*chv->nD + (ichv+1)*chv->nL
        + ichv*chv->nU - ichv*ichv - ichv - 1 ;
   break ;
default :
   fprintf(stderr, "\n fatal error in DChv_diagLocation(%p)"
           "\n bad symflag = %d\n", chv, chv->symflag) ;
   exit(-1) ;
   break ;
}
return(diag) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   return a pointer to the start of the workspace

   created -- 97jul10, cca
   ----------------------------------------------
*/
void *
DChv_workspace(
   DChv   *chv
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_workspace(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
return((void *) DV_entries(&chv->wrkDV)) ; }

/*--------------------------------------------------------------------*/
