/*  findPivot.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose -- find and test a pivot

   markIV -- object that contains marking vectors
   tau    -- upper bound on magnitude of factor entries
   ndelay -- number of delayed rows and columns on input
   pirow  -- pointer to be filled with pivot row
   pjcol  -- pointer to be filled with pivot column
   pntest -- pointer to be incremented with the number of pivot tests

   return value -- size of pivot
     0 --> pivot not found
     1 --> 1x1 pivot in row *pirow and column *pjcol
     2 --> 2x2 pivot in rows and columns *pirow and *pjcol,
           symmetric front only
   
   created -- 97may12, cca
   ------------------------------------------------------------------
*/
int
DChv_findPivot (
   DChv     *chv,
   IV       *markIV,
   double   tau,
   int      ndelay,
   int      *pirow,
   int      *pjcol,
   int      *pntest
) {
int   irow, jcol, nD, ntest, tag, untag ;
int   *colmark, *mark, *rowmark ;
/*
   ---------------
   check the input
   ---------------
*/
if (  chv == NULL || markIV == NULL || tau < 1.0 || ndelay < 0
   || pirow == NULL || pjcol == NULL || pntest == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DChv_findPivot(%p,%p,%f,%d,%p,%p,%p)"
           "\n bad input\n", 
           chv, markIV, tau, ndelay, pirow, pjcol, pntest) ;
   exit(-1) ;
}
untag  = 0 ;
tag    = 1 ;
nD     = chv->nD ;
#if MYDEBUG > 0
fprintf(stdout, 
"\n DChv_findPivot, id = %d, nD = %d, nL = %d, nU = %d, ndelay = %d",
chv->id, chv->nD, chv->nL, chv->nU, ndelay) ;
fflush(stdout) ;
#endif
*pirow = *pjcol = -1 ;
ntest  = *pntest ;
if ( chv->symflag == 0 ) {
/*
   -----------------------------------
   symmetric front, set up mark vector
   -----------------------------------
*/
   IV_setSize(markIV, nD) ;
   mark = IV_entries(markIV) ;
   if ( ndelay > 0 ) {
      IVfill(ndelay, mark, untag) ;
      IVfill(nD - ndelay, mark, tag) ;
   } else {
      IVfill(nD, mark, tag) ;
   }
   while ( 1 ) {
/*
      --------------------
      find candidate pivot
      --------------------
*/
      DChv_fastBunchParlettPivot(chv, mark, tag, &irow, &jcol) ;
      if ( irow == -1 ) {
/*
         ------------------------------------
         unable to find pivot, return failure
         ------------------------------------
*/
         *pntest = ntest ;
         return(0) ;
      } else {
         ntest++ ;
         if ( DChv_testPivot(chv, irow, jcol, tau) == 1 ) {
/*
            -----------------------------------
            pivot passes, return the pivot size
            -----------------------------------
*/
            *pirow  = irow  ;
            *pjcol  = jcol  ;
            *pntest = ntest ;
            if ( irow == jcol ) {
               return(1) ;
            } else {
               return(2) ;
            }
         } else {
/*
            ----------------------------------------------------------
            mark these rows and columns so we don't look at them again
            ----------------------------------------------------------
*/
            mark[irow] = mark[jcol] = untag ;
         }
      }
   }
} else {
/*
   ---------------------------------------
   nonsymmetric front, set up mark vectors
   ---------------------------------------
*/
   IV_setSize(markIV, 2*nD) ;
   rowmark = IV_entries(markIV) ;
   colmark = rowmark + nD ;
   if ( ndelay > 0 ) {
      IVfill(ndelay, rowmark, untag) ;
      IVfill(nD - ndelay, rowmark, tag) ;
      IVfill(ndelay, colmark, untag) ;
      IVfill(nD - ndelay, colmark, tag) ;
   } else {
      IVfill(nD, rowmark, tag) ;
      IVfill(nD, colmark, tag) ;
   }
   while ( 1 ) {
/*
      --------------------
      find candidate pivot
      --------------------
*/
      DChv_quasimax(chv, rowmark, colmark, tag, &irow, &jcol) ;
#if MYDEBUG > 0
      fprintf(stdout, "\n\n quasimax: irow = %d, jcol = %d",
              irow, jcol) ;
      if ( irow != -1 ) {
         fprintf(stdout, ", entry = %20.12e",
                 DChv_entry(chv, irow, jcol)) ;
      }
      fflush(stdout) ;
#endif
      if ( irow == -1 ) {
/*
         ------------------------------------
         unable to find pivot, return failure
         ------------------------------------
*/
         *pntest = ntest ;
         return(0) ;
      } else {
         ntest++ ;
         if ( DChv_testPivot(chv, irow, jcol, tau) == 1 ) {
/*
            -----------------------------------
            pivot passes, return the pivot size
            -----------------------------------
*/
            *pirow  = irow ;
            *pjcol  = jcol ;
            *pntest = ntest ;
#if MYDEBUG > 0
            fprintf(stdout, "\n pivot passes") ;
            fflush(stdout) ;
#endif
            return(1) ;
         } else {
/*
            ----------------------------------------------------------
            mark these rows and columns so we don't look at them again
            ----------------------------------------------------------
*/
#if MYDEBUG > 0
            fprintf(stdout, 
                    "\n pivot fails, marking row %d and column %d", 
                    irow, jcol) ;
            fflush(stdout) ;
#endif
            rowmark[irow] = colmark[jcol] = untag ;
         }
      }
   }
}
return(0) ; }

/*--------------------------------------------------------------------*/
