/*  factor.c  */

#include "../DChv.h"

#define MYDEBUG 0
#define MYCHECK 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------------
   purpose -- factor the pivot chevron

   ndelay -- number of delayed rows and columns
   pivotflag -- enable pivoting or not
      0 --> no pivoting
      1 --> enable pivoting
   pivotsizesIV -- IV object that holds the sizes of the pivots,
      used only when the front is symmetric and pivoting is enabled
   markIV -- IV object used to find pivots, 
      used only when pivoting is enabled
   tau    -- upper bound on the magnitude of the entries 
      in the factors, used only when pivoting is enabled
   pntest -- pointer to be incremented with the number of pivot tests

   return value -- # of eliminated rows and columns

   created -- 97may12, cca
   ------------------------------------------------------------------
*/
int
DChv_factor (
   DChv     *chv,
   int      ndelay,
   int      pivotflag,
   IV       *pivotsizesIV,
   IV       *markIV,
   double   tau,
   int      *pntest
) {
DChv   wrkDChv ;
int    irow, jcol, ncol, nD, nelim, nrow, pivotsize ;
int    *colind, *rowind ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || pivotflag < 0 || pivotflag > 1 || ndelay < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_factor(%p,%d,%d,%p,%p,%f)"
           "\n bad input\n", 
           chv, ndelay, pivotflag, pivotsizesIV, markIV, tau);
   exit(-1) ;
}
if ( pivotflag == 1 ) {
   if ( markIV == NULL ) {
      fprintf(stderr, "\n fatal error in DChv_factor(%p,%d,%d,%p,%p,%f)"
              "\n pivoting enabled, markIV is NULL \n", 
              chv, ndelay, pivotflag, pivotsizesIV, markIV, tau) ;
      exit(-1) ;
   }
   if ( tau < 1.0 ) {
      fprintf(stderr, "\n fatal error in DChv_factor(%p,%d,%d,%p,%p,%f)"
              "\n tau = %f is invalid \n", 
              chv, ndelay, pivotflag, pivotsizesIV, markIV, tau, tau) ;
      exit(-1) ;
   }
}
if ( chv->symflag == 0 && pivotflag == 1 && pivotsizesIV == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_factor(%p,%d,%d,%p,%p,%f)"
         "\n symmetric front, pivoting enabled, pivotsizesIV is NULL\n",
         chv, ndelay, pivotflag, pivotsizesIV, markIV, tau) ;
   exit(-1) ;
}
nD = chv->nD ;
/*
   --------------------------
   set up the working chevron
   --------------------------
*/
DChv_setDefaultFields(&wrkDChv) ;
DChv_rowIndices(chv, &nrow, &rowind) ;
DChv_columnIndices(chv, &ncol, &colind) ;
DChv_initWithPointers(&wrkDChv, chv->id, nD, chv->nL, chv->nU, 
                      chv->symflag, rowind, colind, DChv_entries(chv)) ;
#if MYDEBUG > 0
fprintf(stdout, "\n\n after initializing wrkDChv") ;
DChv_writeForHumanEye(&wrkDChv, stdout) ;
fflush(stdout) ;
#endif
/*
   -------------------------------------------
   switch over the symmetry and pivoting flags
   -------------------------------------------
*/
if ( pivotflag == 0 ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n\n no pivoting") ;
   fflush(stdout) ;
#endif
/*
   -----------------------
   first case, no pivoting
   -----------------------
*/
   nelim = 0 ;
   while ( nelim < nD ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n\n nelim = %d, before update", nelim) ;
      DChv_writeForHumanEye(&wrkDChv, stdout) ;
      fflush(stdout) ;
#endif
      if ( DChv_intUpdRank1(&wrkDChv) == 0 ) {
         break ;
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n\n nelim = %d, after update", nelim) ;
      DChv_writeForHumanEye(&wrkDChv, stdout) ;
      fflush(stdout) ;
#endif
      DChv_shift(&wrkDChv, 1) ;
      nelim++ ;
   }
} else if ( chv->symflag == 0 ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n\n pivoting, symmetric front %d", chv->id) ;
   fflush(stdout) ;
#endif
/*
   -------------------------------------------------
   second case, symmetric front and pivoting enabled
   -------------------------------------------------
*/
   IV_setSize(pivotsizesIV, 0) ;
   nelim = 0 ;
   while ( nelim < nD ) {
/*
      -------------------------
      find the 1x1 or 2x2 pivot
      -------------------------
*/
#if MYDEBUG > 0
      fprintf(stdout, 
            "\n trying to find pivot, nelim = %d, nD = %d, ndelay = %d",
            nelim, nD, ndelay) ;
      fflush(stdout) ;
#endif
      pivotsize = DChv_findPivot(&wrkDChv, markIV, tau, ndelay, 
                                 &irow, &jcol, pntest) ;
      if ( irow > jcol ) {
         int itemp = irow ;
         irow = jcol ;
         jcol = itemp ;
      }
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n pivotsize = %d, local irow = %d, local jcol = %d",
              pivotsize, irow, jcol) ;
      fflush(stdout) ;
#endif
      irow += nelim ;
      jcol += nelim ;
      if ( pivotsize == 0 ) {
/*
         ---------------------------------
         no pivot found, break out of loop
         ---------------------------------
*/
         break ;
      } else {
         ndelay = 0 ;
         if ( irow == jcol ) {
/*
            ------------------------------------------------------
            1x1 pivot found, swap row and column, update and shift
            ------------------------------------------------------
*/
#if MYDEBUG > 0
            fprintf(stdout, "\n\n before swaps") ;
            DChv_writeForHumanEye(chv, stdout) ;
            fflush(stdout) ;
#endif
            DChv_swapRowsAndColumns(chv, nelim, irow) ;
#if MYDEBUG > 0
            fprintf(stdout, "\n\n after swaps") ;
            DChv_writeForHumanEye(chv, stdout) ;
            fflush(stdout) ;
#endif
#if MYDEBUG > 0
            fprintf(stdout, "\n\n nelim = %d, before update", nelim) ;
            DChv_writeForHumanEye(&wrkDChv, stdout) ;
            fflush(stdout) ;
#endif
            if ( DChv_intUpdRank1(&wrkDChv) == 0 ) {
               break ;
            }
#if MYCHECK > 0
{
double   *entries ;
int      ii, nD, nU ;
   entries = wrkDChv.entries ;
   nD      = wrkDChv.nD      ;
   nU      = wrkDChv.nU      ;
   for ( ii = 1 ; ii < nD + nU ; ii++ ) {
      if ( fabs(entries[ii]) > tau ) {
         fprintf(stdout, 
                 "\n fatal error 1 x 1 pivot, ii = %d, tau = %f", 
                 ii, tau) ;
         DChv_writeForHumanEye(&wrkDChv, stdout) ;
         exit(-1) ;
      }
   }
}
#endif
#if MYDEBUG > 0
            fprintf(stdout, "\n\n nelim = %d, after update", nelim) ;
            DChv_writeForHumanEye(&wrkDChv, stdout) ;
            fflush(stdout) ;
#endif
            DChv_shift(&wrkDChv, 1) ;
            nelim++ ;
            IV_push(pivotsizesIV, 1) ;
         } else {
/*
            --------------------------------------------------------
            2x2 pivot found, swap rows and columns, update and shift
            --------------------------------------------------------
*/
#if MYDEBUG > 0
            fprintf(stdout, "\n\n before swaps") ;
            DChv_writeForHumanEye(chv, stdout) ;
            fflush(stdout) ;
#endif
            DChv_swapRowsAndColumns(chv, nelim, irow) ;
            DChv_swapRowsAndColumns(chv, nelim+1, jcol) ;
#if MYDEBUG > 0
            fprintf(stdout, "\n\n after swaps") ;
            DChv_writeForHumanEye(chv, stdout) ;
            fflush(stdout) ;
#endif
#if MYDEBUG > 0
            fprintf(stdout, "\n\n irow = %d, jcol = %d", irow, jcol) ;
            fprintf(stdout, "\n\n nelim = %d, before update", nelim) ;
#endif
#if MYDEBUG > 0
            DChv_writeForHumanEye(&wrkDChv, stdout) ;
            fflush(stdout) ;
#endif
            if ( DChv_intUpdRank2(&wrkDChv) == 0 ) {
               break ;
            }
#if MYCHECK > 0
{
double   *entries ;
int      ii, nD, nU ;
   entries = wrkDChv.entries ;
   nD      = wrkDChv.nD      ;
   nU      = wrkDChv.nU      ;
   for ( ii = 2 ; ii < nD + nU ; ii++ ) {
      if ( fabs(entries[ii]) > tau ) {
         fprintf(stdout, 
                 "\n 1. fatal error 2 x 2 pivot, ii = %d, tau = %f", 
                 ii, tau) ;
         DChv_writeForHumanEye(&wrkDChv, stdout) ;
         exit(-1) ;
      }
   }
   for ( ii = nD + nU + 1 ; ii < 2*(nD + nU) - 1 ; ii++ ) {
      if ( fabs(entries[ii]) > tau ) {
         fprintf(stdout, 
                 "\n 2. fatal error 2 x 2 pivot, ii = %d, tau = %f", 
                 ii, tau) ;
         DChv_writeForHumanEye(&wrkDChv, stdout) ;
         exit(-1) ;
      }
   }
}
#endif
#if MYDEBUG > 0
            fprintf(stdout, "\n\n nelim = %d, after update", nelim) ;
            DChv_writeForHumanEye(&wrkDChv, stdout) ;
            fflush(stdout) ;
#endif
            DChv_shift(&wrkDChv, 2) ;
            nelim += 2 ;
            IV_push(pivotsizesIV, 2) ;
         }
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n\n ok, done with this pivot") ;
      fflush(stdout) ;
#endif
   }
} else {
/*
   ---------------------------------------------------
   third case, nonsymmetric front and pivoting enabled
   ---------------------------------------------------
*/
   nelim = 0 ;
   while ( nelim < nD ) {
/*
      ------------------
      find the 1x1 pivot
      ------------------
*/
      pivotsize = DChv_findPivot(&wrkDChv, markIV, tau, ndelay, 
                                 &irow, &jcol, pntest) ;
      irow += nelim ;
      jcol += nelim ;
#if MYDEBUG > 0
      fprintf(stdout, "\n\n irow = %d, jcol = %d", irow, jcol) ;
      fflush(stdout) ;
#endif
      if ( pivotsize == 0 ) {
/*
         ---------------------------------
         no pivot found, break out of loop
         ---------------------------------
*/
         break ;
      } else {
         ndelay = 0 ;
/*
         ------------------------------------------------------
         1x1 pivot found, swap row and column, update and shift
         ------------------------------------------------------
*/
#if MYDEBUG > 1
         fprintf(stdout, "\n\n before swaps") ;
         DChv_writeForHumanEye(chv, stdout) ;
         fflush(stdout) ;
#endif
         DChv_swapRows(chv, nelim, irow) ;
         DChv_swapColumns(chv, nelim, jcol) ;
#if MYDEBUG > 1
         fprintf(stdout, "\n\n after swaps") ;
         DChv_writeForHumanEye(chv, stdout) ;
         fflush(stdout) ;
#endif
#if MYDEBUG > 0
         fprintf(stdout, "\n\n nelim = %d, before update", nelim) ;
         fflush(stdout) ;
#endif
#if MYDEBUG > 1
         DChv_writeForHumanEye(&wrkDChv, stdout) ;
         fflush(stdout) ;
#endif
         if ( DChv_intUpdRank1(&wrkDChv) == 0 ) {
            break ;
         }
#if MYDEBUG > 0
         fprintf(stdout, "\n\n nelim = %d, after update", nelim) ;
#endif
#if MYDEBUG > 1
         DChv_writeForHumanEye(&wrkDChv, stdout) ;
         fflush(stdout) ;
#endif
         DChv_shift(&wrkDChv, 1) ;
         nelim++ ;
      }
   }
}
return(nelim) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------
   perform an internal rank-1 update

   return code --
     1 if the pivot was nonzero
     0 if the pivot was zero

   created -- 97apr26, cca
   ---------------------------------
*/
int
DChv_intUpdRank1 (
   DChv   *chv
) {
double   fac, lji, uij ;
double   *entries ;
int      coloff, doff, ii, jj, kchv, kk, nD, nL, nU, 
         rowoff, stride, ulast ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_intUpdRank1(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
entries = DChv_entries(chv) ;
if ( chv->symflag != 0 ) {
/*
   ----------------------
   nonsymmetric chevron
   check for a zero pivot
   ----------------------
*/
   doff   = nD + nL - 1 ;
   rowoff = doff - 1 ;
   coloff = doff + 1 ;
   if ( entries[doff] == 0.0 ) {
      return(0) ;
   }
/*
   ---------------------
   scale the column of L
   ---------------------
*/
   fac = 1./entries[doff] ;
   for ( ii = 0 ; ii < doff ; ii++ ) {
      entries[ii] *= fac ;
   }
/*
   --------------------------------
   loop over the following chevrons
   --------------------------------
*/
   stride = 2*nD + nL + nU - 2 ;
   ulast  = stride ;
   for ( kchv = 1 ; kchv < nD ; kchv++ ) {
      doff += stride ;
      stride -= 2 ;
      lji = entries[rowoff] ;
      uij = entries[coloff] ;
      entries[doff] -= lji * uij ;
      for ( ii = 0, kk = doff - rowoff ; ii < rowoff ; ii++, kk++ ) {
         entries[kk] -= entries[ii] * uij ;
      }
      for ( jj = coloff + 1, kk = doff + 1 ; jj <= ulast ; jj++, kk++ ){
         entries[kk] -= lji * entries[jj] ;
      }
      rowoff-- ;
      coloff++ ;
   }
/*
   ------------------
   scale the row of U
   ------------------
*/
   doff = nD + nL - 1 ;
#if MYDEBUG > 0
   fprintf(stdout, 
           "\n doff = %d, ulast = %d, fac = %f", doff, ulast, fac) ;
#endif
   for ( jj = doff + 1 ; jj <= ulast ; jj++ ) {
      entries[jj] *= fac ;
   }
} else {
/*
   ----------------------
   symmetric chevron
   check for a zero pivot
   ----------------------
*/
   doff = 0 ;
   if ( entries[0] == 0.0 ) {
      return(0) ;
   }
/*
   ------------------
   scale the row of U
   ------------------
*/
   coloff = 1 ;
   ulast = nD + nU - 1 ;
   fac = 1./entries[0] ;
   for ( ii = coloff ; ii <= ulast ; ii++ ) {
      entries[ii] *= fac ;
   }
/*
   --------------------------------
   loop over the following chevrons
   --------------------------------
*/
   stride = nD + nU ;
   ulast  = nD + nU - 1 ;
   for ( kchv = 1 ; kchv < nD ; kchv++ ) {
      doff += stride ;
      stride-- ;
      fac = entries[0] * entries[coloff] ;
      entries[doff] -= entries[coloff] * fac ;
      for ( jj = coloff + 1, kk = doff + 1 ; jj <= ulast ; jj++, kk++ ){
         entries[kk] -= entries[jj] * fac ;
      }
      coloff++ ;
   }
}
return(1) ; }
   
/*--------------------------------------------------------------------*/
/*
   ---------------------------------
   perform an internal rank-2 update

   return code --
     1 if the pivot was nonsingular
     0 if the pivot was singular

   created -- 97apr26, cca
   ---------------------------------
*/
int
DChv_intUpdRank2 (
   DChv   *chv
) {
double   arr, ars, ass, fac0, fac1, recip, tmp0, tmp1 ;
double   *entries ;
int      doff, jj0, jj1, jjoff0, jjoff1, jjlast0, 
         kchv, kk, nD, nL, nU, stride ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_intUpdRank2(%p)"
           "\n bad input\n", chv) ;
   exit(-1) ;
}
if ( chv->symflag != 0 ) {
   fprintf(stderr, "\n fatal error in DChv_intUpdRank2(%p)"
           "\n chevron must be symmetric\n", chv) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
entries = DChv_entries(chv) ;
/*
   --------------------------
   check for a singular pivot
   --------------------------
*/
arr = entries[0] ;
ars = entries[1] ;
ass = entries[nD + nU] ;
if ( arr*ass == ars*ars  ) {
   return(0) ;
}
/*
   -----------------------------
   scale the first two rows of U
   -----------------------------
*/
jjoff0  = 2 ;
jjoff1  = nD + nU + 1 ;
jjlast0 = nD + nU - 1 ;
recip  = 1./(arr*ass - ars*ars) ;
for ( jj0 = jjoff0, jj1 = jjoff1 ; jj0 <= jjlast0 ; jj0++, jj1++ ) {
   tmp0 = entries[jj0] ;
   tmp1 = entries[jj1] ;
   entries[jj0] = ( ass*tmp0 - ars*tmp1)*recip ;
   entries[jj1] = (-ars*tmp0 + arr*tmp1)*recip ;
}
/*
   --------------------------------
   loop over the following chevrons
   --------------------------------
*/
doff   = 2*(nD + nU) - 1 ;
stride = nD + nU - 2 ;
for ( kchv = 2 ; kchv < nD ; kchv++ ) {
   fac0 = arr*entries[jjoff0] + ars * entries[jjoff1] ;
   fac1 = ars*entries[jjoff0] + ass * entries[jjoff1] ;
   for ( jj0 = jjoff0, jj1 = jjoff1, kk = doff ; 
         jj0 <= jjlast0 ; jj0++, jj1++, kk++ ) {
      entries[kk] -= fac0 * entries[jj0] + fac1 * entries[jj1] ;
   }
   jjoff0++ ;
   jjoff1++ ;
   doff += stride ;
   stride-- ;
}
return(1) ; }
   
/*--------------------------------------------------------------------*/
