/*  copyEntriesToVector.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- copy entries to a vector. the portion copied
              can be a union of the strict lower portion,
              the diagonal portion, and the strict upper
              portion. there is one restriction, if the strict
              lower and strict upper are to be copied, the
              diagonal will also be copied.
 
   length     -- length of dvec[]
   npivot     -- number of pivots, may be 0
   pivotsizes -- vector of pivot sizes, may be NULL
   dvec[]     -- vector to receive matrix entries
   copyflag   -- flag to denote what part of the entries to copy
      1 --> copy strict lower entries
      2 --> copy diagonal entries
      3 --> copy strict upper entries
   storeflag  -- flag to denote how to store entries in dvec[]
      0 --> store by rows
      1 --> store by columns
 
   return value -- number of entries copied
 
   created -- 97jun05, cca
   ------------------------------------------------------------
*/
int
DChv_copyEntriesToVector (
   DChv     *chv,
   int      npivot,
   int      pivotsizes[],
   int      length,
   double   *dvec,
   int      copyflag, 
   int      storeflag
) {
double   *entries ;
int      mm, ncol, nD, nent, nL, nrow, nU, symflag ;
/*
   --------------------------------------------
   check the input, get dimensions and pointers
   and check that length is large enough
   --------------------------------------------
*/
if (  chv == NULL || length < 0 || dvec == NULL ) {
   fprintf(stderr,
           "\n fatal error in DChv_copyEntriesToVector(%p,%d,%p,,%d,%d)"
           "\n bad input\n", chv, length, dvec, copyflag, storeflag) ;
   exit(-1) ;
}
if ( copyflag  < 1 || copyflag > 3 ) {
   fprintf(stderr,
           "\n fatal error in DChv_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n bad input\n"
           "\n copyflag = %d, must be\n"
           "\n    1 --> strictly lower entries"
           "\n    2 --> diagonal entries"
           "\n    3 --> strictly upper entries",
           chv, length, dvec, copyflag, storeflag, copyflag) ;
   exit(-1) ;
}
if ( storeflag  < 0 || storeflag > 1 ) {
   fprintf(stderr,
           "\n fatal error in DChv_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n bad input\n"
           "\n storeflag = %d, must be\n"
           "\n    0 --> store by rows"
           "\n    1 --> store by columns",
           chv, length, dvec, copyflag, storeflag, storeflag) ;
   exit(-1) ;
}
nD      = chv->nD      ;
nL      = chv->nL      ;
nU      = chv->nU      ;
symflag = chv->symflag ;
nrow    = nD + nL      ;
ncol    = nD + nU      ;
/*
   ------------------------------------------
   compute the number of entries to be copied
   ------------------------------------------
*/
switch ( copyflag ) {
case 1 : /* strictly lower entries  */
   if ( symflag == 0 ) {
      fprintf(stderr,
           "\n fatal error in DChv_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n symflag = %d, copyflag = %d",
           chv, length, dvec, copyflag, storeflag, symflag, copyflag) ;
   exit(-1) ;
   }
   nent = (nD*(nD-1))/2 + nD*nL ;
   break ;
case 2 : /* diagonal entries  */
   if ( pivotsizes == NULL ) {
      nent = nD ;
   } else {
      int   ipivot ;
      for ( ipivot = nent = 0 ; ipivot < npivot ; ipivot++ ) {
         nent += (pivotsizes[ipivot]*(pivotsizes[ipivot] + 1))/2 ;
      }
   }
   break ;
case 3 : /* strictly upper entries  */
   if ( pivotsizes == NULL ) {
      nent = (nD*(nD-1))/2 + nD*nU ;
   } else {
      int   first, ipivot ;
      for ( ipivot = first = nent = 0 ; ipivot < npivot ; ipivot++ ) {
         nent += first * pivotsizes[ipivot] ;
         first += pivotsizes[ipivot] ;
      }
   }
   break ;
default :
   break ;
}
if ( nent > length ) {
   fprintf(stderr,
           "\n fatal error in DChv_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n nent = %d, buffer length = %d",
           chv, length, dvec, copyflag, storeflag, nent, length) ;
   exit(-1) ;
}
/*
   --------------------------------------------
   make life simple, unit stride through dvec[]
   --------------------------------------------
*/
entries = DChv_entries(chv) ;
mm = 0 ;
switch ( copyflag ) {
case 1 :
/*
   ----------------------------------------
   copy lower entries, pivotsizes[] is NULL
   ----------------------------------------
*/
   switch ( storeflag ) {
   case 0 : {
/*
      -------------
      store by rows
      -------------
*/
      int   ii, jj, jjlast, kinc, kk, kstart, stride ;

      kstart = nD + nL - 1 ;
      stride = 2*nD + nL + nU - 1 ;
      for ( ii = 0 ; ii < nrow ; ii++ ) {
         kk   = kstart ;
         kinc = stride ;
         jjlast = (ii < nD) ? (ii - 1) : (nD - 1) ;
         for ( jj = 0 ; jj <= jjlast ; jj++, mm++ ) {
            dvec[mm] = entries[kk] ;
            kk   += kinc ;
            kinc -=   2  ;
         }
         kstart-- ;
      }
      } break ;
   case 1 : {
/*
      ----------------
      store by columns
      ----------------
*/
      int   ii, jj, kk, kstart, stride ;

      kstart = nD + nL - 1 ;
      stride = 2*nD + nL + nU - 2 ;
      for ( jj = 0 ; jj < nD ; jj++ ) {
         kk = kstart + 1 ; 
         for ( ii = jj + 1 ; ii < nrow ; ii++, kk--, mm++ ) {
            dvec[mm] = entries[kk] ;
         }
         kstart += stride ;
         stride -=    2   ;
      }
      } break ;
   }
   break ;
case 2 :
/*
   ---------------------
   copy diagonal entries
   ---------------------
*/
   if ( chv->symflag == 0 ) {
      int   first, ii, jj, ipivot, kk, kstart, last, stride ;

      stride = nD + nU ;
      kstart = 0 ;
      if ( pivotsizes == NULL ) {
         for ( ii = 0, kk = kstart ; ii < nD ; ii++, mm++ ) {
            dvec[mm] = entries[kk] ;
            kk += stride ;
            stride-- ;
         }
      } else {
         for ( ipivot = first = 0 ; ipivot < npivot ; ipivot++ ) {
            last = first + pivotsizes[ipivot] - 1 ;
            for ( ii = first ; ii <= last ; ii++ ) {
               for ( jj = ii, kk = kstart ; 
                     jj <= last ; 
                     jj++, mm++, kk++ ) {
                  dvec[mm] = entries[kk] ;
               }
               kstart += stride ;
               stride-- ;
            }
         }
      }
   } else {
      int   ii, kk, stride ;

      kk = nD + nL - 1 ;
      stride = 2*nD + nL + nU - 2 ;
      for ( ii = 0 ; ii < nD ; ii++, mm++ ) {
         dvec[mm] = entries[kk] ;
         kk += stride ;
         stride -= 2 ;
      }
   }
   break ;
case 3 :
/*
   -------------------------
   copy strict upper entries
   -------------------------
*/
   switch ( storeflag ) {
   case 0 :
/*
      -------------
      store by rows
      -------------
*/
      if ( symflag == 0 ) {
         int   first, ii, ipivot, jj, kk, kstart, last, stride ;

         kstart = 0 ;
         stride = nD + nU ;
         if ( pivotsizes == NULL ) {
            for ( ii = 0 ; ii < nD ; ii++ ) {
               kk = kstart + 1 ;
               for ( jj = ii + 1 ; jj < ncol ; jj++, kk++, mm++ ) {
                  dvec[mm] = entries[kk] ;
               }
               kstart += stride ;
               stride-- ;
            }
         } else {
            for ( ipivot = first = 0 ;
                  ipivot < npivot ;
                  ipivot++ ) {
               last = first + pivotsizes[ipivot] - 1 ;
               for ( ii = first ; ii <= last ; ii++ ) {
                  kk = kstart + last - ii + 1 ;
                  for ( jj = last + 1 ; jj < ncol ; jj++, kk++, mm++ ) {
                     dvec[mm] = entries[kk] ;
                  }
                  kstart += stride ;
                  stride-- ;
               }
               first = last + 1 ;
            }
         }
      } else {
         int   ii, jj, kk, kstart, stride ;

         kstart = nD + nL - 1 ;
         stride = 2*nD + nL + nU - 2 ;
         for ( ii = 0 ; ii < nD ; ii++ ) {
            kk = kstart + 1 ; 
            for ( jj = ii + 1 ; jj < ncol ; jj++, kk++, mm++ ) {
               dvec[mm] = entries[kk] ;
            }
            kstart += stride ;
            stride -= 2 ;
         }
      }
      break ;
   case 1 :
/*
      ----------------
      store by columns
      ----------------
*/
      if ( chv->symflag == 0 ) {
         int   first, ii, iilast, ipivot, jj, 
               kinc, kk, kstart, last, stride ;

         kstart = 0 ;
         stride = nD + nU - 1 ;
         if ( pivotsizes == NULL ) {
            for ( jj = 0 ; jj < ncol ; jj++ ) {
               kk   = kstart ;
               kinc = stride ;
               iilast = (jj < nD) ? (jj - 1) : (nD - 1) ;
               for ( ii = 0 ; ii <= iilast ; ii++, mm++ ) {
                  dvec[mm] = entries[kk] ;
                  kk += kinc ;
                  kinc-- ;
               }
               kstart++ ;
            }
         } else {
            for ( ipivot = mm = first = 0 ; 
                  ipivot < npivot ; 
                  ipivot++ ) {
               last = first + pivotsizes[ipivot] - 1 ;
               for ( jj = first ; jj <= last ; jj++ ) {
                  kk   = kstart ;
                  kinc = stride ;
                  for ( ii = 0 ; ii < first ; ii++, mm++ ) {
                     dvec[mm] = entries[kk] ;
                     kk += kinc ;
                     kinc-- ;
                  }
                  kstart++ ;
               }
               first = last + 1 ;
            }
            for ( jj = nD ; jj < ncol ; jj++ ) {
               kk   = kstart ;
               kinc = stride ;
               for ( ii = 0 ; ii < nD ; ii++, mm++ ) {
                  dvec[mm] = entries[kk] ;
                  kk += kinc ;
                  kinc-- ;
               }
               kstart++ ;
            }
         }
      } else {
         int   ii, iilast, jj, kinc, kk, kstart, stride ;

         kstart = nD + nL - 1 ;
         stride = 2*nD + nL + nU - 3 ;
         for ( jj = 0 ; jj < ncol ; jj++ ) {
            kk     = kstart ; 
            kinc   = stride ;
            iilast = (jj < nD) ? (jj - 1) : (nD - 1) ;
            for ( ii = 0 ; ii <= iilast ; ii++, mm++ ) {
               dvec[mm] = entries[kk] ;
               kk += kinc ;
               kinc -= 2 ;
            }
            kstart++ ;
         }
      }
      break ;
   default :
      break ;
   }
   break ;
default :
   break ;
}
return(mm) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
   purpose -- copy large entries to a vector. the portion copied
              can be a union of the strict lower portion,
              the diagonal portion, and the strict upper
              portion. there is one restriction, if the strict
              lower and strict upper are to be copied, the
              diagonal will also be copied.
 
   npivot     -- number of pivots, may be 0
   pivotsizes -- vector of pivot sizes, may be NULL
   sizes[]    -- vector to receive row/column sizes
   ivec[]     -- vector to receive row/column indices
   dvec[]     -- vector to receive matrix entries
   copyflag   -- flag to denote what part of the entries to copy
      1 --> copy strict lower entries
      3 --> copy strict upper entries
   storeflag  -- flag to denote how to store entries in dvec[]
      0 --> store by rows
      1 --> store by columns
   droptol    -- entry to be copied must be larger than this magnitude
 
   return value -- number of entries copied
 
   created -- 97jun05, cca
   -------------------------------------------------------------------
*/
int
DChv_copyBigEntriesToVector (
   DChv     *chv,
   int      npivot,
   int      pivotsizes[],
   int      sizes[],
   int      ivec[],
   double   dvec[],
   int      copyflag, 
   int      storeflag,
   double   droptol
) {
double   *entries ;
int      mm, ncol, nD, nL, nrow, nU, symflag ;
/*
   --------------------------------------------
   check the input, get dimensions and pointers
   --------------------------------------------
*/
if (  chv == NULL || sizes == NULL || ivec == NULL || dvec == NULL ) {
   fprintf(stderr,
     "\n fatal error in DChv_copyBigEntriesToVector(%p,%p,%p,%p,,%d,%d)"
     "\n bad input\n", chv, sizes, ivec, dvec, copyflag, storeflag) ;
   exit(-1) ;
}
if ( copyflag  < 1 || copyflag > 3 ) {
   fprintf(stderr,
      "\n fatal error in DChv_copyBigEntriesToVector(%p,%p,%p,%p,%d,%d)"
        "\n bad input\n"
        "\n copyflag = %d, must be\n"
        "\n    1 --> strictly lower entries"
        "\n    2 --> diagonal entries"
        "\n    3 --> strictly upper entries",
        chv, sizes, ivec, dvec, copyflag, storeflag, copyflag) ;
   exit(-1) ;
}
if ( storeflag  < 0 || storeflag > 1 ) {
   fprintf(stderr,
      "\n fatal error in DChv_copyBigEntriesToVector(%p,%p,%p,%p,%d,%d)"
      "\n bad input\n"
      "\n storeflag = %d, must be\n"
      "\n    0 --> store by rows"
      "\n    1 --> store by columns",
      chv, sizes, ivec, dvec, copyflag, storeflag, storeflag) ;
   exit(-1) ;
}
nD      = chv->nD      ;
nL      = chv->nL      ;
nU      = chv->nU      ;
symflag = chv->symflag ;
nrow    = nD + nL      ;
ncol    = nD + nU      ;
/*
   --------------------------------------------
   make life simple, unit stride through dvec[]
   --------------------------------------------
*/
entries = DChv_entries(chv) ;
mm = 0 ;
switch ( copyflag ) {
case 1 :
/*
   ----------------------------------------
   copy lower entries, pivotsizes[] is NULL
   ----------------------------------------
*/
   switch ( storeflag ) {
   case 0 : {
/*
      -------------
      store by rows
      -------------
*/
      int   count, ii, jj, jjlast, kinc, kk, kstart, stride ;

      kstart = nD + nL - 1 ;
      stride = 2*nD + nL + nU - 1 ;
      for ( ii = 0 ; ii < nrow ; ii++ ) {
         kk     = kstart ;
         kinc   = stride ;
         jjlast = (ii < nD) ? (ii - 1) : (nD - 1) ;
         for ( jj = count = 0 ; jj <= jjlast ; jj++ ) {
            if ( fabs(entries[kk]) >= droptol ) {
               ivec[mm] = jj ;
               dvec[mm] = entries[kk] ;
               mm++, count++ ;
            }
            kk += kinc ;
            kinc -= 2 ;
         }
         sizes[ii] = count ;
         kstart-- ;
      }
      } break ;
   case 1 : {
/*
      ----------------
      store by columns
      ----------------
*/
      int   count, ii, jj, kk, kstart, stride ;

      kstart = nD + nL - 1 ;
      stride = 2*nD + nL + nU - 2 ;
      for ( jj = 0 ; jj < nD ; jj++ ) {
         kk = kstart + 1 ;
         for ( ii = jj + 1, count = 0 ; ii < nrow ; ii++, kk-- ) {
            if ( fabs(entries[kk]) >= droptol ) {
               ivec[mm] = ii ;
               dvec[mm] = entries[kk] ;
               mm++, count++ ;
            }
         }
         kstart += stride ;
         stride -= 2 ;
         sizes[jj] = count ;
      }
      } break ;
   }
   break ;
case 3 :
/*
   -------------------------
   copy strict upper entries
   -------------------------
*/
   switch ( storeflag ) {
   case 0 :
/*
      -------------
      store by rows
      -------------
*/
      if ( symflag == 0 ) {
         int   count, first, ii, ipivot, jj, kk, kstart, last, stride ;

         kstart = 0 ;
         stride = nD + nU ;
         if ( pivotsizes == NULL ) {
            for ( ii = 0 ; ii < nD ; ii++ ) {
               kk = kstart + 1 ;
               for ( jj = ii + 1, count = 0 ; jj < ncol ; jj++, kk++ ) {
                  if ( fabs(entries[kk]) >= droptol ) {
                     ivec[mm] = jj ;
                     dvec[mm] = entries[kk] ;
                     mm++, count++ ;
                  }
               }
               kstart += stride ;
               stride-- ;
               sizes[ii] = count ;
            }
         } else {
            for ( ipivot = first = 0 ; ipivot < npivot ; ipivot++ ) {
               last = first + pivotsizes[ipivot] - 1 ;
               for ( ii = first ; ii <= last ; ii++ ) {
                  kk = kstart + last - ii + 1 ; 
                  for ( jj = last + 1, count = 0 ; 
                        jj < ncol ; 
                        jj++, kk++ ) {
                     if ( fabs(entries[kk]) >= droptol ) {
                        ivec[mm] = jj ;
                        dvec[mm] = entries[kk] ;
                        mm++, count++ ;
                     }
                  }
                  kstart += stride ;
                  stride-- ;
                  sizes[ii] = count ;
               }
               first = last + 1 ;
            }
         }
      } else {
         int   count, ii, jj, kk, kstart, stride ;

         kstart = nD + nL - 1 ;
         stride = 2*nD + nL + nU - 2 ;
         for ( ii = 0 ; ii < nD ; ii++ ) {
            kk = kstart + 1 ;
            for ( jj = ii + 1, count = 0 ; jj < ncol ; jj++, kk++ ) {
               if ( fabs(entries[kk]) >= droptol ) {
                  ivec[mm] = jj ;
                  dvec[mm] = entries[kk] ;
                  mm++, count++ ;
               }
            }
            kstart += stride ;
            stride -= 2 ;
            sizes[ii] = count ;
         }
      }
      break ;
   case 1 :
/*
      ----------------
      store by columns
      ----------------
*/
      if ( chv->symflag == 0 ) {
         int   count, first, ii, iilast, ipivot, jj, 
               kinc, kk, kstart, last, stride ;

         kstart = 0 ;
         stride = nD + nU - 1 ;
         if ( pivotsizes == NULL ) {
            for ( jj = 0 ; jj < ncol ; jj++ ) {
               kk     = kstart ;
               kinc   = stride ;
               iilast = (jj < nD) ? (jj - 1) : (nD - 1) ;
               for ( ii = count = 0 ; ii <= iilast ; ii++ ) {
                  if ( fabs(entries[kk]) >= droptol ) {
                     ivec[mm] = ii ;
                     dvec[mm] = entries[kk] ;
                     mm++, count++ ;
                  }
                  kk += kinc ;
                  kinc-- ;
               }
               kstart++ ;
               sizes[jj] = count ;
            }
         } else {
            for ( ipivot = first = 0 ; ipivot < npivot ; ipivot++ ) {
               last = first + pivotsizes[ipivot] - 1 ;
               for ( jj = first ; jj <= last ; jj++ ) {
                  kk   = kstart ;
                  kinc = stride ;
                  for ( ii = count = 0 ; ii < first ; ii++ ) {
                     if ( fabs(entries[kk]) >= droptol ) {
                        ivec[mm] = ii ;
                        dvec[mm] = entries[kk] ;
                        mm++, count++ ;
                     }
                     kk += kinc ;
                     kinc-- ;
                  }
                  kstart++ ;
                  sizes[jj] = count ;
               }
               first = last + 1 ;
            }
            for ( jj = nD ; jj < ncol ; jj++ ) {
               kk   = kstart ;
               kinc = stride ;
               for ( ii = count = 0 ; ii < nD ; ii++ ) {
                  if ( fabs(entries[kk]) >= droptol ) {
                     ivec[mm] = ii ;
                     dvec[mm] = entries[kk] ;
                     mm++, count++ ;
                  }
                  kk += kinc ;
                  kinc-- ;
               }
               kstart++ ;
               sizes[jj] = count ;
            }
         }
      } else {
         int   count, ii, iilast, jj, kinc, kk, kstart, stride ;

         kstart = nD + nL - 1 ;
         stride = 2*nD + nL + nU - 3 ;
         for ( jj = 0 ; jj < ncol ; jj++ ) {
            kk = kstart ; 
            kinc = stride ;
            iilast = (jj < nD) ? (jj - 1) : (nD - 1) ;
            for ( ii = count = 0 ; ii <= iilast ; ii++ ) {
               if ( fabs(entries[kk]) >= droptol ) {
                  ivec[mm] = ii ;
                  dvec[mm] = entries[kk] ;
                  mm++, count++ ;
               }
               kk += kinc ;
               kinc -= 2 ;
            }
            kstart++ ;
            sizes[jj] = count ;
         }
      }
      break ;
   default :
      break ;
   }
   break ;
default :
   break ;
}
return(mm) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------- 
   purpose -- return the number of entries 
   whose magnitude is larger than droptol.

   countflag -- which entries to count
     1 --> count large entries in L
     3 --> count large entries in U
 
   created -- 97jun07, cca
   ---------------------------------------
*/
int
DChv_countBigEntries (
   DChv     *chv,
   int      npivot,
   int      pivotsizes[],
   int      countflag,
   double   droptol
) {
double   *entries ;
int      count, nD, nL, nU ;
/*
   --------------------------------------------
   check the input, get dimensions and pointers
   --------------------------------------------
*/
if (  chv == NULL ) {
   fprintf(stderr,
     "\n fatal error in DChv_countBigEntries(%p,%d,%p,%d,%f)"
     "\n bad input\n", chv, npivot, pivotsizes, countflag, droptol) ;
   exit(-1) ;
}
if ( countflag != 1 && countflag != 3 ) {
   fprintf(stderr,
        "\n fatal error in DChv_countBigEntries(%p,%d,%p,%d,%f)"
        "\n bad input\n"
        "\n countflag = %d, must be\n"
        "\n    1 --> strictly lower entries"
        "\n    2 --> diagonal entries"
        "\n    3 --> strictly upper entries",
        chv, npivot, pivotsizes, countflag, droptol, countflag) ;
   exit(-1) ;
}
if ( countflag == 1 && chv->symflag == 0 ) {
   fprintf(stderr,
        "\n fatal error in DChv_countBigEntries(%p,%d,%p,%d,%f)"
        "\n countflag = %d --> lower entries but chevron is symmetric",
        chv, npivot, pivotsizes, countflag, droptol, countflag) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
entries = DChv_entries(chv) ;
switch ( countflag ) {
case 1 : {
   int   ii, jj, jlast, kinc, kk, kstart, nrow, stride ;
/*
      -----------------------------------------------------
      count the number of big entries in the lower triangle
      -----------------------------------------------------
*/
   nrow   = nD + nL ;
   count  = 0 ;
   kstart = nD + nL - 1 ;
   stride = 2*nD + nL + nU - 1 ;
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      kk   = kstart ;
      kinc = stride ;
      jlast = (ii < nD) ? (ii - 1) : (nD - 1) ;
      for ( jj = 0 ; jj <= jlast ; jj++ ) {
#if MYDEBUG > 0
         fprintf(stdout, "\n A(%d,%d) = %20.12e, kk = %d",
                 ii, jj, entries[kk], kk) ;
#endif
         if ( fabs(entries[kk]) > droptol ) {
#if MYDEBUG > 0
            fprintf(stdout, ", keep") ;
#endif
            count++ ;
         }
         kk   += kinc ;
         kinc -=   2  ;
      }
      kstart-- ;
   }
   } break ;
case 3 : {
   int   first, ii, iilast, ipivot, jj, kinc, kk, kstart, 
         last, ncol, stride ;

   ncol = nD + nU ;
   if ( chv->symflag == 0 ) {
#if MYDEBUG > 0
      fprintf(stdout, 
              "\n symmetric chevron, npivot = %d, pivotsizes = %p", 
              npivot, pivotsizes) ;
#endif
/*
      --------------------
      chevron is symmetric
      --------------------
*/
      count  = 0 ;
      kstart = 0 ;
      stride = nD + nU - 1 ;
      if ( pivotsizes == NULL ) {
/*
         -------------
         D is diagonal
         -------------
*/
         for ( jj = 0 ; jj < ncol ; jj++ ) {
            kk   = kstart ;
            kinc = stride ;
            iilast = (jj < nD) ? (jj - 1) : (nD - 1) ;
            for ( ii = 0 ; ii <= iilast ; ii++ ) {
#if MYDEBUG > 0
               fprintf(stdout, "\n A(%d,%d) = %20.12e, kk = %d",
                       ii, jj, entries[kk], kk) ;
#endif
               if ( fabs(entries[kk]) > droptol ) {
#if MYDEBUG > 0
                  fprintf(stdout, ", keep") ;
#endif
                  count++ ;
               }
               kk += kinc ;
               kinc-- ;
            }
            kstart++ ;
         }
      } else {
/*
         ---------------------------------
         D can have 1 x 1 and 2 x 2 pivots
         ---------------------------------
*/
         for ( ipivot = first = 0 ; ipivot < npivot ; ipivot++ ) {
            last = first + pivotsizes[ipivot] - 1 ;
            for ( jj = first ; jj <= last ; jj++ ) {
#if MYDEBUG > 0
               fprintf(stdout, 
          "\n ipivot = %d, jj = %d, first = %d, last = %d, kstart = %d",
                       ipivot, jj, first, last, kstart) ;
#endif
               kk   = kstart ;
               kinc = stride ;
               for ( ii = 0 ; ii < first ; ii++ ) {
#if MYDEBUG > 0
                  fprintf(stdout, "\n A(%d,%d) = %20.12e, kk = %d",
                          ii, jj, entries[kk], kk) ;
#endif
                  if ( fabs(entries[kk]) > droptol ) {
#if MYDEBUG > 0
                     fprintf(stdout, ", keep") ;
#endif
                     count++ ;
                  }
                  kk += kinc ;
                  kinc-- ;
               }
               kstart++ ;
            }
            first = last + 1 ;
         }
         for ( jj = nD ; jj < ncol ; jj++ ) {
            kk   = kstart ;
            kinc = stride ;
            for ( ii = 0 ; ii < nD ; ii++ ) {
#if MYDEBUG > 0
               fprintf(stdout, "\n A(%d,%d) = %20.12e, kk = %d",
                       ii, jj, entries[kk], kk) ;
#endif
               if ( fabs(entries[kk]) > droptol ) {
#if MYDEBUG > 0
                  fprintf(stdout, ", keep") ;
#endif
                  count++ ;
               }
               kk += kinc ;
               kinc-- ;
            }
            kstart++ ;
         }
      }
   } else {
/*
      --------------------------------------
      chevron is nonsymmetric, D is diagonal
      --------------------------------------
*/
      count  = 0 ;
      kstart = nD + nL - 1 ;
      stride = 2*nD + nL + nU - 3 ;
      for ( jj = 0 ; jj < ncol ; jj++ ) {
         kk     = kstart ;
         kinc   = stride ;
         iilast = (jj < nD) ? (jj - 1) : (nD - 1) ;
         for ( ii = 0 ; ii <= iilast ; ii++ ) {
#if MYDEBUG > 0
            fprintf(stdout, "\n A(%d,%d) = %20.12e, kk = %d",
                    ii, jj, entries[kk], kk) ;
#endif
            if ( fabs(entries[kk]) > droptol ) {
#if MYDEBUG > 0
               fprintf(stdout, ", keep") ;
#endif
               count++ ;
            }
            kk += kinc ;
            kinc -= 2 ;
         }
         kstart++ ;
      }
   }
   } break ;
default :
   break ;
}
return(count) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   purpose -- copy the trailing chevron that starts at offset
 
   created -- 97may16, cca
   ----------------------------------------------------------
*/
void
DChv_copyTrailingPortion (
   DChv   *chvI,
   DChv   *chvJ,
   int    offset
) {
int   first, ncolI, ncolJ, nDJ, nentToCopy, nLJ, nrowI, nrowJ, nUJ ;
int   *colindI, *colindJ, *rowindI, *rowindJ ;
/*
   --------------
   check the data
   --------------
*/
if ( chvI == NULL || chvJ == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DChv_copyTrailingPortion(%p,%p,%d)"
           "\n bad input\n", chvI, chvJ, offset) ;
   exit(-1) ;
}
DChv_dimensions(chvJ, &nDJ, &nLJ, &nUJ) ;
if ( offset < 0 || offset >= nDJ ) {
   fprintf(stderr, 
           "\n fatal error in DChv_copyTrailingPortion(%p,%p,%d)"
           "\n nDJ = %d, offset = %d\n", 
           chvI, chvJ, offset, nDJ, offset) ;
   exit(-1) ;
}
/*
   ----------------------------------------------
   initialize the chvI object and find the number
   of and first location of the entries to copy
   ----------------------------------------------
*/
DChv_columnIndices(chvJ, &ncolJ, &colindJ) ;
if ( chvJ->symflag == 0 ) {
   DChv_init(chvI, chvJ->id, nDJ - offset, 0, nUJ, chvJ->symflag) ;
   DChv_columnIndices(chvI, &ncolI, &colindI) ;
   IVcopy(nDJ + nUJ - offset, colindI, colindJ + offset) ;
   first = offset*(nDJ + nUJ) - (offset*(offset-1))/2 ;
   nentToCopy = (nDJ*(nDJ+1))/2 + nDJ*nUJ - first ;
} else {
   DChv_rowIndices(chvJ, &nrowJ, &rowindJ) ;
   DChv_init(chvI, chvJ->id, nDJ - offset, nLJ, nUJ, chvJ->symflag) ;
   DChv_columnIndices(chvI, &ncolI, &colindI) ;
   IVcopy(nDJ + nUJ - offset, colindI, colindJ + offset) ;
   DChv_rowIndices(chvI, &nrowI, &rowindI) ;
   IVcopy(nDJ + nLJ - offset, rowindI, rowindJ + offset) ;
   first = offset*(2*nDJ + nLJ + nUJ - offset) ;
   nentToCopy = nDJ*(nDJ + nLJ + nUJ) - first ;
}
DVcopy(nentToCopy, DChv_entries(chvI), DChv_entries(chvJ) + first) ;
 
return ; }

/*--------------------------------------------------------------------*/
