/*  add.c  */

#include "../DChv.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ----------------------------------
   add an entry into the DChv object.
   irow and jcol are global id's.

   created -- 97apr26, cca
   ----------------------------------
*/
void
DChv_addEntry (
   DChv     *chv,
   int      irow,
   int      jcol,
   double   entry
) {
int      ii, iloc, jloc, loc, ncol, nD, nL, nrow, nU ;
int      *colind, *rowind ;
double   *entries ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || irow < 0 || jcol < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_addEntry(%p,%d,%d,%e)"
           "\n bad input\n", chv, irow, jcol, entry) ;
   exit(-1) ;
}
/*
   -------------------------------
   extract pointers and dimensions
   -------------------------------
*/
DChv_dimensions(chv, &nD, &nL, &nU) ;
DChv_rowIndices(chv, &nrow, &rowind) ;
DChv_columnIndices(chv, &ncol, &colind) ;
entries = DChv_entries(chv) ;
/*
   -----------------------------------------
   find the local ids for the row and column
   -----------------------------------------
*/
for ( iloc = 0 ; iloc < nrow ; iloc++ ) {
   if ( rowind[iloc] == irow ) {
      break ;
   }
}
if ( iloc == nrow ) {
   fprintf(stderr, "\n warning error in DChv_addEntry(%p,%d,%d,%e)"
           "\n irow = %d not found in row ids\n", 
           chv, irow, jcol, entry, irow) ;
   return ;
}
for ( jloc = 0 ; jloc < ncol ; jloc++ ) {
   if ( colind[jloc] == jcol ) {
      break ;
   }
}
if ( jloc == ncol ) {
   fprintf(stderr, "\n warning error in DChv_addEntry(%p,%d,%d,%e)"
           "\n jcol = %d not found in column ids\n", 
           chv, irow, jcol, entry, jcol) ;
   return ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n entry (%d,%d) --> (%d,%d)",
        irow, jcol, iloc, jloc) ;
#endif
if ( iloc >= nD && jloc >= nD ) {
   fprintf(stderr, "\n fatal error in DChv_addEntry(%p,%d,%d,%e)"
           "\n iloc = %d, jloc = %d, both >= nD = %d\n", 
           chv, irow, jcol, entry, iloc, jloc, nD) ;
   exit(-1) ;
}
/*
   ------------------------------
   find the location of the entry
   ------------------------------
*/
switch ( chv->symflag ) {
case 0 :
   if ( iloc <= jloc ) {
      loc = iloc*(nD + nU) - (iloc*(iloc-1))/2 + jloc - iloc ;
   } else {
      fprintf(stderr, "\n warning error in DChv_addEntry(%p,%d,%d,%e)"
              "\n (irow,jcol) = (%d,%d), (iloc,jloc) = (%d,%d)",
              chv, irow, jcol, entry, irow, jcol, iloc, jloc) ;
      return ;
   }
   break ;
case 1 :
case 2 :
   ii = (iloc <= jloc) ? iloc : jloc ;
   loc = (2*ii + 1)*nD + (ii + 1)*nL 
       + ii*nU - ii*ii - ii - 1 + jloc - iloc ;
   break ;
default :
   fprintf(stderr, "\n fatal error in DChv_addEntry(%p,%d,%d,%e)"
           "\n chv->symflag = %d, must be 0, 1 or 2\n",
           chv, irow, jcol, entry, chv->symflag) ;
   exit(-1) ;
   break ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n loc = %d", loc) ;
#endif
/*
   -------------
   add the entry
   -------------
*/
entries[loc] += entry ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------
   add an entry into the DChv object.
   irow and jcol are local id's.

   created -- 97may14, cca
   ----------------------------------
*/
void
DChv_addEntryLocal (
   DChv     *chv,
   int      irow,
   int      jcol,
   double   entry
) {
double   *diag ;
int      nD, nL, nU ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || irow < 0 || jcol < 0 ) {
   fprintf(stderr, "\n fatal error in DChv_addEntry(%p,%d,%d,%e)"
           "\n bad input\n", chv, irow, jcol, entry) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
if ( (irow > nD && jcol > nD)
   || irow > nD + nL
   || jcol > nD + nU ) {
   fprintf(stderr, "\n fatal error in DChv_addEntry(%p,%d,%d,%e)"
           "\n nD = %d, nL = %d, nU = %d, irow = %d, jcol = %d\n", 
           chv, irow, jcol, entry, nD, nL, nU, irow, jcol) ;
   exit(-1) ;
}
if ( irow <= jcol ) {
   diag = DChv_diagLocation(chv, irow) ;
} else {
   diag = DChv_diagLocation(chv, jcol) ;
}
diag[jcol - irow] += entry ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   add a scaled multiple of a simple chevron to a DChv object.
   the indices are offsets.
   note: for this purpose, (assembling original entries into the
   matrix), the row and column indices of the chevron are identical.
   also, the indices of both the DChv object and the chvind[]
   vector are assumed to be in ascending order.

   created -- 97may01, cca
   -----------------------------------------------------------------
*/
void
DChv_addChevron (
   DChv     *chv,
   double   alpha,
   int      ichv,
   int      chvsize,
   int      chvind[],
   double   chvent[]
) {
int      ierr, ii, iloc, jcol, jj, jjfirst, jjlast, nD, nL, nU, offset ;
int      *colind ;
double   *diag ;
/*
   ---------------
   check the input
   ---------------
*/
if (  chv == NULL || ichv < 0 || chvsize < 0 
   || chvind == NULL || chvent == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DChv_addChevron(%p,%f,%d,%d,%p,%p)"
           "\n bad input\n", 
           chv, alpha, ichv, chvsize, chvind, chvent) ;
   exit(-1) ;
}
if ( chv->symflag < 0 || chv->symflag > 2 ) {
   fprintf(stderr, 
           "\n fatal error in DChv_addChevron(%p,%f,%d,%d,%p,%p)"
           "\n chevron must have symmetric structure\n", 
           chv, alpha, ichv, chvsize, chvind, chvent) ;
   exit(-1) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n alpha = %f, ichv = %d, chvsize = %d", 
        alpha, ichv, chvsize) ;
#endif
if ( chvsize == 0 || alpha == 0.0 ) {
/*
   ----------------------------
   quick return, nothing to add
   ----------------------------
*/
   return ;
}
#if MYDEBUG > 0 
fprintf(stdout, "\n\n DChv_addChevron(%d): ", ichv) ;
IVfp80(stdout, chvsize, chvind, 80, &ierr) ;
DVfprintf(stdout, chvsize, chvent) ;
fflush(stdout) ;
#endif
nD     = chv->nD ;
nL     = chv->nL ;
nU     = chv->nU ;
colind = chv->colind ;
/*
   -------------------------------------
   locate the chevron in the DChv object 
   that will accumulate these entries
   -------------------------------------
*/
for ( iloc = 0 ; iloc < nD ; iloc++ ) {
   if ( colind[iloc] == ichv ) {
      break ;
   }
}
if ( iloc == nD ) {
/*
   ----------------------------------------
   unable to assemble these entries, return
   ----------------------------------------
*/
   fprintf(stderr, "\n fatal error in DChv_addChevron(%p,%d,%d,%p,%p)"
           "\n chevron id %d not found in colind[]",
           chv, ichv, chvsize, chvind, chvent, ichv) ;
   exit(-1) ;
}
if ( chv->symflag == 0 ) {
   diag = DChv_diagLocation(chv, iloc) - iloc ;
#if MYDEBUG > 0 
   fprintf(stdout, "\n ichv = %d, iloc = %d, diag = %p" 
           "\n chv->entries = %p, diag - chv->entries = %d",
           ichv, iloc, diag, chv->entries,
           diag - chv->entries) ;
   fflush(stdout) ;
#endif
/*
   ---------------------
   symmetric chevron
   get the local indices
   ---------------------
*/
   jjlast = nD + nU - 1 ;
   for ( ii = 0, jj = iloc ; ii < chvsize ; ii++ ) {
      if ( (offset = chvind[ii]) < 0 ) {
         fprintf(stderr, 
                 "\n fatal error in DChv_addChevron(%p,%d,%d,%p,%p)"
                 "\n negative offset %d\n", 
                 chv, ichv, chvsize, chvind, chvent, chvind[ii]) ;
         exit(-1) ;
      }
      jcol = ichv + offset ;
#if MYDEBUG > 0 
      fprintf(stdout, "\n ii = %d, offset = %d, jcol = %d",
              ii, offset, jcol) ;
      fflush(stdout) ;
#endif
      while ( jj <= jjlast && jcol != colind[jj] ) {
         jj++ ;
      }
#if MYDEBUG > 0 
      fprintf(stdout, ", jj = %d", jj) ;
      fflush(stdout) ;
#endif
      if ( jj > jjlast ) {
         fprintf(stderr, 
                 "\n fatal error in DChv_addChevron(%p,%d,%d,%p,%p)"
                 "\n jcol %d not found in colind[]\n", 
                 chv, ichv, chvsize, chvind, chvent, jcol) ;
         exit(-1) ;
      }
      chvind[ii] = jj ;
   }
#if MYDEBUG > 0 
   fprintf(stdout, "\n local indices") ;
   IVfp80(stdout, chvsize, chvind, 80, &ierr) ;
   fflush(stdout) ;
#endif
/*
   --------------------
   assemble the chevron
   --------------------
*/
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < chvsize ; ii++ ) {
#if MYDEBUG > 0 
   fprintf(stdout, "\n location %d", &diag[chvind[ii]] - chv->entries) ;
   fflush(stdout) ;
#endif
         diag[chvind[ii]] += chvent[ii] ;
      }
   } else {
      for ( ii = 0 ; ii < chvsize ; ii++ ) {
         diag[chvind[ii]] += alpha * chvent[ii] ;
      }
   }
/*
   ------------------------------------
   restore the indices to their offsets
   ------------------------------------
*/
   for ( ii = 0 ; ii < chvsize ; ii++ ) {
      chvind[ii] = colind[chvind[ii]] - ichv ;
   }
#if MYDEBUG > 0 
   fprintf(stdout, "\n restored indices") ;
   IVfp80(stdout, chvsize, chvind, 80, &ierr) ;
   fflush(stdout) ;
#endif
} else if ( chv->symflag <= 2 ) {
   diag = DChv_diagLocation(chv, iloc) ;
#if MYDEBUG > 0 
   fprintf(stdout, "\n ichv = %d, iloc = %d, diag = %p" 
           "\n chv->entries = %p, diag - chv->entries = %d",
           ichv, iloc, diag, chv->entries,
           diag - chv->entries) ;
   fflush(stdout) ;
#endif
/*
   -----------------------------------------
   nonsymmetric chevron, symmetric structure
   overwrite chvind[] with local indices
   -----------------------------------------
*/
   jjfirst = iloc ;
   jjlast  = nD + nU - 1 ;
   for ( ii = 0, jj = jjlast ; ii < chvsize ; ii++ ) {
      if ( (offset = chvind[ii]) >= 0 ) {
         break ;
      } 
      jcol = ichv - offset ;
      while ( jj >= jjfirst && jcol != colind[jj] ) {
         jj-- ;
      }
      if ( jj < jjfirst ) {
         fprintf(stderr, 
                 "\n fatal error in DChv_addChevron(%p,%d,%d,%p,%p)"
                 "\n jcol %d not found in colind[]\n", 
                 chv, ichv, chvsize, chvind, chvent, jcol) ;
         exit(-1) ;
      }
      chvind[ii] = -jj + iloc ;
   }
   for ( jj = jjfirst ; ii < chvsize ; ii++ ) {
      jcol = ichv + chvind[ii] ;
      while ( jj <= jjlast && jcol != colind[jj] ) {
         jj++ ;
      }
      if ( jj > jjlast ) {
         fprintf(stderr, 
                 "\n fatal error in DChv_addChevron(%p,%d,%d,%p,%p)"
                 "\n jcol %d not found in colind[]\n", 
                 chv, ichv, chvsize, chvind, chvent, jcol) ;
         exit(-1) ;
      }
      chvind[ii] = jj - iloc ;
   }
#if MYDEBUG > 0 
   fprintf(stdout, "\n local indices") ;
   IVfp80(stdout, chvsize, chvind, 80, &ierr) ;
   fflush(stdout) ;
#endif
/*
   --------------------
   assemble the chevron
   --------------------
*/
#if MYDEBUG > 0 
   fprintf(stdout, "\n nonsymmetric DChv, diag = %p", diag) ;
   fflush(stdout) ;
#endif
   if ( alpha == 1.0 ) {
      for ( ii = 0 ; ii < chvsize ; ii++ ) {
#if MYDEBUG > 0 
         fprintf(stdout, "\n  diag[%d] += %12.4e, ii = %d", 
                 chvind[ii], chvent[ii], ii) ;
         fflush(stdout) ;
#endif
         diag[chvind[ii]] += chvent[ii] ;
      }
   } else {
      for ( ii = 0 ; ii < chvsize ; ii++ ) {
#if MYDEBUG > 0 
         fprintf(stdout, "\n  diag[%d] += %12.4e, ii = %d", 
                 chvind[ii], chvent[ii], ii) ;
         fflush(stdout) ;
#endif
         diag[chvind[ii]] += alpha * chvent[ii] ;
      }
   }
/*
   ------------------------------------
   restore the indices to their offsets
   ------------------------------------
*/
   for ( ii = 0 ; ii < chvsize ; ii++ ) {
      if ( chvind[ii] < 0 ) {
         chvind[ii] = ichv - colind[iloc - chvind[ii]] ;
      } else {
         chvind[ii] = colind[chvind[ii] + iloc] - ichv ;
      }
   }
#if MYDEBUG > 0 
   fprintf(stdout, "\n restored indices") ;
   IVfp80(stdout, chvsize, chvind, 80, &ierr) ;
   fflush(stdout) ;
#endif
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   assemble DChv object chvI into DChv object chvJ.
   note: the two objects must be of the same symmetry type,
         the row indices of chvI must nest into those of chvJ,
         the column indices of chvI must nest into those of chvJ.

   created -- 97may01, cca
   --------------------------------------------------------------
*/
void
DChv_assembleDChv (
   DChv   *chvJ,
   DChv   *chvI
) {
double   *diagI, *diagJ ;
int      ierr, ii, ichvI, ichvJ, jj, ncolI, ncolJ, nDI, nDJ, 
         nLI, nLJ, nrowI, nrowJ, nUI, nUJ, offset ;
int      *colindJ, *colindI, *rowindI, *rowindJ ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL || chvI == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_assembleDChv(%p,%p)"
           "\n bad input\n", chvJ, chvI) ;
   exit(-1) ;
}
if (  chvI->symflag < 0 || chvI->symflag > 2
   || chvJ->symflag < 0 || chvJ->symflag > 2
   || chvI->symflag != chvJ->symflag ) {
   fprintf(stderr, "\n fatal error in DChv_assembleDChv(%p,%p)"
           "\n chvI->symflag = %d, chvJ->symflag = %d\n", 
           chvJ, chvI, chvI->symflag, chvJ->symflag) ;
   exit(-1) ;
}
/*
   -------------------------------------
   get the dimensions of the two objects
   -------------------------------------
*/
DChv_dimensions(chvJ, &nDJ, &nLJ, &nUJ) ;
DChv_dimensions(chvI, &nDI, &nLI, &nUI) ;
if (  nDI + nLI > nDJ + nLJ ||  nDI + nUI > nDJ + nUJ ) {
   fprintf(stderr, "\n fatal error in DChv_assembleDChv(%p,%p)"
           "\n bad dimensions"
           "\n nDI = %d, nLI = %d, nUI = %d"
           "\n nDI = %d, nLI = %d, nUI = %d",
           chvJ, chvI, nDI, nLI, nUI, nDJ, nLJ, nUJ) ;
   exit(-1) ;
}
/*
   -----------------
   get the local ids
   -----------------
*/
DChv_columnIndices(chvJ, &ncolJ, &colindJ) ;
DChv_columnIndices(chvI, &ncolI, &colindI) ;
#if MYDEBUG > 0
fprintf(stdout, "\n colindI") ;
IVfp80(stdout, ncolI, colindI, 80, &ierr) ;
fprintf(stdout, "\n colindJ") ;
IVfp80(stdout, ncolJ, colindJ, 80, &ierr) ;
#endif
for ( ii = 0, jj = 0 ; ii < ncolI ; ii++ ) {
   while ( jj < ncolJ && colindI[ii] != colindJ[jj] ) {
#if MYDEBUG > 0
      fprintf(stdout, "\n colindI[%d] = %d, colindJ[%d] = %d",
              ii, colindI[ii], jj, colindJ[jj]) ;
#endif
      jj++ ;
   }
   if ( jj == ncolJ ) {
      break ;
   }
   colindI[ii] = jj ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n local column indices") ;
IVfp80(stdout, ncolI, colindI, 80, &ierr) ;
#endif
if ( jj == ncolJ ) {
   fprintf(stderr, "\n fatal error in DChv_assembleDChv(%p,%p)"
           "\n column indicesI do not nest in indicesJ\n", chvJ, chvI) ;
   fprintf(stderr, "\n colindI") ;
   IVfp80(stderr, ncolI, colindI, 80, &ierr) ;
   fprintf(stderr, "\n colindJ") ;
   IVfp80(stderr, ncolJ, colindJ, 80, &ierr) ;
   exit(-1) ;
}
switch ( chvJ->symflag ) {
case 0 :
case 1 :
/*
   -------------------
   symmetric structure
   -------------------
*/
   nrowI   = ncolI   ;
   rowindI = colindI ;
   break ;
case 2 :
/*
   ----------------------
   nonsymmetric structure
   ----------------------
*/
   DChv_rowIndices(chvJ, &nrowJ, &rowindJ) ;
   DChv_rowIndices(chvI, &nrowI, &rowindI) ;
#if MYDEBUG > 0
   fprintf(stdout, "\n rowindI") ;
   IVfp80(stdout, nrowI, rowindI, 80, &ierr) ;
   fprintf(stdout, "\n rowindJ") ;
   IVfp80(stdout, nrowJ, rowindJ, 80, &ierr) ;
#endif
   for ( ii = 0, jj = 0 ; ii < nrowI ; ii++ ) {
      while ( jj < nrowJ && rowindI[ii] != rowindJ[jj] ) {
         jj++ ;
      }
      if ( jj == nrowJ ) {
         break ;
      }
      rowindI[ii] = jj ;
   }
   if ( jj == nrowJ ) {
      fprintf(stderr, "\n fatal error in DChv_assembleDChv(%p,%p)"
              "\n row indicesI do not nest in indicesJ\n", chvJ, chvI) ;
      fprintf(stderr, "\n rowindI") ;
      IVfp80(stderr, nrowI, rowindI, 80, &ierr) ;
      fprintf(stderr, "\n rowindJ") ;
      IVfp80(stderr, nrowJ, rowindJ, 80, &ierr) ;
      exit(-1) ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n local row indices") ;
   IVfp80(stdout, nrowI, rowindI, 80, &ierr) ;
#endif
   break ;
default :
   break ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n local column indices") ;
IVfp80(stdout, ncolI, colindI, 80, &ierr) ;
fprintf(stdout, "\n local row indices") ;
IVfp80(stdout, nrowI, rowindI, 80, &ierr) ;
#endif
/*
   ---------------------------
   loop over the chevrons in I
   ---------------------------
*/
for ( ichvI = 0 ; ichvI < nDI ; ichvI++ ) {
   ichvJ = colindI[ichvI] ;
   if ( ichvJ != rowindI[ichvI] ) {
      fprintf(stderr, "\n fatal error in DChv_assembleDChv(%p,%p)"
              "\n ichvI = %d, ichvJ = %d, rowindI[ichvI] = %d",
              chvJ, chvI, ichvI, ichvJ, rowindI[ichvI]) ;
      exit(-1) ;
   }
   diagI = DChv_diagLocation(chvI, ichvI) ;
   diagJ = DChv_diagLocation(chvJ, ichvJ) ;
#if MYDEBUG > 0
   fprintf(stdout, 
          "\n ichvI = %d, diagI - entriesI = %d"
          "\n ichvJ = %d, diagJ - entriesJ = %d",
          ichvI, diagI - chvI->entries, ichvJ, diagJ - chvJ->entries) ;
#endif
   diagJ[0] += diagI[0] ;
   switch ( chvI->symflag ) {
   case 0 :
/*
      --------------------------------------
      symmetric structure, symmetric entries
      --------------------------------------
*/
      for ( ii = ichvI + 1, jj = 1 ; ii < ncolI ; ii++, jj++ ) {
         offset = colindI[ii] - ichvJ ;
#if MYDEBUG > 0
         fprintf(stdout, 
                "\n ii = %d, jj = %d, offset = %d", ii, jj, offset) ;
#endif
         diagJ[offset] += diagI[jj] ;
      }
      break ;
   case 1 :
/*
      -----------------------------------------
      symmetric structure, nonsymmetric entries
      -----------------------------------------
*/
      for ( ii = ichvI + 1, jj = 1 ; ii < ncolI ; ii++, jj++ ) {
         offset = colindI[ii] - ichvJ ;
         diagJ[offset]  += diagI[jj]  ;
         diagJ[-offset] += diagI[-jj] ;
      }
      break ;
   case 2 :
/*
      --------------------------------------------
      nonsymmetric structure, nonsymmetric entries
      --------------------------------------------
*/
      for ( ii = ichvI + 1, jj = 1 ; ii < ncolI ; ii++, jj++ ) {
         offset = colindI[ii] - ichvJ ;
#if MYDEBUG > 0
         fprintf(stdout, "\n diagJ[%d] += diagI[%d]", offset, jj) ;
#endif
         diagJ[offset] += diagI[jj] ;
      }
         for ( ii = ichvI + 1, jj = -1 ; ii < nrowI ; ii++, jj-- ) {
            offset = rowindI[ii] - ichvJ ;
#if MYDEBUG > 0
         fprintf(stdout, "\n diagJ[%d] += diagI[%d]", -offset, jj) ;
#endif
            diagJ[-offset] += diagI[jj] ;
         }
      break ;
   default :
      break ;
   }
}
/*
   -------------------
   restore the indices
   -------------------
*/
for ( ii = 0 ; ii < ncolI ; ii++ ) {
   colindI[ii] = colindJ[colindI[ii]] ;
}
if ( chvJ->symflag == 2 ) {
   for ( ii = 0 ; ii < nrowI ; ii++ ) {
      rowindI[ii] = rowindJ[rowindI[ii]] ;
   }
}
return ; }
   
/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   add DChv object chvI into chvJ 
   note: the two objects must be of the same symmetry type,
         and the same size

   created -- 97may30, cca
   --------------------------------------------------------------
*/
void
DChv_addDChv (
   DChv   *chvJ,
   DChv   *chvI
) {
int      nD, nent, nL, nU ;
/*
   ---------------
   check the input
   ---------------
*/
if ( chvJ == NULL || chvI == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_addDChv(%p,%p)"
           "\n bad input\n", chvJ, chvI) ;
   exit(-1) ;
}
if ( chvJ->symflag != chvI->symflag
   || (nD = chvJ->nD) != chvI->nD
   || (nL = chvJ->nL) != chvI->nL
   || (nU = chvJ->nU) != chvI->nU ) {
   fprintf(stderr, "\n fatal error in DChv_addDChv(%p,%p)"
           "\n chevrons are not of the same type or size\n", 
           chvJ, chvI) ;
   exit(-1) ;
}
if ( chvJ->symflag == 0 ) {
   nent = (nD*(nD+1))/2 + nD*nU ;
} else {
   nent = nD*(nD + nL + nU) ;
}
DVadd(nent, chvJ->entries, chvI->entries) ;

return ; }

/*--------------------------------------------------------------------*/
