/*  IO.c  */

#include "../DChv.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   purpose -- to write the object to a file
              in human readable form

   created -- 97apr23, cca
   ----------------------------------------
*/
void
DChv_writeForHumanEye (
   DChv   *chv,
   FILE     *fp
) {
DA2      mtx ;
int      ierr, ncol, nD, nL, nrow, nU ;
int      *colind, *rowind ; 
/*
   ---------------
   check the input
   ---------------
*/
if ( chv == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DChv_writeForHumanEye(%p,%p)"
           "\n bad input\n", chv, fp) ;
   exit(-1) ;
}
DChv_dimensions(chv, &nD, &nL, &nU) ;
fprintf(fp, 
        "\n DChv object at address %p"
        "\n id = %d, nD = %d, nL = %d, nU = %d, symflag = %d",
        chv, chv->id, nD, nL, nU, chv->symflag) ;
if ( DChv_isSymmetric(chv) == 1 ) {
   fprintf(fp, "\n chv is symmetric") ;
} else if ( DChv_isNonsymmetric(chv) == 1 ) {
   fprintf(fp, "\n chv is nonsymmetric") ;
} else {
   fprintf(fp, "\n chv is neither symmetric nor nonsymmetric") ;
}
DChv_rowIndices(chv, &nrow, &rowind) ;
if ( nrow > 0 && rowind != NULL ) {
   fprintf(fp, "\n chv's row indices at %p", rowind) ;
   IVfp80(fp, nrow, rowind, 80, &ierr) ;
}
DChv_columnIndices(chv, &ncol, &colind) ;
if ( ncol > 0 && colind != NULL ) {
   fprintf(fp, "\n chv's column indices at %p", colind) ;
   IVfp80(fp, ncol, colind, 80, &ierr) ;
}
/*
   --------------------
   load the (1,1) block
   --------------------
*/
DA2_setDefaultFields(&mtx) ;
DChv_fill11block(chv, &mtx) ;
fprintf(fp, "\n (1,1) block") ;
DA2_writeForHumanEye(&mtx, fp) ;
if ( nU > 0 ) {
/*
   --------------------
   load the (1,2) block
   --------------------
*/
   DChv_fill12block(chv, &mtx) ;
   fprintf(fp, "\n (1,2) block") ;
   DA2_writeForHumanEye(&mtx, fp) ;
}
if ( nL > 0 && DChv_isNonsymmetric(chv) == 1 ) {
/*
   --------------------
   load the (2,1) block
   --------------------
*/
   DChv_fill21block(chv, &mtx) ;
   fprintf(fp, "\n (2,1) block") ;
   DA2_writeForHumanEye(&mtx, fp) ;
}
DA2_clearData(&mtx) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   purpose -- print out a dense nonsymmetric front
  
   created -- 97may13, cca
   -----------------------------------------------
*/
void
printDenseNonsymmetricFront (
   int      nD,
   int      nL,
   int      nU,
   double   diagent[],
   double   lowerent[],
   double   upperent[],
   FILE     *fp
) {
DA2   mtx ;
int   ii, jj, kk ;
/*
   ---------------
   check the input
   ---------------
*/
if ( nD <= 0 || nL < 0 || nU < 0 
   || diagent == NULL 
   || (nD > 1 && lowerent == NULL)
   || (nD > 1 && upperent == NULL)
   || fp == NULL ) {
   fprintf(stderr, "\n fatal error in printDenseNonsymmetricFront()"
           "\n bad input\n") ;
   exit(-1) ;
}
DA2_setDefaultFields(&mtx) ;
/*
   ------------------------
   load the diagonal matrix
   ------------------------
*/
DA2_resize(&mtx, nD, nD, 1, nD) ;
DA2_zero(&mtx) ;
for ( ii = 0, kk = 0 ; ii < nD ; ii++ ) {
   DA2_setEntry(&mtx, ii, ii, diagent[ii]) ;
   for ( jj = 0 ; jj < ii ; jj++, kk++ ) {
      DA2_setEntry(&mtx, ii, jj, lowerent[kk]) ;
      DA2_setEntry(&mtx, jj, ii, upperent[kk]) ;
   }
}
fprintf(fp, "\n (1,1) block") ;
fflush(fp) ;
DA2_writeForHumanEye(&mtx, fp) ;
fflush(fp) ;
/*
   ---------------------
   load the upper matrix
   ---------------------
*/
if ( nU > 0 ) {
   DA2_resize(&mtx, nD, nU, 1, nD) ;
   DA2_zero(&mtx) ;
   for ( jj = 0, kk = (nD*(nD-1))/2 ; jj < nU ; jj++ ) {
      for ( ii = 0 ; ii < nD ; ii++, kk++ ) {
         DA2_setEntry(&mtx, ii, jj, upperent[kk]) ;
      }
   }
   fprintf(fp, "\n (1,2) block") ;
   fflush(fp) ;
   DA2_writeForHumanEye(&mtx, fp) ;
   fflush(fp) ;
}
/*
   ---------------------
   load the lower matrix
   ---------------------
*/
if ( nU > 0 ) {
   DA2_resize(&mtx, nL, nD, nD, 1) ;
   DA2_zero(&mtx) ;
   for ( ii = 0, kk = (nD*(nD-1))/2 ; ii < nL ; ii++ ) {
      for ( jj = 0 ; jj < nD ; jj++, kk++ ) {
         DA2_setEntry(&mtx, ii, jj, lowerent[kk]) ;
      }
   }
   fprintf(fp, "\n (2,1) block") ;
   fflush(fp) ;
   DA2_writeForHumanEye(&mtx, fp) ;
   fflush(fp) ;
}
return ; }
 
/*--------------------------------------------------------------------*/
/*
   --------------------------------------------
   purpose -- print out a dense symmetric front
 
   created -- 97may13, cca
   --------------------------------------------
*/
void
printDenseSymmetricFront (
   int      nD,
   int      nU,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   FILE     *fp
) {
DA2   mtx ;
int   first, ii, ipivot, jj, last, mm ;
/*
   ---------------
   check the input
   ---------------
*/
if ( nD <= 0 || nU < 0
   || diagent == NULL
   || (nD > 1 && upperent == NULL)
   || fp == NULL ) {
   fprintf(stderr, "\n fatal error in printDenseNonsymmetricFront()"
           "\n bad input\n") ;
   exit(-1) ;
}
DA2_setDefaultFields(&mtx) ;
/*
   ------------------------
   load the diagonal matrix
   ------------------------
*/
DA2_resize(&mtx, nD, nD, 1, nD) ;
DA2_zero(&mtx) ;
if ( pivotsizes == NULL ) {
   for ( ii = 0 ; ii < nD ; ii++ ) {
      DA2_setEntry(&mtx, ii, ii, diagent[ii]) ;
   }
} else {
   for ( ipivot = 0, first = mm = 0 ; ipivot < npivot ; ipivot++ ) {
      last = first + pivotsizes[ipivot] - 1 ;
      for ( ii = first ; ii <= last ; ii++ ) {
         DA2_setEntry(&mtx, ii, ii, diagent[mm++]) ;
         for ( jj = ii + 1 ; jj <= last ; jj++, mm++ ) {
            DA2_setEntry(&mtx, ii, jj, diagent[mm]) ;
            DA2_setEntry(&mtx, jj, ii, diagent[mm]) ;
         }
      }
      first = last + 1 ;
   }
}
if ( pivotsizes == NULL ) {
   for ( jj = mm = 0 ; jj < nD ; jj++ ) {
      for ( ii = 0 ; ii < jj ; ii++, mm++ ) {
         DA2_setEntry(&mtx, ii, jj, upperent[mm]) ;
      }
   }
} else {
   for ( ipivot = 0, first = mm = 0 ; ipivot < npivot ; ipivot++ ) {
      last = first + pivotsizes[ipivot] - 1 ;
      for ( jj = first ; jj <= last ; jj++ ) {
         for ( ii = 0 ; ii < first ; ii++, mm++ ) {
            DA2_setEntry(&mtx, ii, jj, upperent[mm]) ;
         }
      }
      first = last + 1 ;
   }
}
fprintf(fp, "\n (1,1) block") ;
fflush(fp) ;
DA2_writeForHumanEye(&mtx, fp) ;
fflush(fp) ;
/*
   ---------------------
   load the upper matrix
   ---------------------
*/
if ( nU > 0 ) {
   DA2_resize(&mtx, nD, nU, 1, nD) ;
   DA2_zero(&mtx) ;
   for ( jj = 0 ; jj < nU ; jj++ ) {
      for ( ii = 0 ; ii < nD ; ii++, mm++ ) {
         DA2_setEntry(&mtx, ii, jj, upperent[mm]) ;
      }
   }
   fprintf(fp, "\n (1,2) block") ;
   fflush(fp) ;
   DA2_writeForHumanEye(&mtx, fp) ;
   fflush(fp) ;
}
return ; }
 
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------
   purpose -- print out a sparse nonsymmetric front
 
   created -- 97may13, cca
   ------------------------------------------------
*/
void
printSparseNonsymmetricFront (
   int      nD,
   int      nL,
   int      nU,
   double   diagent[],
   int      sizesL[],
   int      lowerind[],
   double   lowerent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   FILE     *fp
) {
DA2   mtx ;
int   ii, jj, kk, lsizei, mm, nn, usizej ;
/*
   ---------------
   check the input
   ---------------
*/
if ( nD <= 0 || nL < 0 || nU < 0
   || diagent == NULL
   || fp == NULL ) {
   fprintf(stderr, "\n fatal error in printDenseNonsymmetricFront()"
           "\n bad input\n") ;
   exit(-1) ;
}
if ( nD > 0 || (nD == 1 && nL > 0) ) {
   if (  sizesL == NULL || lowerind == NULL || lowerent == NULL ) {
      fprintf(stderr, "\n fatal error in printDenseNonsymmetricFront()"
              "\n bad input for L\n") ;
      exit(-1) ;
   }
}
if ( nD > 0 || (nD == 1 && nU > 0) ) {
   if (  sizesU == NULL || upperind == NULL || upperent == NULL ) {
      fprintf(stderr, "\n fatal error in printDenseNonsymmetricFront()"
              "\n bad input for U\n") ;
      exit(-1) ;
   }
}
DA2_setDefaultFields(&mtx) ;
/*
   ------------------------
   load the diagonal matrix
   ------------------------
*/
DA2_resize(&mtx, nD, nD, 1, nD) ;
DA2_zero(&mtx) ;
for ( jj = 0 ; jj < nD ; jj++ ) {
   DA2_setEntry(&mtx, jj, jj, diagent[jj]) ;
}
for ( jj = 0, mm = 0 ; jj < nD ; jj++ ) {
   if ( (usizej = sizesU[jj]) > 0 ) {
      for ( kk = 0 ; kk < usizej ; kk++, mm++ ) {
         DA2_setEntry(&mtx, upperind[mm], jj, upperent[mm]) ;
      }
   }
}
for ( ii = 0, nn = 0 ; ii < nD ; ii++ ) {
   if ( (lsizei = sizesL[ii]) > 0 ) {
      for ( kk = 0 ; kk < lsizei ; kk++, nn++ ) {
         DA2_setEntry(&mtx, ii, lowerind[nn], lowerent[nn]) ;
      }
   }
}
fprintf(fp, "\n (1,1) block") ;
fflush(fp) ;
DA2_writeForHumanEye(&mtx, fp) ;
fflush(fp) ;
/*
   ---------------------
   load the upper matrix
   ---------------------
*/
if ( nU > 0 ) {
   DA2_resize(&mtx, nD, nU, 1, nD) ;
   DA2_zero(&mtx) ;
   for ( jj = 0 ; jj < nU ; jj++ ) {
      if ( (usizej = sizesU[jj + nD]) > 0 ) {
         for ( kk = 0 ; kk < usizej ; kk++, mm++ ) {
            DA2_setEntry(&mtx, upperind[mm], jj, upperent[mm]) ;
         }
      }
   }
   fprintf(fp, "\n (1,2) block") ;
   fflush(fp) ;
   DA2_writeForHumanEye(&mtx, fp) ;
   fflush(fp) ;
}
/*
   ---------------------
   load the lower matrix
   ---------------------
*/
if ( nL > 0 ) {
   DA2_resize(&mtx, nL, nD, nD, 1) ;
   DA2_zero(&mtx) ;
   for ( ii = 0 ; ii < nL ; ii++ ) {
      if ( (lsizei = sizesL[ii + nD]) > 0 ) {
         for ( kk = 0 ; kk < lsizei ; kk++, nn++ ) {
            DA2_setEntry(&mtx, ii, lowerind[nn], lowerent[nn]) ;
         }
      }
   }
   fprintf(fp, "\n (2,1) block") ;
   fflush(fp) ;
   DA2_writeForHumanEye(&mtx, fp) ;
   fflush(fp) ;
}
return ; }
 
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   purpose -- print out a sparse symmetric front
 
   created -- 97may13, cca
   ---------------------------------------------
*/
void
printSparseSymmetricFront (
   int      nD,
   int      nU,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   FILE     *fp
) {
DA2   mtx ;
int   first, ii, ipivot, jj, kk, last, mm, usizej ;
/*
   ---------------
   check the input
   ---------------
*/
if ( nD <= 0 || nU < 0
   || npivot <= 0 || pivotsizes == NULL
   || diagent == NULL
   || (nD > 1 && upperent == NULL)
   || fp == NULL ) {
   fprintf(stderr, "\n fatal error in printDenseNonsymmetricFront()"
           "\n bad input\n") ;
   exit(-1) ;
}
DA2_setDefaultFields(&mtx) ;
/*
   ------------------------
   load the diagonal matrix
   ------------------------
*/
DA2_resize(&mtx, nD, nD, 1, nD) ;
DA2_zero(&mtx) ;
if ( pivotsizes == NULL ) {
   for ( ii = 0 ; ii < nD ; ii++ ) {
      DA2_setEntry(&mtx, ii, ii, diagent[ii]) ;
   }
} else {
   for ( ipivot = 0, first = mm = 0 ; ipivot < npivot ; ipivot++ ) {
      last = first + pivotsizes[ipivot] - 1 ;
      for ( ii = first ; ii <= last ; ii++ ) {
         DA2_setEntry(&mtx, ii, ii, diagent[mm++]) ;
         for ( jj = ii + 1 ; jj <= last ; jj++, mm++ ) {
            DA2_setEntry(&mtx, ii, jj, diagent[mm]) ;
            DA2_setEntry(&mtx, jj, ii, diagent[mm]) ;
         }
      }
      first = last + 1 ;
   }
}
/*
fprintf(fp, "\n (1,1) block") ;
fflush(fp) ;
DA2_writeForHumanEye(&mtx, fp) ;
fflush(fp) ;
*/
for ( jj = 0, mm = 0 ; jj < nD ; jj++ ) {
   if ( (usizej = sizesU[jj]) > 0 ) {
      for ( kk = 0 ; kk < usizej ; kk++, mm++ ) {
         ii = upperind[mm] ;
         DA2_setEntry(&mtx, ii, jj, upperent[mm]) ;
      }
   }
}
fprintf(fp, "\n (1,1) block") ;
fflush(fp) ;
DA2_writeForHumanEye(&mtx, fp) ;
fflush(fp) ;
/*
   ---------------------
   load the upper matrix
   ---------------------
*/
if ( nU > 0 ) {
   DA2_resize(&mtx, nD, nU, 1, nD) ;
   DA2_zero(&mtx) ;
   for ( jj = 0 ; jj < nU ; jj++ ) {
      if ( (usizej = sizesU[jj + nD]) > 0 ) {
         for ( kk = 0 ; kk < usizej ; kk++, mm++ ) {
            ii = upperind[mm] ;
            DA2_setEntry(&mtx, ii, jj, upperent[mm]) ;
         }
      }
   }
   fprintf(fp, "\n (1,2) block") ;
   fflush(fp) ;
   DA2_writeForHumanEye(&mtx, fp) ;
   fflush(fp) ;
}
return ; }
 
/*--------------------------------------------------------------------*/
