/*  test_sparseSymUpd.c  */

#include "../DChv.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -----------------------------------------
   test the sparse symmetric update methods.

   created -- 97may15, cca
   -----------------------------------------
*/
{
DChv     *chvI, *chvJ ;
double   droptol, ops, t1, t2 ;
double   *diagent, *entriesI, *entriesJ, *upperent ;
Drand    *drand ;
DV       *tempDV ;
FILE     *msgFile ;
int      count, ierr, msglvl, I, ii, iifirst, iilast, jj, J, 
         length, nDI, nDJ, nentDI, nentI, nentJ, nentUI, 
         npivot, nUI, nUJ, nzf, offset, seed, symflag ;
int      *colindI, *colindJ, *ivec, *pivotsizes, 
         *sizesU, *upperind ;

if ( argc != 10 ) {
   fprintf(stdout, 
           "\n\n usage : %s msglvl msgFile "
           "\n         nDI nUI offset nDJ nUJ seed droptol"
           "\n    msglvl  -- message level"
           "\n    msgFile -- message file"
           "\n    nDI     -- # of rows and columns in the (1,1) block"
           "\n    nUI     -- # of columns in the (1,2) block"
           "\n    offset  -- offset from I to J"
           "\n    nDJ     -- # of rows and columns in the (1,1) block"
           "\n    nUJ     -- # of columns in the (1,2) block"
           "\n    seed    -- random number seed"
           "\n    droptol -- drop tolerance"
           "\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
nDI     = atoi(argv[3]) ;
nUI     = atoi(argv[4]) ;
offset  = atoi(argv[5]) ;
nDJ     = atoi(argv[6]) ;
nUJ     = atoi(argv[7]) ;
seed    = atoi(argv[8]) ;
droptol = atof(argv[9]) ;
fprintf(msgFile, 
        "\n test_sparseSymUpd"
        "\n msglvl  = %d" 
        "\n msgFile = %s" 
        "\n nDI     = %d" 
        "\n nUI     = %d" 
        "\n offset  = %d" 
        "\n nDJ     = %d" 
        "\n nUJ     = %d" 
        "\n seed    = %d"
        "\n droptol = %f",
        msglvl, argv[2], nDI, nUI, offset, nDJ, nUJ, seed, droptol) ;
if ( nDI <= 0 || nUI < 0 || offset < 0 || nDJ <= 0 || nUJ < 0 ) {
   fprintf(stderr, "\n invalid input"
           "\n nDI = %d, nUI = %d, offset = %d, nDJ = %d, nUJ = %d",
           nDI, nUI, offset, nDJ, nUJ) ;
   exit(-1) ;
}
if ( nUI - offset > nDJ + nUJ ) {
   fprintf(stderr, "\n invalid input"
           "\n nUI - offset = %d > %d = nDJ + nUJ",
           nUI - offset, nDJ + nUJ) ;
   exit(-1) ;
}
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_init(drand) ;
Drand_setSeed(drand, seed) ;
/*
   ---------------
   get the indices
   ---------------
*/
length  = 2*(nDI + offset + nDJ + nUJ) ;
ivec    = IVinit(length, -1) ;
colindI = IVinit(nDI + nUI, -1) ;
colindJ = IVinit(nDJ + nUJ, -1) ;
IVramp(length, ivec, 0, 1) ;
IVshuffle(length, ivec, ++seed) ;
IVqsortUp(nDI + offset + nDJ + nUJ, ivec) ;
IVcopy(nDI + offset, colindI, ivec) ;
IVcopy(nDJ + nUJ, colindJ, ivec + nDI + offset) ;
IVshuffle(nDJ + nUJ, ivec + nDI + offset, ++seed) ;
IVqsortUp(nUI - offset, ivec + nDI + offset) ;
IVcopy(nUI - offset, colindI + nDI + offset, ivec + nDI + offset) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n colindI") ;
   IVfp80(msgFile, nDI + nUI, colindI, 80, &ierr) ;
   fprintf(msgFile, "\n colindJ") ;
   IVfp80(msgFile, nDJ + nUJ, colindJ, 80, &ierr) ;
}
/*
   -----------------------------------------
   compute iifirst, iilast and local indices
   -----------------------------------------
*/
iifirst = nDI + offset ;
for ( ii = iifirst, jj = 0 ; ii < nDI + nUI ; ii++ ) {
   while ( colindI[ii] > colindJ[jj] ) {
      jj++ ;
   }
   colindI[ii] = jj ;
}
for ( ii = iifirst, iilast = -1 ; ii < nDI + nUI ; ii++ ) {
   if ( colindI[ii] >= nDJ ) { 
      break ;
   }
   iilast = ii ;
}
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n iifirst = %d, iilast = %d",
           iifirst, iilast) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n colindI") ;
   IVfp80(msgFile, nDI + nUI, colindI, 80, &ierr) ;
}
/*
   ----------------------------
   initialize the DChv objects
   ----------------------------
*/
Drand_setNormal(drand, 0.0, 1.0) ;
symflag = 0 ;
I = 0 ;
J = 1 ;
MARKTIME(t1) ;
chvJ = DChv_new() ;
DChv_init(chvJ, J, nDJ, 0, nUJ, symflag) ;
IVcopy(nDJ + nUJ, chvJ->colind, colindJ) ;
chvI = DChv_new() ;
DChv_init(chvI, I, nDI, 0, nUI, symflag) ;
IVcopy(nDI + nUI, chvI->colind, colindI) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize chv object",
        t2 - t1) ;
fflush(msgFile) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n DChv I") ;
   DChv_writeForHumanEye(chvI, msgFile) ;
   fprintf(msgFile, "\n DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
/*
   ------------------------------------
   load the entries with random entries
   ------------------------------------
*/
nentI = (nDI*(nDI+1))/2 + nDI*nUI ;
nentJ = (nDJ*(nDJ+1))/2 + nDJ*nUJ ;
entriesI = DChv_entries(chvI) ;
entriesJ = DChv_entries(chvJ) ;
Drand_fillDvector(drand, nentI, entriesI) ;
/*
Drand_fillDvector(drand, nentJ, entriesJ) ;
*/
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n DChv I") ;
   DChv_writeForHumanEye(chvI, msgFile) ;
}
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
/*
   ----------------------------
   get the pivot sizes for chvI
   ----------------------------
*/
Drand_setUniform(drand, 1, 2.999) ;
pivotsizes = IVinit(nDI, 0) ;
Drand_fillIvector(drand, nDI, pivotsizes) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n initial pivotsizes[] : ") ;
   IVfp80(msgFile, nDI, pivotsizes, 80, &ierr) ;
}
for ( npivot = count = nentDI = 0 ; npivot < nDI ; npivot++ ) {
   count += pivotsizes[npivot] ;
   if ( count > nDI ) {
      pivotsizes[npivot]-- ;
      count-- ;
   } 
   if ( pivotsizes[npivot] == 1 ) {
      nentDI++ ;
   } else {
      nentDI += 3 ;
   }
   if ( count == nDI ) {
      break ;
   }
}
npivot++ ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n final pivotsizes[] : ") ;
   IVfp80(msgFile, npivot, pivotsizes, 80, &ierr) ;
}
/*
   ---------------------------------
   get the diagonal entries for chvI
   ---------------------------------
*/
diagent = DVinit(nentDI, 0.0) ;
DChv_copyEntriesToVector(chvI, npivot, pivotsizes, 
                         nentDI, diagent, 2, 1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n diag entries stored") ;
   DVfprintf(msgFile, nentDI, diagent) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------------
   drop entries and get a sparse representation for chvI
   -----------------------------------------------------
*/
nentUI = DChv_countBigEntries(chvI, npivot, pivotsizes, 3, droptol) ;
sizesU   = IVinit(nDI + nUI, 0.0) ;
upperind = IVinit(nentUI, -1) ;
upperent = DVinit(nentUI, 0.0) ;
DChv_copyBigEntriesToVector(chvI, npivot, pivotsizes, 
                          sizesU, upperind, upperent, 3, 1, droptol) ;
for ( ii = iifirst, nzf = 0, ops = 0.0 ; ii <= iilast ; ii++ ) {
   if ( sizesU[ii] > 0 ) {
      nzf += sizesU[ii] ;
      for ( jj = ii ; jj < nDI + nUI ; jj++ ) {
         ops += 2*sizesU[jj] ;
      }
   }
}
fprintf(msgFile, "\n nzf = %d, ops = %.0f, ops/nzf = %.3f", 
        nzf, ops, ops/nzf) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n sizesU") ;
   IVfp80(msgFile, nDI + nUI, sizesU, 80, &ierr) ;
   fprintf(msgFile, "\n upperind") ;
   IVfp80(msgFile, nentUI, upperind, 80, &ierr) ;
   fprintf(msgFile, "\n upper entries stored") ;
   DVfprintf(msgFile, nentUI, upperent) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n sparse front") ;
   printSparseSymmetricFront(nDI, nUI, npivot, pivotsizes, diagent, 
                             sizesU, upperind, upperent, msgFile) ;
}
/*
   --------------------------
   compute the update to chvJ
   --------------------------
*/
tempDV = DV_new() ;
DV_setSize(tempDV, nDI) ;
MARKTIME(t1) ;
DChv_sparseSymmetricUpdate(chvJ, nDI, nDI + nUI, npivot, pivotsizes, 
                           diagent, sizesU, upperind, upperent, 
                           colindI, iifirst, iilast, tempDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n 1x1 kernel, %8.3f CPU, %.3f mflops",
        t2 - t1, 1.e-6*ops/(t2 - t1)) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n after update, DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
fprintf(msgFile, "\n maxabs chvJ = %20.12e", DChv_maxabs(chvJ)) ;

fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
