/*  test_sparseNonsymUpd.c  */

#include "../DChv.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   --------------------------------------------
   test the sparse nonsymmetric update methods.

   created -- 97may15, cca
   --------------------------------------------
*/
{
DChv     *chvI, *chvJ ;
double   droptol, ops, t1, t2 ;
double   *diagent, *entriesI, *entriesJ, *lowerent, *upperent ;
Drand    *drand ;
DV       *tempDV ;
FILE     *msgFile ;
int      ierr, msglvl, I, ii, iifirst, iilast, jj, J, 
         length, nDI, nDJ, nentDI, nentI, nentJ, nentLI, nentUI, 
         nLI, nLJ, nUI, nUJ, nzf, offset, seed, symflag ;
int      *colindI, *colindJ, *ivec, *lowerind, 
         *sizesL, *sizesU, *upperind ;

if ( argc != 10 ) {
   fprintf(stdout, 
           "\n\n usage : %s msglvl msgFile "
           "\n         nDI nUI offset nDJ nUJ seed droptol"
           "\n    msglvl  -- message level"
           "\n    msgFile -- message file"
           "\n    nDI     -- # of rows and columns in the (1,1) block"
           "\n    nUI     -- # of columns in the (1,2) block"
           "\n    offset  -- offset from I to J"
           "\n    nDJ     -- # of rows and columns in the (1,1) block"
           "\n    nUJ     -- # of columns in the (1,2) block"
           "\n    seed    -- random number seed"
           "\n    droptol -- drop tolerance"
           "\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
nDI     = atoi(argv[3]) ;
nUI     = atoi(argv[4]) ;
offset  = atoi(argv[5]) ;
nDJ     = atoi(argv[6]) ;
nUJ     = atoi(argv[7]) ;
seed    = atoi(argv[8]) ;
droptol = atof(argv[9]) ;
nLI     = nUI ;
nLJ     = nUJ ;
fprintf(msgFile, 
        "\n test_sparseNonsymUpd"
        "\n msglvl  = %d" 
        "\n msgFile = %s" 
        "\n nDI     = %d" 
        "\n nUI     = %d" 
        "\n offset  = %d" 
        "\n nDJ     = %d" 
        "\n nUJ     = %d" 
        "\n seed    = %d"
        "\n droptol = %f",
        msglvl, argv[2], nDI, nUI, offset, nDJ, nUJ, seed, droptol) ;
if ( nDI <= 0 || nUI < 0 || offset < 0 || nDJ <= 0 || nUJ < 0 ) {
   fprintf(stderr, "\n invalid input"
           "\n nDI = %d, nUI = %d, offset = %d, nDJ = %d, nUJ = %d",
           nDI, nUI, offset, nDJ, nUJ) ;
   exit(-1) ;
}
if ( nUI - offset > nDJ + nUJ ) {
   fprintf(stderr, "\n invalid input"
           "\n nUI - offset = %d > %d = nDJ + nUJ",
           nUI - offset, nDJ + nUJ) ;
   exit(-1) ;
}
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_init(drand) ;
Drand_setSeed(drand, seed) ;
/*
   ---------------
   get the indices
   ---------------
*/
length  = 2*(nDI + offset + nDJ + nUJ) ;
ivec    = IVinit(length, -1) ;
colindI = IVinit(nDI + nUI, -1) ;
colindJ = IVinit(nDJ + nUJ, -1) ;
IVramp(length, ivec, 0, 1) ;
IVshuffle(length, ivec, ++seed) ;
IVqsortUp(nDI + offset + nDJ + nUJ, ivec) ;
IVcopy(nDI + offset, colindI, ivec) ;
IVcopy(nDJ + nUJ, colindJ, ivec + nDI + offset) ;
IVshuffle(nDJ + nUJ, ivec + nDI + offset, ++seed) ;
IVqsortUp(nUI - offset, ivec + nDI + offset) ;
IVcopy(nUI - offset, colindI + nDI + offset, ivec + nDI + offset) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n colindI") ;
   IVfp80(msgFile, nDI + nUI, colindI, 80, &ierr) ;
   fprintf(msgFile, "\n colindJ") ;
   IVfp80(msgFile, nDJ + nUJ, colindJ, 80, &ierr) ;
}
/*
   -----------------------------------------
   compute iifirst, iilast and local indices
   -----------------------------------------
*/
iifirst = nDI + offset ;
for ( ii = iifirst, jj = 0 ; ii < nDI + nUI ; ii++ ) {
   while ( colindI[ii] > colindJ[jj] ) {
      jj++ ;
   }
   colindI[ii] = jj ;
}
for ( ii = iifirst, iilast = -1 ; ii < nDI + nUI ; ii++ ) {
   if ( colindI[ii] >= nDJ ) { 
      break ;
   }
   iilast = ii ;
}
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n iifirst = %d, iilast = %d",
           iifirst, iilast) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n colindI") ;
   IVfp80(msgFile, nDI + nUI, colindI, 80, &ierr) ;
}
/*
   ----------------------------
   initialize the DChv objects
   ----------------------------
*/
Drand_setNormal(drand, 0.0, 1.0) ;
symflag = 2 ;
I = 0 ;
J = 1 ;
MARKTIME(t1) ;
chvJ = DChv_new() ;
DChv_init(chvJ, J, nDJ, nUJ, nUJ, symflag) ;
IVcopy(nDJ + nUJ, chvJ->colind, colindJ) ;
if ( symflag == 2 ) {
   IVcopy(nDJ + nUJ, chvJ->rowind, colindJ) ;
}
chvI = DChv_new() ;
DChv_init(chvI, I, nDI, nUI, nUI, symflag) ;
IVcopy(nDI + nUI, chvI->colind, colindI) ;
if ( symflag == 2 ) {
   IVcopy(nDI + nUI, chvI->rowind, colindI) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize chv object",
        t2 - t1) ;
fflush(msgFile) ;
if ( msglvl > 5 ) {
   fprintf(msgFile, "\n DChv I") ;
   DChv_writeForHumanEye(chvI, msgFile) ;
   fprintf(msgFile, "\n DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
/*
   ------------------------------------
   load the entries with random entries
   ------------------------------------
*/
nentI    = nDI*(nDI + nLI + nUI) ;
nentJ    = nDJ*(nDJ + nLJ + nUJ) ;
entriesI = DChv_entries(chvI) ;
entriesJ = DChv_entries(chvJ) ;
Drand_fillDvector(drand, nentI, entriesI) ;
/*
Drand_fillDvector(drand, nentJ, entriesJ) ;
*/
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n DChv I") ;
   DChv_writeForHumanEye(chvI, msgFile) ;
}
if ( msglvl > 5 ) {
   fprintf(msgFile, "\n DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
/*
   ---------------------------------
   get the diagonal entries for chvI
   ---------------------------------
*/
nentDI = nDI ;
diagent = DVinit(nentDI, 0.0) ;
DChv_copyEntriesToVector(chvI, 0, NULL, nentDI, diagent, 2, 1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n diag entries stored") ;
   DVfprintf(msgFile, nentDI, diagent) ;
   fflush(msgFile) ;
}
/*
   -----------------------------------------------------
   drop entries and get a sparse representation for chvI
   -----------------------------------------------------
*/
nentUI   = DChv_countBigEntries(chvI, 0, NULL, 3, droptol) ;
sizesU   = IVinit(nDI + nUI, 0.0) ;
upperind = IVinit(nentUI, -1) ;
upperent = DVinit(nentUI, 0.0) ;
DChv_copyBigEntriesToVector(chvI, 0, NULL, sizesU, upperind, upperent,
                            3, 1, droptol) ;
nentLI   = DChv_countBigEntries(chvI, 0, NULL, 1, droptol) ;
sizesL   = IVinit(nDI + nLI, 0.0) ;
lowerind = IVinit(nentLI, -1) ;
lowerent = DVinit(nentLI, 0.0) ;
DChv_copyBigEntriesToVector(chvI, 0, NULL, sizesL, lowerind, lowerent,
                            1, 0, droptol) ;
for ( ii = iifirst, nzf = 0, ops = 0.0 ; ii <= iilast ; ii++ ) {
   if ( sizesU[ii] > 0 ) {
      nzf += sizesU[ii] ;
      for ( jj = ii ; jj < nDI + nLI ; jj++ ) {
         ops += 2*sizesL[jj] ;
      }
   }
   if ( sizesL[ii] > 0 ) {
      nzf += sizesL[ii] ;
      for ( jj = ii ; jj < nDI + nUI ; jj++ ) {
         ops += 2*sizesU[jj] ;
      }
   }
}
fprintf(msgFile, "\n nzf = %d, ops = %.0f, ops/nzf = %.3f", 
        nzf, ops, ops/nzf) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n sizesL") ;
   IVfp80(msgFile, nDI + nLI, sizesL, 80, &ierr) ;
   fprintf(msgFile, "\n lowerind") ;
   IVfp80(msgFile, nentLI, lowerind, 80, &ierr) ;
   fprintf(msgFile, "\n lower entries stored") ;
   DVfprintf(msgFile, nentLI, lowerent) ;
   fprintf(msgFile, "\n sizesU") ;
   IVfp80(msgFile, nDI + nUI, sizesU, 80, &ierr) ;
   fprintf(msgFile, "\n upperind") ;
   IVfp80(msgFile, nentUI, upperind, 80, &ierr) ;
   fprintf(msgFile, "\n upper entries stored") ;
   DVfprintf(msgFile, nentUI, upperent) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n\n sparse front") ;
   printSparseNonsymmetricFront(nDI, nLI, nUI, diagent, sizesL,
      lowerind, lowerent, sizesU, upperind, upperent, msgFile) ;
}
/*
   --------------------------
   compute the update to chvJ
   --------------------------
*/
tempDV = DV_new() ;
DV_setSize(tempDV, nDI) ;
MARKTIME(t1) ;
DChv_sparseNonsymmetricUpdate(chvJ, nDI, nDI + nUI, diagent, 
   sizesL, lowerind, lowerent, sizesU, upperind, upperent, 
   colindI, iifirst, iilast, tempDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n 1x1 kernel, %8.3f CPU, %.3f mflops",
        t2 - t1, 1.e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n after update, DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
fprintf(msgFile, "\n maxabs chvJ = %20.12e", DChv_maxabs(chvJ)) ;

fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
