/*  test_search.c  */

#include "../DChv.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------
   test the search and pivot test methods.

   created -- 97apr24, cca
   ---------------------------------------
*/
{
DChv     *chv ;
double   maxval, tau, t1, t2 ;
double   *entries ;
Drand    *drand ;
FILE     *msgFile ;
int      ii, irow, jcol, msglvl, ncol, nD, nent, nL, nrow, nU, 
         seed, symflag, tag ;
int      *colind, *colmark, *rowind, *rowmark ;

if ( argc != 9 ) {
   fprintf(stdout, 
           "\n\n usage : %s msglvl msgFile nD nL nU symflag seed tau"
           "\n    msglvl  -- message level"
           "\n    msgFile -- message file"
           "\n    nD      -- # of rows and columns in the (1,1) block"
           "\n    nL      -- # of rows in the (2,1) block"
           "\n    nU      -- # of columns in the (1,2) block"
           "\n    symflag -- symmetry flag"
           "\n       0 --> symmetric"
           "\n       1 --> nonsymmetric"
           "\n    seed    -- random number seed"
           "\n    tau     -- stability tolerance"
           "\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
nD      = atoi(argv[3]) ;
nL      = atoi(argv[4]) ;
nU      = atoi(argv[5]) ;
symflag = atoi(argv[6]) ;
seed    = atoi(argv[7]) ;
tau     = atof(argv[8]) ;
fprintf(msgFile, "\n test_search:"
        "\n msglvl = %d"
        "\n msgFile = %s"
        "\n nD      = %d"
        "\n nL      = %d"
        "\n nU      = %d"
        "\n symflag = %d"
        "\n seed    = %d"
        "\n tau     = %f",
        msglvl, argv[2], nD, nL, nU, symflag, seed, tau) ;
/*
   -----------------------------
   check for errors in the input
   -----------------------------
*/
if (  nD <= 0 || nL < 0 || nU < 0 
   || symflag < 0 || symflag > 1 || tau < 0.0 ) {
   fprintf(stderr, "\n invalid input"
      "\n nD = %d, nL = %d, nU = %d, symflag = %d, tau = %f\n",
           nD, nL, nU, symflag, tau) ;
   exit(-1) ;
}
if ( symflag == 0 && nL != nU ) {
   fprintf(stderr, "\n invalid input"
      "\n symflag = %d, nL = %d, nU = %d", symflag, nL, nU) ;
   exit(-1) ;
}
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_init(drand) ;
Drand_setSeed(drand, seed) ;
Drand_setNormal(drand, 0.0, 1.0) ;
/*
   ----------------------------
   initialize the DChv object
   ----------------------------
*/
MARKTIME(t1) ;
chv = DChv_new() ;
DChv_init(chv, 0, nD, nL, nU, symflag) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize chv object",
        t2 - t1) ;
fflush(msgFile) ;
DChv_columnIndices(chv, &ncol, &colind) ;
IVramp(ncol, colind, 0, 1) ;
if ( symflag == 1 ) {
   DChv_rowIndices(chv, &nrow, &rowind) ;
   IVramp(nrow, rowind, 0, 1) ;
}
if ( msglvl > 3 ) {
   DChv_writeForHumanEye(chv, msgFile) ;
}
/*
   ------------------------------------
   load the entries with random entries
   ------------------------------------
*/
entries = DChv_entries(chv) ;
if ( symflag == 0 ) {
   nent = (nD*(nD+1))/2 + nD*nU ;
} else {
   nent = nD*(nD + nL + nU) ;
}
Drand_fillDvector(drand, nent, entries) ;
if ( msglvl > 0 ) {
   DChv_writeForHumanEye(chv, msgFile) ;
}
/*
   ------------------
   find the row maxes
   ------------------
*/
fprintf(msgFile, "\n row maxes") ;
colmark = IVinit(nD, -1) ;
tag = -1 ;
for ( irow = 0 ; irow < nD ; irow++ ) {
   jcol = DChv_maxabsInRow(chv, irow, colmark, tag, &maxval) ;
   fprintf(msgFile, "\n row %d, jcol %d, value %20.12e",
           irow, jcol, maxval) ;
}
fprintf(msgFile, "\n column maxes") ;
rowmark = IVinit(nD, -1) ;
tag = -1 ;
for ( jcol = 0 ; jcol < nD ; jcol++ ) {
   irow = DChv_maxabsInColumn(chv, jcol, rowmark, tag, &maxval) ;
   fprintf(msgFile, "\n row %d, jcol %d, value %20.12e",
           irow, jcol, maxval) ;
}
if ( symflag == 1 ) {
   double   arr ;
/*
   --------------------------
   find the quasimax elements
   --------------------------
*/
   IVfill(nD, rowmark, -1) ;
   IVfill(nD, colmark, -1) ;
   tag = -1 ;
   for ( ii = 0 ; ii < nD ; ii++ ) {
      DChv_quasimax(chv, rowmark, colmark, tag, &irow, &jcol) ;
      if ( irow == -1 ) {
         break ;
      }
      arr = DChv_entry(chv, irow, jcol) ;
      fprintf(msgFile, "\n\n quasimax entry (%d,%d) = %20.12e",
              irow, jcol, arr) ;
      if ( 1 == DChv_testPivot(chv, irow, jcol, tau) ) {
         fprintf(msgFile, "\n pivot passes") ;
      } else {
         fprintf(msgFile, "\n pivot fails") ;
      }
      rowmark[irow] = ii ;
      colmark[jcol] = ii ;
   }
} else {
   double   arr, ars, ass ;
   int      itmp ;
   int      *mark ;
/*
   ------------------------------------
   find the fast bunch-parlett elements
   ------------------------------------
*/
   fprintf(msgFile, "\n\n fast bunch-parlett pivots") ;
   tag = -1 ;
   mark = IVinit(nD, -1) ;
   IVfill(nD, mark, -1) ;
   for ( ii = 0 ; ii < nD ; ii++ ) {
      DChv_fastBunchParlettPivot(chv, mark, tag, &irow, &jcol) ;
      fprintf(msgFile, "\n\n FBP (%d,%d)", irow, jcol) ;
      if ( irow == -1 ) {
         break ;
      } else if ( irow == jcol ) {
         arr = DChv_entry(chv, irow, jcol) ;
         fprintf(msgFile, "\n 1 x 1 pivot (%d,%d) = %20.12e",
                 irow, irow, arr) ;
         if ( 1 == DChv_testPivot(chv, irow, jcol, tau) ) {
            fprintf(msgFile, "\n pivot passes") ;
         } else {
            fprintf(msgFile, "\n pivot fails") ;
         }
         mark[irow] = ii ;
      } else {
         if ( irow > jcol ) {
            itmp = irow ;
            irow = jcol ; 
            jcol = itmp ;
         }
         arr = DChv_entry(chv, irow, irow) ;
         ars = DChv_entry(chv, irow, jcol) ;
         ass = DChv_entry(chv, jcol, jcol) ;
         fprintf(msgFile, "\n arr = %20.12e", arr) ;
         fprintf(msgFile, "\n ars = %20.12e", ars) ;
         fprintf(msgFile, "\n ass = %20.12e", ass) ;
         fprintf(msgFile, 
                 "\n 2 x 2 pivot (%2d,%2d) = [ %20.12e %20.12e ]"
                 "\n                       [ %20.12e %20.12e ]",
                 irow, jcol, arr, ars, ars, ass) ;
         if ( 1 == DChv_testPivot(chv, irow, jcol, tau) ) {
            fprintf(msgFile, "\n pivot passes") ;
         } else {
            fprintf(msgFile, "\n pivot fails") ;
         }
         mark[irow] = ii ;
         mark[jcol] = ii ;
      }
   }
   IVfree(mark) ;
}
fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
