/*  test_denseNonsymUpd.c  */

#include "../DChv.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   -------------------------------------------
   test the dense nonsymmetric update methods.

   created -- 97may15, cca
   -------------------------------------------
*/
{
DChv     *chvI, *chvJ, *chvK, *chvL ;
double   ops, t1, t2 ;
double   *diagent, *entriesI, *entriesJ, *entriesK, *entriesL,
         *keepent, *lowerent, *upperent ;
Drand    *drand ;
DV       *tempDV ;
FILE     *msgFile ;
int      ierr, msglvl, I, ii, iifirst, iilast, jj, J, K, L,
         length, nDI, nDJ, nentDI, nentI, nentJ, nentLI, nentUI, 
         nLI, nLJ, nout, nUI, nUJ, nupd, offset, seed, symflag ;
int      *colindI, *colindJ, *ivec ;

if ( argc != 9 ) {
   fprintf(stdout, 
           "\n\n usage : %s msglvl msgFile nDI nUI offset nDJ nUJ seed "
           "\n    msglvl  -- message level"
           "\n    msgFile -- message file"
           "\n    nDI     -- # of rows and columns in the (1,1) block"
           "\n    nUI     -- # of columns in the (1,2) block"
           "\n    offset  -- offset from I to J"
           "\n    nDJ     -- # of rows and columns in the (1,1) block"
           "\n    nUJ     -- # of columns in the (1,2) block"
           "\n    seed    -- random number seed"
           "\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
nDI          = atoi(argv[3]) ;
nUI          = atoi(argv[4]) ;
offset       = atoi(argv[5]) ;
nDJ          = atoi(argv[6]) ;
nUJ          = atoi(argv[7]) ;
seed         = atoi(argv[8]) ;
fprintf(msgFile, 
        "\n test_denseNonsymUpd"
        "\n msglvl       = %d" 
        "\n msgFile      = %s" 
        "\n nDI          = %d" 
        "\n nUI          = %d" 
        "\n offset       = %d" 
        "\n nDJ          = %d" 
        "\n nUJ          = %d" 
        "\n seed         = %d",
        msglvl, argv[2], nDI, nUI, offset, nDJ, nUJ, seed) ;
if ( nDI <= 0 || nUI < 0 || offset < 0 || nDJ <= 0 || nUJ < 0 ) {
   fprintf(stderr, "\n invalid input"
           "\n nDI = %d, nUI = %d, offset = %d, nDJ = %d, nUJ = %d",
           nDI, nUI, offset, nDJ, nUJ) ;
   exit(-1) ;
}
if ( nUI - offset > nDJ + nUJ ) {
   fprintf(stderr, "\n invalid input"
           "\n nUI - offset = %d > %d = nDJ + nUJ",
           nUI - offset, nDJ + nUJ) ;
   exit(-1) ;
}
nLI = nUI ;
nLJ = nUJ ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_init(drand) ;
Drand_setSeed(drand, seed) ;
/*
   ---------------
   get the indices
   ---------------
*/
length  = 2*(nDI + offset + nDJ + nUJ) ;
ivec    = IVinit(length, -1) ;
colindI = IVinit(nDI + nUI, -1) ;
colindJ = IVinit(nDJ + nUJ, -1) ;
IVramp(length, ivec, 0, 1) ;
IVshuffle(length, ivec, ++seed) ;
IVqsortUp(nDI + offset + nDJ + nUJ, ivec) ;
IVcopy(nDI + offset, colindI, ivec) ;
IVcopy(nDJ + nUJ, colindJ, ivec + nDI + offset) ;
IVshuffle(nDJ + nUJ, ivec + nDI + offset, ++seed) ;
IVqsortUp(nUI - offset, ivec + nDI + offset) ;
IVcopy(nUI - offset, colindI + nDI + offset, ivec + nDI + offset) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n colindI") ;
   IVfp80(msgFile, nDI + nUI, colindI, 80, &ierr) ;
   fprintf(msgFile, "\n colindJ") ;
   IVfp80(msgFile, nDJ + nUJ, colindJ, 80, &ierr) ;
}
/*
   -----------------------------------------
   compute iifirst, iilast and local indices
   -----------------------------------------
*/
iifirst = nDI + offset ;
for ( ii = iifirst, jj = 0 ; ii < nDI + nUI ; ii++ ) {
   while ( colindI[ii] > colindJ[jj] ) {
      jj++ ;
   }
   colindI[ii] = jj ;
}
for ( ii = iifirst, iilast = -1 ; ii < nDI + nUI ; ii++ ) {
   if ( colindI[ii] >= nDJ ) { 
      break ;
   }
   iilast = ii ;
}
nupd = iilast - iifirst + 1 ;
nout = nDI + nUI - iilast - 1 ;
ops  = 2*nDI*nupd*(nupd + 2*nout) ;
if ( msglvl > 0 ) {
   fprintf(msgFile, "\n iifirst = %d, iilast = %d",
           iifirst, iilast) ;
   fprintf(msgFile, "\n nupd = %d, nout = %d, ops = %.0f",
           nupd, nout, ops) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n colindI") ;
   IVfp80(msgFile, nDI + nUI, colindI, 80, &ierr) ;
}
/*
   ----------------------------
   initialize the DChv objects
   ----------------------------
*/
Drand_setNormal(drand, 0.0, 1.0) ;
symflag = 2 ;
I = 0 ;
J = 1 ;
K = 2 ;
L = 3 ;
MARKTIME(t1) ;
chvL = DChv_new() ;
DChv_init(chvL, L, nDJ, nUJ, nUJ, symflag) ;
IVcopy(nDJ + nUJ, chvL->colind, colindJ) ;
if ( symflag == 2 ) {
   IVcopy(nDJ + nUJ, chvL->colind, colindJ) ;
}
chvK = DChv_new() ;
DChv_init(chvK, K, nDJ, nUJ, nUJ, symflag) ;
IVcopy(nDJ + nUJ, chvK->colind, colindJ) ;
if ( symflag == 2 ) {
   IVcopy(nDJ + nUJ, chvK->colind, colindJ) ;
}
chvJ = DChv_new() ;
DChv_init(chvJ, J, nDJ, nUJ, nUJ, symflag) ;
IVcopy(nDJ + nUJ, chvJ->colind, colindJ) ;
if ( symflag == 2 ) {
   IVcopy(nDJ + nUJ, chvJ->colind, colindJ) ;
}
chvI = DChv_new() ;
DChv_init(chvI, I, nDI, nUI, nUI, symflag) ;
IVcopy(nDJ + nUJ, chvI->colind, colindJ) ;
if ( symflag == 2 ) {
   IVcopy(nDI + nUI, chvI->colind, colindI) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize chv object",
        t2 - t1) ;
fflush(msgFile) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n DChv I") ;
   DChv_writeForHumanEye(chvI, msgFile) ;
   fprintf(msgFile, "\n DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
/*
   ------------------------------------
   load the entries with random entries
   ------------------------------------
*/
nentI    = nDI*(nDI + nLI + nUI) ;
nentJ    = nDJ*(nDJ + nLJ + nUJ) ;
entriesI = DChv_entries(chvI) ;
entriesJ = DChv_entries(chvJ) ;
entriesK = DChv_entries(chvK) ;
entriesL = DChv_entries(chvL) ;
keepent  = DVinit(nentJ, 0.0) ;
Drand_fillDvector(drand, nentI, entriesI) ;
/*
Drand_fillDvector(drand, nentJ, keepent) ;
*/
DVcopy(nentJ, entriesJ, keepent) ;
DVcopy(nentJ, entriesK, keepent) ;
DVcopy(nentJ, entriesL, keepent) ;
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n DChv I") ;
   DChv_writeForHumanEye(chvI, msgFile) ;
   fprintf(msgFile, "\n DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
/*
   fprintf(msgFile, "\n DChv K") ;
   DChv_writeForHumanEye(chvK, msgFile) ;
   fprintf(msgFile, "\n DChv L") ;
   DChv_writeForHumanEye(chvL, msgFile) ;
*/
}
/*
   ---------------------------------
   get the diagonal entries for chvI
   ---------------------------------
*/
nentDI  = nDI ;
diagent = DVinit(nentDI, 0.0) ;
DChv_copyEntriesToVector(chvI, 0, NULL, nentDI, diagent, 2, 1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n diag entries stored") ;
   DVfprintf(msgFile, nentDI, diagent) ;
   fflush(msgFile) ;
}
/*
   ------------------------------
   get the lower entries for chvI
   ------------------------------
*/
nentLI = (nDI*(nDI-1))/2 + nDI*nLI ;
lowerent = DVinit(nentLI, 0.0) ;
DChv_copyEntriesToVector(chvI, 0, NULL, nentLI, lowerent, 1, 0) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n lower entries stored") ;
   DVfprintf(msgFile, nentLI, lowerent) ;
   fflush(msgFile) ;
}
/*
   ---------------------------------
   get the upper entries for chvI
   ---------------------------------
*/
nentUI = (nDI*(nDI-1))/2 + nDI*nUI ;
upperent = DVinit(nentUI, 0.0) ;
DChv_copyEntriesToVector(chvI, 0, NULL, nentUI, upperent, 3, 1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n upper entries stored") ;
   DVfprintf(msgFile, nentUI, upperent) ;
   fflush(msgFile) ;
}
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n dense front") ;
   printDenseNonsymmetricFront(nDI, nLI, nUI, 
                            diagent, lowerent, upperent, msgFile) ;
}
/*
   ---------------------------------------------
   compute the update to chvJ using a 1x1 kernel
   ---------------------------------------------
*/
tempDV = DV_new() ;
DV_setSize(tempDV, nDI) ;
MARKTIME(t1) ;
DChv_denseNonsymmetricUpdate1(chvJ, nDI, nDI + nUI, diagent, lowerent,
                          upperent, colindI, iifirst, iilast, tempDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n 1x1 kernel, %8.3f CPU, %.3f mflops",
        t2 - t1, 1.e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n after update, DChv J") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
fprintf(msgFile, "\n frobnorm chvJ = %20.12e", DChv_frobNorm(chvJ)) ;
DVcopy(nentJ, keepent, entriesJ) ;
/*
   ---------------------------------------------
   compute the update to chvK using a 2x2 kernel
   ---------------------------------------------
*/
tempDV = DV_new() ;
DV_setSize(tempDV, 2*nDI) ;
MARKTIME(t1) ;
DChv_denseNonsymmetricUpdate2(chvK, nDI, nDI + nUI, diagent, lowerent,
                           upperent, colindI, iifirst, iilast, tempDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n 2x2 kernel, %8.3f CPU, %.3f mflops",
        t2 - t1, 1.e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n after update, DChv K") ;
   DChv_writeForHumanEye(chvK, msgFile) ;
}
fprintf(msgFile, "\n frobNorm chvK = %20.12e", DChv_frobNorm(chvK)) ;
DChv_sub(chvJ, chvK) ;
fprintf(msgFile, "\n 2x2: frobNorm error = %20.12e", 
        DChv_frobNorm(chvJ)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n error DChv ") ;
   DChv_writeForHumanEye(chvJ, msgFile) ;
}
/*
   ---------------------------------------------
   compute the update to chvL using a 3x3 kernel
   ---------------------------------------------
*/
/*
tempDV = DV_new() ;
DV_setSize(tempDV, 3*nDI) ;
DVcopy(nentJ, entriesJ, keepent) ;
MARKTIME(t1) ;
DChv_denseNonsymmetricUpdate3(chvJ, nDI, nDI + nUI, diagent, lowerent,
                           upperent, colindI, iifirst, iilast, tempDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n 3x3 kernel, %8.3f CPU, %.3f mflops",
        t2 - t1, 1.e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n after update, DChv L") ;
   DChv_writeForHumanEye(chvL, msgFile) ;
}
fprintf(msgFile, "\n frobNorm chvK = %20.12e", DChv_frobNorm(chvL)) ;
DChv_sub(chvJ, chvL) ;
fprintf(msgFile, "\n 3x3: frobNorm error = %20.12e", DChv_frobNorm(chvJ)) ;
*/

fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
