/*  test_copyEntriesToVector.c  */

#include "../DChv.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ------------------------------------
   test the copyEntriesToVector routine

   created -- 97jun05, cca,
   ------------------------------------
*/
{
DChv     *chv ;
double   t1, t2 ;
double   *dvec ;
Drand    *drand ;
FILE     *msgFile ;
int      copyflag, count, first, ierr, ipivot, msglvl, 
         nD, nent, nL, npivot,
         nU, pivotingflag, seed, storeflag, symflag, total ;
int      *pivotsizes ;

if ( argc != 11 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile nD nL nU symflag "
"\n         pivotingflag copyflag storeflag seed"
"\n    msglvl    -- message level"
"\n    msgFile   -- message file"
"\n    nD        -- # of rows and columns in the (1,1) block"
"\n    nL        -- # of rows in the (2,1) block"
"\n    nU        -- # of columns in the (1,2) block"
"\n    symflag   -- symmetry flag"
"\n        0 --> symmetric"
"\n        1 --> nonsymmetric"
"\n    pivotingflag -- pivoting flag"
"\n        if symflag = 1 and pivotingflag = 1 then"
"\n           construct pivotsizes[] vector"
"\n        endif"
"\n    copyflag  -- flag to denote what entries to copy"
"\n        1 --> strict lower entries"
"\n        2 --> diagonal entries"
"\n        3 --> strict upper entries"
"\n    storeflag -- flag to denote how to store entries"
"\n        0 --> store by rows"
"\n        1 --> store by columns"
"\n    seed      -- random number seed"
"\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
nD           = atoi(argv[3]) ;
nL           = atoi(argv[4]) ;
nU           = atoi(argv[5]) ;
symflag      = atoi(argv[6]) ;
pivotingflag = atoi(argv[7]) ;
copyflag     = atoi(argv[8]) ;
storeflag    = atoi(argv[9]) ;
seed         = atoi(argv[10]) ;
if ( msglvl > 0 ) {
   switch ( copyflag ) {
   case 1  : fprintf(msgFile, "\n\n strict lower entries") ; break ;
   case 2  : fprintf(msgFile, "\n\n diagonal entries") ; break ;
   case 3  : fprintf(msgFile, "\n\n strict upper entries") ; break ;
   default : 
      fprintf(stderr, "\n bad value %d for copyflag", copyflag) ;
      break ;
   }
   switch ( storeflag ) {
   case 0  : fprintf(msgFile, "\n\n STORE BY ROWS") ; break ;
   case 1  : fprintf(msgFile, "\n\n STORE BY COLUMNS") ; break ;
   default : 
      fprintf(stderr, "\n bad value %d for storeflag", storeflag) ;
      break ;
   }
}
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_init(drand) ;
Drand_setNormal(drand, 0.0, 1.0) ;
Drand_setSeed(drand, seed) ;
/*
   --------------------------
   initialize the DChv object
   --------------------------
*/
MARKTIME(t1) ;
chv = DChv_new() ;
DChv_init(chv, 37, nD, nL, nU, symflag) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize matrix objects",
        t2 - t1) ;
MARKTIME(t1) ;
if ( symflag == 0 ) {
   nent = (nD*(nD+1))/2 + nD*nU ;
} else {
   nent = nD*(nD + nL + nU) ;
}
Drand_fillDvector(drand, nent, chv->entries) ;
MARKTIME(t2) ;
fprintf(msgFile, 
        "\n CPU : %.3f to fill matrix with random numbers", t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n matrix A") ;
   DChv_writeForHumanEye(chv, msgFile) ;
   fflush(msgFile) ;
}
if ( symflag == 0 && pivotingflag == 1 ) {
/*
   ------------------------------
   create the pivotsizes[] vector
   ------------------------------
*/
   drand = Drand_new() ;
   Drand_init(drand) ;
   Drand_setUniform(drand, 1, 2.999) ;
   Drand_setSeed(drand, seed) ;
   pivotsizes = IVinit(nD, 0) ;
   Drand_fillIvector(drand, nD, pivotsizes) ;
   fprintf(msgFile, "\n initial pivotsizes[] : ") ;
   IVfp80(msgFile, nD, pivotsizes, 80, &ierr) ;
   for ( npivot = count = 0 ; npivot < nD ; npivot++ ) {
      count += pivotsizes[npivot] ;
      if ( count > nD ) {
         pivotsizes[npivot]-- ;
         count-- ;
      } 
      if ( count == nD ) {
         break ;
      }
   }
   npivot++ ;
   fprintf(msgFile, "\n final pivotsizes[] : ") ;
   IVfp80(msgFile, npivot, pivotsizes, 80, &ierr) ;
} else {
   npivot = 0 ;
   pivotsizes = NULL ;
}
/*
   --------------------------
   switch over the test cases
   --------------------------
*/
switch ( copyflag ) {
case 1 :
/*
   -----------------------------------------
   copy the strict lower entries to a vector
   -----------------------------------------
*/
   nent = (nD*(nD-1))/2 + nD*nL ;
   dvec = DVinit(nent, 0.0) ;
   MARKTIME(t1) ;
   total = DChv_copyEntriesToVector(chv, npivot, pivotsizes,
                                    nent, dvec, copyflag, storeflag) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n\n copy strict lower entries") ;
   fprintf(msgFile, "\n CPU : %.3f to copy %d entries", t2 - t1, total);
   fflush(msgFile) ;
   if ( msglvl > 3 ) {
      fprintf(msgFile, "\n\n strict lower entries of A = %d", total) ;
      DVfprintf(msgFile, nent, dvec) ;
      fflush(msgFile) ;
   }
   if ( total != nent ) {
      fprintf(stderr, "\n whoa, error!"
              "\n %d strict lower entries, total = %d",
              nent, total) ;
      exit(-1) ;
   }
   break ;
case 2 :
/*
   -------------------------------------
   copy the diagonal entries to a vector
   -------------------------------------
*/
   if ( pivotsizes == NULL ) {
      nent = nD ;
   } else {
      for ( ipivot = 0, nent = 0 ; ipivot < npivot ; ipivot++ ) {
         nent += (pivotsizes[ipivot]*(pivotsizes[ipivot]+1))/2 ;
      }
   }
   dvec = DVinit(nent, 0.0) ;
   MARKTIME(t1) ;
   total = DChv_copyEntriesToVector(chv, npivot, pivotsizes,
                                    nent, dvec, copyflag, storeflag) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n\n copy diagonal entries") ;
   fprintf(msgFile, "\n CPU : %.3f to copy %d entries", t2 - t1, total);
   fflush(msgFile) ;
   if ( msglvl > 3 ) {
      fprintf(msgFile, "\n\n diagonal entries of A = %d", total) ;
      DVfprintf(msgFile, nent, dvec) ;
      fflush(msgFile) ;
   }
   if ( total != nent ) {
      fprintf(stderr, "\n whoa, error!"
              "\n %d diagonal entries, total = %d",
              nent, total) ;
      exit(-1) ;
   }
   break ;
case 3 :
/*
   -----------------------------------------
   copy the strict upper entries to a vector
   -----------------------------------------
*/
   if ( pivotsizes == NULL ) {
      nent = (nD*(nD-1))/2 + nD * nU ;
   } else {
      for ( ipivot = first = nent = 0 ; ipivot < npivot ; ipivot++ ) {
         nent += first*pivotsizes[ipivot] ;
         first += pivotsizes[ipivot] ;
      }
      nent += nD * nU ;
   }
   dvec = DVinit(nent, 0.0) ;
   MARKTIME(t1) ;
   total = DChv_copyEntriesToVector(chv, npivot, pivotsizes,
                                    nent, dvec, copyflag, storeflag) ;
   MARKTIME(t2) ;
   fprintf(msgFile, "\n\n copy strict upper entries") ;
   fprintf(msgFile, "\n CPU : %.3f to copy %d entries", t2 - t1, total);
   fflush(msgFile) ;
   if ( msglvl > 3 ) {
      fprintf(msgFile, "\n\n strict upper entries of A = %d", total) ;
      DVfprintf(msgFile, nent, dvec) ;
      fflush(msgFile) ;
   }
   if ( total != nent ) {
      fprintf(stderr, "\n whoa, error!"
              "\n %d strict upper entries, total = %d",
              nent, total) ;
      exit(-1) ;
   }
   break ;
}
fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
