/*  DChv.h  */

#include "../IV.h"
#include "../DV.h"
#include "../DA2.h"
#include "../cfiles.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   basic chevron object

   id      -- object's id
   nD      -- number of rows and columns in the diagonal (1,1) block
   nL      -- number of rows in the lower (2,1) block
   nU      -- number of columns in the upper (1,2) block
   symflag -- symmetry flag
      0 --> chevron is symmetric
      1 --> chevron has symmetric structure and  nonsymmetric entries
      2 --> chevron has nonsymmetric structure and  nonsymmetric entries
   rowind  -- pointer to the row indices
   colind  -- pointer to the column indices
   entries -- pointer to the entries
   wrkDV   -- working storage DV object
   next    -- next DChv in a singly linked list
   -----------------------------------------------------------------
*/
typedef struct _DChv   DChv ;
struct _DChv {
   int      id       ;
   int      nD       ;
   int      nL       ;
   int      nU       ;
   int      symflag  ;
   int      *rowind  ;
   int      *colind  ;
   double   *entries ;
   DV       wrkDV    ;
   DChv     *next    ;
} ;
/*
   -------------------------------------------------
   example of storage layout for indices and entries 

   nonsymmetric case, nD = 6, nL = 4, nU = 5
   +---------------------------------------+
   |      10 11 12 13 14 15 16 17 18 19 20 |
   |   +-----------------------------------+
   |   | +---------------------------------+
   | 9 | | 9 10 11 12 13 14 15 16 17 18 19 |
   | 8 | | 8 28 29 30 31 32 33 34 35 36 37 |
   | 7 | | 7 27 45 46 47 48 49 50 51 52 53 |
   | 6 | | 6 26 44 60 61 62 63 64 65 66 67 |
   | 5 | | 5 25 43 59 73 74 75 76 77 78 79 |
   | 4 | | 4 24 42 58 72 84 85 86 87 88 89 |
   | 3 | | 3 23 41 57 71 83 +--------------+
   | 2 | | 2 22 40 56 70 82 |
   | 1 | | 1 21 39 55 69 81 |
   | 0 | | 0 20 38 54 68 80 |
   +---+ +------------------+

   symmetric case, nD = 6, nU = 5
         +---------------------------------+
         | 0  1  2  3  4  5  6  7  8  9 10 |
         +---------------------------------+
         +---------------------------------+
         | 0  1  2  3  4  5  6  7  8  9 10 |
         |   11 12 13 14 15 16 17 18 19 20 |
         |      21 22 23 44 25 26 27 28 29 |
         |         30 31 32 33 34 35 36 37 |
         |            38 39 40 41 42 43 44 |
         |               45 46 47 48 49 50 |
         +---------------------------------+
   -------------------------------------------------
*/
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor

   created -- 97apr23, cca
   -----------------------
*/
DChv *
DChv_new ( 
   void 
) ;
/*
   -----------------------
   set the default fields

   created -- 97apr23, cca
   -----------------------
*/
void
DChv_setDefaultFields (
   DChv   *chv
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage

   created -- 97apr23, cca
   --------------------------------------------------
*/
void
DChv_clearData ( 
   DChv   *chv 
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data

   created -- 97apr23, cca
   ------------------------------------------
*/
void
DChv_free ( 
   DChv   *chv 
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in instance.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------
   return the id of the chevron

   created -- 97apr19, cca
   ----------------------------
*/
int
DChv_id (
   DChv   *chv
) ;
/*
   ------------------------------------------------
   return the symmetry flag of the chevron
   return value = 0 --> chevron has
   symmetric structure and symmetric entries,
   return value = 1 --> chevron is nonsymmetric
   symmetric structure and nonsymmetric entries,
   return value = 2 --> chevron is nonsymmetric
   nonsymmetric structure and nonsymmetric entries,

   created -- 97apr19, cca
   ------------------------------------------------
*/
int
DChv_symmetryFlag (
   DChv   *chv
) ;
/*
   -----------------------------------------------
   return 1 if the chv is symmetric, 0 otherwise

   created -- 97apr19, cca
   -----------------------------------------------
*/
int
DChv_isSymmetric (
   DChv   *chv
) ;
/*
   ------------------------------------------
   return 1 if the chv is symmetric structure

   created -- 97apr19, cca
   -----------------------------------------
*/
int
DChv_hasSymmetricStructure (
   DChv   *chv
) ;
/*
   --------------------------------------------------
   return 1 if the chv is nonsymmetric, 0 otherwise

   created -- 97apr19, cca
   --------------------------------------------------
*/
int
DChv_isNonsymmetric (
   DChv   *chv
) ;
/*
   --------------------------------------------------
   fill *pnD with nD, *pnL with nL, and *pnU with nU.

   created -- 97apr19, cca
   --------------------------------------------------
*/
void
DChv_dimensions (
   DChv   *chv,
   int    *pnD,
   int    *pnL,
   int    *pnU
) ;
/*
   ----------------------------------------------
   fill *pnrow with nD + nL, *prowind with rowind

   created -- 97apr19, cca
   ----------------------------------------------
*/
void
DChv_rowIndices (
   DChv   *chv,
   int    *pnrow,
   int    **prowind
) ;
/*
   ----------------------------------------------
   fill *pncol with nD + nU, *pcolind with colind

   created -- 97apr19, cca
   ----------------------------------------------
*/
void
DChv_columnIndices (
   DChv   *chv,
   int    *pncol,
   int    **pcolind
) ;
/*
   --------------------------------------------
   fill *pentries with a pointer to the entries

   created -- 97apr19, cca
   --------------------------------------------
*/
double *
DChv_entries(
   DChv   *chv
) ;
/*
   -----------------------------------------
   return the location of the diagonal entry
   for the ichv'th chevron

   created -- 97apr19, cca
   -----------------------------------------
*/
double *
DChv_diagLocation(
   DChv   *chv,
   int    ichv
) ;
/*
   ----------------------------------------------
   return a pointer to the start of the workspace

   created -- 97jul10, cca
   ----------------------------------------------
*/
void *
DChv_workspace(
   DChv   *chv
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------
   return the number of bytes needed to store the chevron

   created -- 97jul03, cca
   ------------------------------------------------------
*/
int
DChv_nbytesNeeded (
   int   nD,
   int   nL,
   int   nU,
   int   symflag
) ;
/*
   ----------------------------------------------------------------
   return the number of bytes in the workspace owned by this object

   created -- 97may24, cca
   ----------------------------------------------------------------
*/
int
DChv_nbytesInWorkspace (
   DChv   *chv
) ;
/*
   ----------------------------------------------------------------
   set the number of bytes in the workspace owned by this object

   created -- 97aug21, cca
   ----------------------------------------------------------------
*/
void
DChv_setNbytesInWorkspace (
   DChv   *chv,
   int    nbytes
) ;
/*
   ----------------------------
   purpose -- set the fields

   created -- 97aug21, cca
   ----------------------------
*/
void
DChv_setFields (
   DChv     *chv,
   int      id,
   int      nD,
   int      nL,
   int      nU,
   int      symflag
) ;
/*
   ----------------------------
   purpose -- basic initializer

   created -- 97apr23, cca
   ----------------------------
*/
void
DChv_init (
   DChv     *chv,
   int      id,
   int      nD,
   int      nL,
   int      nU,
   int      symflag
) ;
/*
   ------------------------------------
   purpose -- initializer with pointers

   created -- 97apr23, cca
   ------------------------------------
*/
void
DChv_initWithPointers (
   DChv     *chv,
   int      id,
   int      nD,
   int      nL,
   int      nU,
   int      symflag,
   int      *rowind,
   int      *colind,
   double   *entries
) ;
/*
   -------------------------------------------------------------
   purpose -- to initialize the object from its working storage,
              used when the object is an MPI message

   created -- 97jul18
   -------------------------------------------------------------
*/
void
DChv_initFromBuffer (
   DChv   *chv
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in add.c -------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------
   add an entry into the DChv object.
   irow and jcol are global id's.

   created -- 97apr26, cca
   ----------------------------------
*/
void
DChv_addEntry (
   DChv     *chv,
   int      irow,
   int      jcol,
   double   entry
) ;
/*
   ----------------------------------
   add an entry into the DChv object.
   irow and jcol are local id's.

   created -- 97may14, cca
   ----------------------------------
*/
void
DChv_addEntryLocal (
   DChv     *chv,
   int      irow,
   int      jcol,
   double   entry
) ;
/*
   -----------------------------------------------------------------
   add a scaled multiple of a simple chevron to a DChv object.
   the indices are offsets.
   note: for this purpose, (assembling original entries into the
   matrix), the row and column indices of the chevron are identical.
   also, the indices of both the DChv object and the chvind[]
   vector are assumed to be in ascending order.

   created -- 97may01, cca
   -----------------------------------------------------------------
*/
void
DChv_addChevron (
   DChv     *chv,
   double   alpha,
   int      ichv,
   int      chvsize,
   int      chvind[],
   double   chvent[]
) ;
/*
   --------------------------------------------------------------
   assemble DChv object chvI into DChv object chvJ.
   note: the two objects must be of the same symmetry type,
         the row indices of chvI must nest into those of chvJ,
         the column indices of chvI must nest into those of chvJ.

   created -- 97may01, cca
   --------------------------------------------------------------
*/
void
DChv_assembleDChv (
   DChv   *chvJ,
   DChv   *chvI
) ;
/*
   --------------------------------------------------------------
   add DChv object chvI into chvJ 
   note: the two objects must be of the same symmetry type,
         and the same size

   created -- 97may30, cca
   --------------------------------------------------------------
*/
void
DChv_addDChv (
   DChv   *chvJ,
   DChv   *chvI
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in assemble.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------------
   purpose -- assemble the postponed data from the children


   newchv     -- DChv object to contain fully assembled front
   oldchv     -- DChv object that contains former front
   firstchild -- pointer to first child in the list of children
                 DChv objects to be merged into the new front

   return value -- # of delayed rows and columns added to the front

   created -- 97may10, cca
   ----------------------------------------------------------------
*/
int
DChv_assemblePostponedData (
   DChv   *newchv,
   DChv   *oldchv,
   DChv   *firstchild
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in copy.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------
   purpose -- copy entries to a vector. the portion copied
              can be a union of the strict lower portion,
              the diagonal portion, and the strict upper
              portion. there is one restriction, if the strict
              lower and strict upper are to be copied, the
              diagonal will also be copied.
 
   length     -- length of dvec[]
   npivot     -- number of pivots, may be 0
   pivotsizes -- vector of pivot sizes, may be NULL
   dvec[]     -- vector to receive matrix entries
   copyflag   -- flag to denote what part of the entries to copy
      1 --> copy strict lower entries
      2 --> copy diagonal entries
      3 --> copy strict upper entries
   storeflag  -- flag to denote how to store entries in dvec[]
      0 --> store by rows
      1 --> store by columns
 
   return value -- number of entries copied
 
   created -- 97jun05, cca
   ------------------------------------------------------------
*/
int
DChv_copyEntriesToVector (
   DChv     *chv,
   int      npivot,
   int      pivotsizes[],
   int      length,
   double   *dvec,
   int      copyflag, 
   int      storeflag
) ;
/*
   -------------------------------------------------------------------
   purpose -- copy large entries to a vector. the portion copied
              can be a union of the strict lower portion,
              the diagonal portion, and the strict upper
              portion. there is one restriction, if the strict
              lower and strict upper are to be copied, the
              diagonal will also be copied.
 
   npivot     -- number of pivots, may be 0
   pivotsizes -- vector of pivot sizes, may be NULL
   sizes[]    -- vector to receive row/column sizes
   ivec[]     -- vector to receive row/column indices
   dvec[]     -- vector to receive matrix entries
   copyflag   -- flag to denote what part of the entries to copy
      1 --> copy strict lower entries
      3 --> copy strict upper entries
   storeflag  -- flag to denote how to store entries in dvec[]
      0 --> store by rows
      1 --> store by columns
   droptol    -- entry to be copied must be larger than this magnitude
 
   return value -- number of entries copied
 
   created -- 97jun05, cca
   -------------------------------------------------------------------
*/
int
DChv_copyBigEntriesToVector (
   DChv     *chv,
   int      npivot,
   int      pivotsizes[],
   int      sizes[],
   int      ivec[],
   double   dvec[],
   int      copyflag, 
   int      storeflag,
   double   droptol
) ;
/*
   --------------------------------------- 
   purpose -- return the number of entries 
   whose magnitude is larger than droptol.

   countflag -- which entries to count
     1 --> count large entries in L
     3 --> count large entries in U
 
   created -- 97jun07, cca
   ---------------------------------------
*/
int
DChv_countBigEntries (
   DChv     *chv,
   int      npivot,
   int      pivotsizes[],
   int      countflag,
   double   droptol
) ;
/*
   ----------------------------------------------------------
   purpose -- copy the trailing chevron that starts at offset
 
   created -- 97may16, cca
   ----------------------------------------------------------
*/
void
DChv_copyTrailingPortion (
   DChv   *chvI,
   DChv   *chvJ,
   int    offset
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in factor.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- factor the pivot chevron

   ndelay -- number of delayed rows and columns
   pivotflag -- enable pivoting or not
      0 --> no pivoting
      1 --> enable pivoting
   pivotsizesIV -- IV object that holds the sizes of the pivots,
      used only when the front is symmetric and pivoting is enabled
   markIV -- IV object used to find pivots, 
      used only when pivoting is enabled
   tau    -- upper bound on the magnitude of the entries 
      in the factors, used only when pivoting is enabled
   pntest -- pointer to be incremented with the number of pivot tests

   return value -- # of eliminated rows and columns

   created -- 97may12, cca
   ------------------------------------------------------------------
*/
int
DChv_factor (
   DChv     *chv,
   int      ndelay,
   int      pivotflag,
   IV       *pivotsizesIV,
   IV       *markIV,
   double   tau,
   int      *pntest
) ;
/*
   ---------------------------------
   perform an internal rank-1 update

   return code --
     1 if the pivot was nonzero
     0 if the pivot was zero

   created -- 97apr26, cca
   ---------------------------------
*/
int
DChv_intUpdRank1 (
   DChv   *chv
) ;
/*
   ---------------------------------
   perform an internal rank-2 update

   return code --
     1 if the pivot was nonsingular
     0 if the pivot was singular

   created -- 97apr26, cca
   ---------------------------------
*/
int
DChv_intUpdRank2 (
   DChv   *chv
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in fill.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------
   fill DA2 object with (1,1) block

   created -- 97apr24, cca
   --------------------------------
*/
void
DChv_fill11block (
   DChv   *chv,
   DA2    *mtx
) ;
/*
   --------------------------------
   fill DA2 object with (1,2) block

   created -- 97apr24, cca
   --------------------------------
*/
void
DChv_fill12block (
   DChv   *chv,
   DA2    *mtx
) ;
/*
   --------------------------------
   fill DA2 object with (2,1) block

   created -- 97apr24, cca
   --------------------------------
*/
void
DChv_fill21block (
   DChv   *chv,
   DA2    *mtx
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in findPivot.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   purpose -- find and test a pivot

   markIV -- object that contains marking vectors
   tau    -- upper bound on magnitude of factor entries
   ndelay -- number of delayed rows and columns on input
   pirow  -- pointer to be filled with pivot row
   pjcol  -- pointer to be filled with pivot column
   pntest -- pointer to be incremented with the number of pivot tests

   return value -- size of pivot
     0 --> pivot not found
     1 --> 1x1 pivot in row *pirow and column *pjcol
     2 --> 2x2 pivot in rows and columns *pirow and *pjcol,
           symmetric front only
   
   created -- 97may12, cca
   ------------------------------------------------------------------
*/
int
DChv_findPivot (
   DChv     *chv,
   IV       *markIV,
   double   tau,
   int      ndelay,
   int      *pirow,
   int      *pjcol,
   int      *pntest
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in search.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------------------
   find the first unmarked entry in the diagonal with largest magnitude
   if ( mark[jj] == tag ) then
      we can compare this entry
   endif

   created -- 97apr24, cca
   --------------------------------------------------------------------
*/
int
DChv_maxabsInDiagonal (
   DChv     *chv,
   int      mark[],
   int      tag,
   double   *pmaxval
) ;
/*
   ----------------------------------------------------------------
   find the first unmarked entry in row irow with largest magnitude
   if ( colmark[jj] == tag ) then
      we can compare this entry
   endif

   created -- 97apr24, cca
   ----------------------------------------------------------------
*/
int
DChv_maxabsInRow (
   DChv     *chv,
   int      irow,
   int      colmark[],
   int      tag,
   double   *pmaxval
) ;
/*
   -------------------------------------------------------------------
   find the first unmarked entry in column jcol with largest magnitude

   created -- 97apr24, cca
   -------------------------------------------------------------------
*/
int
DChv_maxabsInColumn (
   DChv     *chv,
   int      jcol,
   int      rowmark[],
   int      tag,
   double   *pmaxval
) ;
/*
   ----------------------------------------------------------
   return a quasimax entry from the unmarked rows and columns
   and fill *pirow and *pjcol with its location

   created -- 97apr24, cca
   ----------------------------------------------------------
*/
double
DChv_quasimax (
   DChv     *chv,
   int      rowmark[],
   int      colmark[],
   int      tag,
   int      *pirow,
   int      *pjcol
) ;
/*
   ---------------------------------------------------------------
   find a 1x1 or 2x2 pivot using the fast Bunch-Parlett algorithm.
   used only with symmetric chevrons.

   created -- 97apr25, cca
   ---------------------------------------------------------------
*/
void
DChv_fastBunchParlettPivot (
   DChv    *chv,
   int     mark[],
   int     tag,
   int     *pirow,
   int     *pjcol
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in swap.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   swap rows irow and jrow

   created -- 97apr24, cca
   -----------------------
*/
void
DChv_swapRows (
   DChv   *chv,
   int    irow,
   int    jrow
) ;
/*
   --------------------------
   swap columns icol and jcol

   created -- 97apr24, cca
   --------------------------
*/
void
DChv_swapColumns (
   DChv   *chv,
   int      icol,
   int      jcol
) ;
/*
   -------------------------------
   swap rows and columns ii and jj

   created -- 97apr24, cca
   -------------------------------
*/
void
DChv_swapRowsAndColumns (
   DChv   *chv,
   int    ii,
   int    jj
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in testPivot.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------
   return value --
      1 if pivot passes the stability test
      0 if pivot fails the stability test

   created -- 97apr26, cca
   ---------------------------------------
*/
int
DChv_testPivot (
   DChv     *chv,
   int      irow,
   int      jcol,
   double   tau 
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in update.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 1x1 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseSymmetricUpdate1 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 2x2 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseSymmetricUpdate2 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 3x3 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseSymmetricUpdate3 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*
   -----------------------------------------
   update a DChv object using a sparse front

   created -- 97may15, cca
   -----------------------------------------
*/
void
DChv_sparseSymmetricUpdate (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 1x1 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseNonsymmetricUpdate1 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   double   diagent[],
   double   lowerent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*
   ----------------------------------------
   update a DChv object using a dense front
   using a 2x2 sdot kernel

   created -- 97may15, cca
   ----------------------------------------
*/
void
DChv_denseNonsymmetricUpdate2 (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   double   diagent[],
   double   lowerent[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*
   -----------------------------------------
   update a DChv object using a sparse front

   created -- 97may15, cca
   -----------------------------------------
*/
void
DChv_sparseNonsymmetricUpdate (
   DChv     *chvJ,
   int      nDI,
   int      ncolI,
   double   diagent[],
   int      sizesL[],
   int      lowerind[],
   double   lowerent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   int      locind[],
   int      iifirst,
   int      iilast,
   DV       *tempDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------------------
   shift the indices, entries and adjust the nD dimension.
   note: shift can be positive or negative

   created -- 97apr26, cca
   -------------------------------------------------------
*/
void
DChv_shift (
   DChv   *chv,
   int    shift
) ;
/*
   --------------------------------------
   return the value of entry (irow, jcol)

   created -- 97apr26, cca
   --------------------------------------
*/
double
DChv_entry (
   DChv   *chv,
   int    irow,
   int    jcol
) ;
/*
   ------------------------------------------
   return a pointer to the entry (irow, jcol)

   created -- 97apr30, cca
   ------------------------------------------
*/
double *
DChv_pointerToEntry (
   DChv   *chv,
   int    irow,
   int    jcol
) ;
/*
   ----------------------------------------------------------
   return the maximum magnitude of the entries in the chevron

   created -- 97may15, cca
   ----------------------------------------------------------
*/
double
DChv_maxabs (
   DChv   *chv 
) ;
/*
   -------------------------------------------------------
   return the frobenius norm of the entries in the chevron

   created -- 97may15, cca
   -------------------------------------------------------
*/
double
DChv_frobNorm (
   DChv   *chv 
) ;
/*
   -----------------------
   subtract chvI from chvJ

   created -- 97may15, cca
   -----------------------
*/
void
DChv_sub (
   DChv   *chvJ,
   DChv   *chvI 
) ;
/*
   -------------------------------
   zero the entries in the chevron

   created -- 97may15, cca
   -------------------------------
*/
void
DChv_zero (
   DChv   *chv
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------
   purpose -- to write the object to a file
              in human readable form

   created -- 97apr23, cca
   ----------------------------------------
*/
void
DChv_writeForHumanEye (
   DChv   *chv,
   FILE     *fp
) ;
/*
   -----------------------------------------------
   purpose -- print out a dense nonsymmetric front
  
   created -- 97may13, cca
   -----------------------------------------------
*/
void
printDenseNonsymmetricFront (
   int      nD,
   int      nL,
   int      nU,
   double   diagent[],
   double   lowerent[],
   double   upperent[],
   FILE     *fp
) ;
/*
   --------------------------------------------
   purpose -- print out a dense symmetric front
 
   created -- 97may13, cca
   --------------------------------------------
*/
void
printDenseSymmetricFront (
   int      nD,
   int      nU,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   double   upperent[],
   FILE     *fp
) ;
/*
   ------------------------------------------------
   purpose -- print out a sparse nonsymmetric front
 
   created -- 97may13, cca
   ------------------------------------------------
*/
void
printSparseNonsymmetricFront (
   int      nD,
   int      nL,
   int      nU,
   double   diagent[],
   int      sizesL[],
   int      lowerind[],
   double   lowerent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   FILE     *fp
) ;
/*
   ---------------------------------------------
   purpose -- print out a sparse symmetric front
 
   created -- 97may13, cca
   ---------------------------------------------
*/
void
printSparseSymmetricFront (
   int      nD,
   int      nU,
   int      npivot,
   int      pivotsizes[],
   double   diagent[],
   int      sizesU[],
   int      upperind[],
   double   upperent[],
   FILE     *fp
) ;
/*--------------------------------------------------------------------*/
