/*  util.c  */

#include "../DA2.h"
#include "../../Drand.h"

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   return the number of bytes taken by the object

   created -- 96apr19, cca
   ----------------------------------------------
*/
int
DA2_sizeOf (
   DA2   *mtx
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_sizeOf(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
return( sizeof(struct _DA2) + mtx->nowned * sizeof(double) ) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   shift the base of the entries and adjust dimensions

   mtx(0:n1-rowoff-1,0:n2-coloff-1) = mtx(rowoff:n1-1,coloff:n2-1) 

   created -- 96apr19, cca
   ---------------------------------------------------------------
*/
void
DA2_shiftBase (
   DA2   *mtx,
   int   rowoff,
   int   coloff
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_shiftbase(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
mtx->n1 -= rowoff ;
mtx->n2 -= coloff ;
mtx->entries += rowoff*mtx->inc1 + coloff*mtx->inc2 ;

return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   returns 1 if the storage is row major, otherwise returns zero.

   created -- 96apr19, cca
   --------------------------------------------------------------
*/
int
DA2_rowMajor ( 
   DA2   *mtx 
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_rowMajor(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if ( mtx->inc2 == 1 ) {
   return(1) ;
} else {
   return(0) ;
} }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   returns 1 if the storage is column major, otherwise returns zero.

   created -- 96apr19, cca
   -----------------------------------------------------------------
*/
int
DA2_columnMajor ( 
   DA2   *mtx 
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_columnMajor(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if ( mtx->inc1 == 1 ) {
   return(1) ;
} else {
   return(0) ;
} }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   transpose the matrix
 
   created -- 96jun29, cca
   -----------------------
*/
void
DA2_transpose (
   DA2   *mtx
) {
int   inc1, n1 ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_transpose(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
n1        = mtx->n1   ;
mtx->n1   = mtx->n2   ;
mtx->n2   = n1        ;
inc1      = mtx->inc1 ;
mtx->inc1 = mtx->inc2 ;
mtx->inc2 = inc1      ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   extract row[*] = mtx(irow,*)

   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractRow ( 
   DA2      *mtx, 
   double   row[],
   int      irow 
) {
double   *entries ;
int      inc2, j, k, n2 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || row == NULL || mtx->entries == NULL
   || irow < 0 || irow >= mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_extractRow(%p,%p,%d)"
           "\n bad input\n", mtx, row, irow) ;
   exit(-1) ;
}
k       = irow * mtx->inc1 ;
n2      = mtx->n2   ;
inc2    = mtx->inc2 ;
entries = mtx->entries ;
for ( j = 0 ; j < n2 ; j++, k += inc2 ) {
   row[j] = entries[k] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   extract col[*] = mtx(*,jcol)

   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractColumn ( 
   DA2      *mtx, 
   double   col[],
   int      jcol 
) {
double   *entries ;
int      i, inc1, k, n1 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || col == NULL || mtx->entries == NULL
   || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_extractColumn(%p,%p,%d)"
           "\n bad input\n", mtx, col, jcol) ;
   exit(-1) ;
}
k       = jcol * mtx->inc2 ;
n1      = mtx->n1   ;
inc1    = mtx->inc1 ;
entries = mtx->entries ;
for ( i = 0 ; i < n1 ; i++, k += inc1 ) {
   col[i] = entries[k] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set mtx(irow,*) = y[*]

   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setRow ( 
   DA2      *mtx, 
   double   row[],
   int      irow 
) {
double   *entries ;
int      inc2, j, k, n2 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || row == NULL || irow < 0 || irow >= mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_setRow(%p,%p,%d)"
           "\n bad input\n", mtx, row, irow) ;
   exit(-1) ;
}
k       = irow * mtx->inc1 ;
n2      = mtx->n2   ;
inc2    = mtx->inc2 ;
entries = mtx->entries ;
for ( j = 0 ; j < n2 ; j++, k += inc2 ) {
   entries[k] = row[j] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set mtx(*,jcol) = y[*]

   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setColumn ( 
   DA2      *mtx, 
   double   col[],
   int      jcol 
) {
double   *entries ;
int      inc1, i, k, n1 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || col == NULL || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_setColumn(%p,%p,%d)"
           "\n bad input\n", mtx, col, jcol) ;
   exit(-1) ;
}
k       = jcol * mtx->inc2 ;
n1      = mtx->n1   ;
inc1    = mtx->inc1 ;
entries = mtx->entries ;
for ( i = 0 ; i < n1 ; i++, k += inc1 ) {
   entries[k] = col[i] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   extract row[*] = mtx(irow,*)

   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractRowDV ( 
   DA2   *mtx, 
   DV    *rowDV,
   int   irow 
) {
double   *entries, *row ;
int      inc2, j, k, n2 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || rowDV == NULL || mtx->entries == NULL
   || irow < 0 || irow >= mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_extractRowDV(%p,%p,%d)"
           "\n bad input\n", mtx, rowDV, irow) ;
   exit(-1) ;
}
if ( DV_size(rowDV) < (n2 = mtx->n2) ) {
   DV_setSize(rowDV, n2) ;
}
row = DV_entries(rowDV) ;
k       = irow * mtx->inc1 ;
inc2    = mtx->inc2 ;
entries = mtx->entries ;
for ( j = 0 ; j < n2 ; j++, k += inc2 ) {
   row[j] = entries[k] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   extract col[*] = mtx(*,jcol)

   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractColumnDV ( 
   DA2   *mtx, 
   DV    *colDV,
   int   jcol 
) {
double   *entries, *col ;
int      i, inc1, k, n1 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || colDV == NULL || mtx->entries == NULL
   || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_extractColumnDV(%p,%p,%d)"
           "\n bad input\n", mtx, colDV, jcol) ;
   exit(-1) ;
}
if ( DV_size(colDV) < (n1 = mtx->n1) ) {
   DV_setSize(colDV, n1) ;
}
col = DV_entries(colDV) ;
k       = jcol * mtx->inc2 ;
inc1    = mtx->inc1 ;
entries = mtx->entries ;
for ( i = 0 ; i < n1 ; i++, k += inc1 ) {
   col[i] = entries[k] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set mtx(irow,*) = y[*]

   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setRowDV ( 
   DA2      *mtx, 
   DV       *rowDV,
   int      irow 
) {
double   *entries, *row ;
int      inc2, j, k, n2 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || rowDV == NULL || DV_size(rowDV) != (n2 = mtx->n2)
   || irow < 0 || irow >= mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_setRowDV(%p,%p,%d)"
           "\n bad input\n", mtx, rowDV, irow) ;
   exit(-1) ;
}
k       = irow * mtx->inc1 ;
inc2    = mtx->inc2 ;
entries = mtx->entries ;
row     = DV_entries(rowDV) ;
for ( j = 0 ; j < n2 ; j++, k += inc2 ) {
   entries[k] = row[j] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set mtx(*,jcol) = y[*]

   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setColumnDV ( 
   DA2      *mtx, 
   DV       *colDV,
   int      jcol 
) {
double   *col, *entries ;
int      inc1, i, k, n1 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || colDV == NULL || DV_size(colDV) != (n1 = mtx->n1)
   || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_setColumnDV(%p,%p,%d)"
           "\n bad input\n", mtx, colDV, jcol) ;
   exit(-1) ;
}
k       = jcol * mtx->inc2 ;
inc1    = mtx->inc1 ;
entries = mtx->entries ;
col     = DV_entries(colDV) ;
for ( i = 0 ; i < n1 ; i++, k += inc1 ) {
   entries[k] = col[i] ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   return a(irow, jcol)

   created -- 96apr19, cca
   -----------------------
*/
double
DA2_entry ( 
   DA2      *mtx, 
   int      irow, 
   int      jcol 
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL
   || irow < 0 || irow >= mtx->n1 
   || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_entry(%p,%d,%d)"
           "\n bad input\n", mtx, irow, jcol) ;
   if ( mtx != NULL ) {
      DA2_writeForHumanEye(mtx, stderr) ;
   }
   exit(-1) ;
}
return(mtx->entries[irow*mtx->inc1 + jcol*mtx->inc2]) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------
   set mtx(irow,jcol) = y

   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setEntry ( 
   DA2      *mtx, 
   int      irow, 
   int      jcol,
   double   value
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL
   || irow < 0 || irow >= mtx->n1 
   || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_setEntry(%p,%d,%d,%e)"
           "\n bad input\n", mtx, irow, jcol, value) ;
   exit(-1) ;
}
mtx->entries[irow*mtx->inc1 + jcol*mtx->inc2] = value ;
return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------
   add and entry

   mtx(irow,jcol) = mtx(irow,jcol) + value

   created -- 96apr19, cca
   ---------------------------------------
*/
void
DA2_addEntry ( 
   DA2      *mtx, 
   int      irow, 
   int      jcol,
   double   value
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL
   || irow < 0 || irow >= mtx->n1 
   || jcol < 0 || jcol >= mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_setEntry(%p,%d,%d,%e)"
           "\n bad input\n", mtx, irow, jcol, value) ;
   exit(-1) ;
}
mtx->entries[irow*mtx->inc1 + jcol*mtx->inc2] += value ;
return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   purpose -- test symmetry of the matrix

   if matrix is not square
      return zero
   else if |entry (i,j) - entry(j,i)| > tau then
      return zero
   else
      return 1
   endif

   created -- 96sep15, cca
   ---------------------------------------------
*/
int  
DA2_isSymmetric (
   DA2      *mtx,
   double   tau
) {
double   aij, aji ;
int      inc1, inc2, irow, jcol, n ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_isSymmetric(%p,%12.4e)"
           "\n bad input\n", mtx, tau) ;
   exit(-1) ;
}
if ( (n = mtx->n1) <= 0 || n != mtx->n2 ) {
   return(0) ;
}
inc1 = mtx->inc1 ;
inc2 = mtx->inc2 ;
for ( jcol = 0 ; jcol < n ; jcol++ ) {
   for ( irow = jcol + 1 ; irow < n ; irow++ ) {
      aij = DA2_entry(mtx, irow, jcol) ;
      aji = DA2_entry(mtx, jcol, irow) ;
      if ( fabs(aij - aji) > tau ) {
         fprintf(stdout, 
                 "\n symmetric fails"
                 "\n entry(%d,%d) = %20.12e"
                 "\n entry(%d,%d) = %20.12e",
                 irow, jcol, aij, jcol, irow, aji) ;
         return(0) ;
      }
   }
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------
   fill the matrix with uniform random numbers in [lower, upper]

   created -- 96apr20, cca
   -------------------------------------------------------------
*/
void
DA2_fillRandomUniform (
   DA2      *a,
   double   lower,
   double   upper,
   int      seed
) {
double   *entries ;
int      i, inc1, inc2, j, n1, n2 ;
Drand    drand ;
/*
   ---------------
   check the input
   ---------------
*/
if ( a == NULL
   || (n1 = a->n1) <= 0
   || (n2 = a->n2) <= 0
   || (inc1 = a->inc1) <= 0
   || (inc2 = a->inc2) <= 0
   || (entries = a->entries) == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_fillRandomUniform(%p,%f,%f,%d)"
           "\n bad input\n",
           a, lower, upper, seed) ;
   if ( a != NULL ) {
      DA2_writeForHumanEye(a, stderr) ;
   }
   exit(-1) ;
}
/*
   ----------------
   fill the entries
   ----------------
*/
Drand_setDefaultFields(&drand) ;
Drand_init(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, lower, upper) ;
for ( j = 0 ; j < n2 ; j++ ) {
   for ( i = 0 ; i < n1 ; i++ ) {
      entries[i*inc1 + j*inc2] = Drand_value(&drand) ;
   }
} 

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   fill the matrix with normal(0,1) random numbers

   created -- 96apr20, cca
   -----------------------------------------------
*/
void
DA2_fillRandomNormal (
   DA2      *a,
   double   mean,
   double   variance,
   int      seed
) {
double   *entries ;
int      i, inc1, inc2, j, n1, n2 ;
Drand    drand ;
/*
   ---------------
   check the input
   ---------------
*/
if ( a == NULL
   || (n1 = a->n1) <= 0
   || (n2 = a->n2) <= 0
   || (inc1 = a->inc1) <= 0
   || (inc2 = a->inc2) <= 0
   || (entries = a->entries) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_fillRandomNormal(%p,%d)"
           "\n bad input\n",
           a, seed) ;
   exit(-1) ;
}
/*
   ----------------
   fill the entries
   ----------------
*/
Drand_setDefaultFields(&drand) ;
Drand_init(&drand) ;
Drand_setSeed(&drand, seed) ;
Drand_setUniform(&drand, mean, variance) ;
for ( j = 0 ; j < n2 ; j++ ) {
   for ( i = 0 ; i < n1 ; i++ ) {
      entries[i*inc1 + j*inc2] = Drand_value(&drand) ;
   }
} 

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------
   fill the matrix with the identity matrix

   created -- 96jun29, cca
   ----------------------------------------
*/
void
DA2_fillWithIdentity (
   DA2   *a
) {
double   *entries ;
int      ii, inc, inc1, inc2, j, n ;
/*
   ---------------
   check the input
   ---------------
*/
if ( a == NULL
   || (n = a->n1) <= 0
   || n != a->n2
   || (inc1 = a->inc1) <= 0
   || (inc2 = a->inc2) <= 0
   || (inc1 != 1 && inc2 != 1)
   || (entries = a->entries) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_fillWithIdentity(%p)"
           "\n bad input\n", a) ;
   exit(-1) ;
}
inc = (inc1 == 1) ? inc2 : inc1 ;
DA2_zero(a) ;
for ( j = 0, ii = 0 ; j < n ; j++, ii += inc + 1 ) {
   entries[ii] = 1.0 ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------
   fill the matrix with zeros

   created -- 96jun29, cca
   --------------------------
*/
void
DA2_zero (
   DA2   *a
) {
double   *entries ;
int      i, inc1, inc2, j, n1, n2 ;
/*
   ---------------
   check the input
   ---------------
*/
if ( a == NULL
   || (n1 = a->n1) <= 0
   || (n2 = a->n2) <= 0
   || (inc1 = a->inc1) <= 0
   || (inc2 = a->inc2) <= 0
   || (entries = a->entries) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_zero(%p)"
           "\n bad input\n", a) ;
   exit(-1) ;
}
for ( j = 0 ; j < n2 ; j++ ) {
   for ( i = 0 ; i < n1 ; i++ ) {
      entries[i*inc1 + j*inc2] = 0.0 ;
   }
} 

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------
   copy one matrix into another
      A := B

   created  -- 96may03, cca
   modified -- 96aug24, cca
      nrow = min(A->n1, B->n1)
      ncol = min(A->n2, B->n2)
   ----------------------------
*/
void
DA2_copy (
   DA2   *A,
   DA2   *B
) {
double   *entA, *entB ;
int      inc1A, inc1B, inc2A, inc2B, irow, jcol, 
         ncol, ncolA, ncolB, nrow, nrowA, nrowB ;
/*
   ---------------
   check the input
   ---------------
*/
if (  A == NULL
   || (nrowA = A->n1) < 0
   || (ncolA = A->n2) < 0
   || (inc1A = A->inc1) <= 0
   || (inc2A = A->inc2) <= 0
   || (entA = A->entries) == NULL
   || B == NULL
   || (nrowB = B->n1) < 0
   || (ncolB = B->n2) < 0
   || (inc1B = B->inc1) <= 0
   || (inc2B = B->inc2) <= 0 
   || (entB = B->entries) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_copy(%p,%p)"
           "\n bad input\n", A, B) ;
   if ( A != NULL ) {
      fprintf(stderr, "\n\n first DA2 object") ;
      DA2_writeStats(A, stderr) ;
   }
   if ( B != NULL ) {
      fprintf(stderr, "\n\n second DA2 object") ;
      DA2_writeStats(B, stderr) ;
   }
   exit(-1) ;
}
nrow = (nrowA <= nrowB) ? nrowA : nrowB ;
ncol = (ncolA <= ncolB) ? ncolA : ncolB ;
if ( inc1A == 1 && inc1B == 1 ) {
   double   *colA = entA, *colB = entB ;
   for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
      for ( irow = 0 ; irow < nrow ; irow++ ) {
         colA[irow] = colB[irow] ;
      }
      colA += inc2A ;
      colB += inc2B ;
   }
} else if ( inc2A == 1 && inc2B == 1 ) {
   double   *rowA = entA, *rowB = entB ;
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
         rowA[jcol] = rowB[jcol] ;
      }
      rowA += inc1A ;
      rowB += inc1B ;
   }
} else {
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
         entA[irow*inc1A + jcol*inc2A] = entB[irow*inc1B + jcol*inc2B] ;
      }
   }
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------------
   purpose -- 
      given a square array, copy the entries in the upper triangle 
      to the lower triangle to make the matrix symmetric

   created -- 96sep15, cca
   ---------------------------------------------------------------
*/
void
DA2_copyUpperToLower (
   DA2   *mtx
) {
double   *col, *row ;
int      inc1, inc2, irow, jcol, jj, kk, n ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_copyLowerToUpper(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if ( (n = mtx->n1) < 0 || n != mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_copyLowerToUpper(%p)"
           "\n bad structure, n1 = %d, n2 = %d\n", 
           mtx, mtx->n1, mtx->n2) ;
   exit(-1) ;
}
inc1 = mtx->inc1 ;
inc2 = mtx->inc2 ;
col = row = mtx->entries ;
for ( jcol = 0 ; jcol < n ; jcol++ ) {
   for ( irow = jcol + 1, jj = irow*inc1, kk = irow*inc2 ;
         irow < n ;
         irow++, jj += inc1, kk += inc2 ) {
      col[jj] = row[kk] ;
   }
   col += inc2 ;
   row += inc1 ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- copy entries to a vector. the portion copied
              can be a union of the strict lower portion,
              the diagonal portion, and the strict upper
              portion. there is one restriction, if the strict
              lower and strict upper are to be copied, the
              diagonal will also be copied.

   length    -- length of dvec[]
   dvec[]    -- vector to receive matrix entries
   copyflag  -- flag to denote what part of the entries to move
      1 --> move strict lower entries
      2 --> move lower entries (includes the diagonal)
      3 --> move diagonal entries
      4 --> move upper entries (includes the diagonal)
      5 --> move strict upper entries
      6 --> move all entries
   storeflag -- flag to denote how to store entries in dvec[]
      0 --> store by rows
      1 --> store by columns

   return value -- number of entries copied

   created -- 97jun03, cca, dkw
   ------------------------------------------------------------
*/
int
DA2_copyEntriesToVector (
   DA2      *mtx,
   int      length,
   double   *dvec,
   int      copyflag, 
   int      storeflag
) {
int      inc1, inc2, kk, ncol, ndiag, nent, nrow ;
/*
   --------------------------------------------
   check the input, get dimensions and pointers
   and check that length is large enough
   --------------------------------------------
*/
if (  mtx == NULL || length < 0 || dvec == NULL ) {
   fprintf(stderr,
           "\n fatal error in DA2_copyEntriesToVector(%p,%d,%p,,%d,%d)"
           "\n bad input\n", mtx, length, dvec, copyflag, storeflag) ;
   exit(-1) ;
}
if ( copyflag  < 1 || copyflag > 6 ) {
   fprintf(stderr,
           "\n fatal error in DA2_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n bad input\n" 
           "\n copyflag = %d, must be\n" 
           "\n    1 --> strictly lower entries"
           "\n    2 --> lower entries"
           "\n    3 --> diagonal entries"
           "\n    4 --> strictly upper entries"
           "\n    5 --> upper entries"
           "\n    6 --> all entries",
           mtx, length, dvec, copyflag, storeflag, copyflag) ;
   exit(-1) ;
}
if ( storeflag  < 0 || storeflag > 1 ) {
   fprintf(stderr,
           "\n fatal error in DA2_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n bad input\n" 
           "\n storeflag = %d, must be\n" 
           "\n    0 --> store by rows"
           "\n    1 --> store by columns",
           mtx, length, dvec, copyflag, storeflag, storeflag) ;
   exit(-1) ;
}
nrow = mtx->n1 ;
ncol = mtx->n2 ;
inc1 = mtx->inc1 ;
inc2 = mtx->inc2 ;
if ( nrow >= ncol ) {
   ndiag = ncol ;
} else {
   ndiag = nrow ;
}
/*
   ------------------------------------------
   compute the number of entries to be copied
   ------------------------------------------
*/
switch ( copyflag ) {
case 1 : /* strictly lower entries  */
   nent = (ndiag*(ndiag - 1))/2 + (nrow - ndiag)*ncol ;
   break ;
case 2 : /* lower entries  */
   nent = (ndiag*(ndiag + 1))/2 + (nrow - ndiag)*ncol ;
   break ;
case 3 : /* diagonal entries  */
   nent = ndiag ;
   break ;
case 4 : /* upper entries  */
   nent = (ndiag*(ndiag + 1))/2 + (ncol - ndiag)*nrow ;
   break ;
case 5 : /* strictly upper entries  */
   nent = (ndiag*(ndiag - 1))/2 + (ncol - ndiag)*nrow ;
   break ;
case 6 : /* all entries  */
   nent = nrow*ncol ;
   break ;
default :
   break ;
}
if ( nent > length ) {
   fprintf(stderr,
           "\n fatal error in DA2_copyEntriesToVector(%p,%d,%p,%d,%d)"
           "\n nent = %d, buffer length = %d", 
           mtx, length, dvec, copyflag, storeflag, nent, length) ;
   exit(-1) ;
}
/*
   --------------------------------------------
   make life simple, unit stride through dvec[]
   --------------------------------------------
*/
kk = 0 ;
if ( storeflag == 0 ) {
   int      irow, jstart, jcol, jend, jj ;
   double   *row ;
/*
   -----------------
   loop over rows
   -----------------
*/
   switch ( copyflag ) {
   case 1 :
/*
      ----------------------
      strictly lower entries
      ----------------------
*/
      for ( irow = 0, row = mtx->entries, kk = 0 ; 
            irow < nrow ; 
            irow++, row += inc1 ) {
         jstart = 0 ;
         jend   = (irow < ndiag) ? irow - 1 : ndiag - 1 ;
         for ( jcol = jstart, jj = jcol*inc2 ;
               jcol <= jend ; 
               jcol++, jj += inc2, kk++ ) {
            dvec[kk] = row[jj] ;
         }
      }
      break ;
   case 2 :
/*
      ------------------------------------
      lower entries including the diagonal
      ------------------------------------
*/
      for ( irow = 0, row = mtx->entries, kk = 0 ; 
            irow < nrow ; 
            irow++, row += inc1 ) {
         jstart = 0    ;
         jend   = (irow < ndiag) ? irow : ndiag - 1 ;
         for ( jcol = jstart, jj = jcol*inc2 ;
               jcol <= jend ; 
               jcol++, jj += inc2, kk++ ) {
            dvec[kk] = row[jj] ;
         }
      }
      break ;
   case 3 :
/*
      -----------------
      just the diagonal
      -----------------
*/
      for ( irow = 0, row = mtx->entries, kk = 0 ; 
            irow < ndiag ; 
            irow++, row += inc1, kk++ ) {
         dvec[kk] = row[irow*inc2] ;
      }
      break ;
   case 4 :
/*
      ------------------------------------
      upper entries including the diagonal
      ------------------------------------
*/
      for ( irow = 0, row = mtx->entries, kk = 0 ; 
            irow < nrow ; 
            irow++, row += inc1 ) {
         jstart = irow ;
         jend   = ncol - 1 ;
         for ( jcol = jstart, jj = jcol*inc2 ;
               jcol <= jend ; 
               jcol++, jj += inc2, kk++ ) {
            dvec[kk] = row[jj] ;
         }
      }
      break ;
   case 5 :
/*
      --------------------------
      strictly the upper entries
      --------------------------
*/
      for ( irow = 0, row = mtx->entries, kk = 0 ; 
            irow < nrow ; 
            irow++, row += inc1 ) {
         jstart = irow + 1 ;
         jend   = ncol - 1 ;
         for ( jcol = jstart, jj = jcol*inc2 ;
               jcol <= jend ; 
               jcol++, jj += inc2, kk++ ) {
            dvec[kk] = row[jj] ;
         }
      }
      break ;
   case 6 :
/*
      -----------
      all entries
      -----------
*/
      for ( irow = 0, row = mtx->entries, kk = 0 ; 
            irow < nrow ; 
            irow++, row += inc1 ) {
         jstart = 0 ;
         jend   = ncol - 1 ;
         for ( jcol = jstart, jj = 0 ;
               jcol <= jend ; 
               jcol++, jj += inc2, kk++ ) {
            dvec[kk] = row[jj] ;
         }
      }
      break ;
   default :
      break ;
   }
} else {
   int      iend, ii, irow, istart, jcol ;
   double   *col ;
/*
   -----------------
   loop over columns
   -----------------
*/
   kk = 0 ;
   switch ( copyflag ) {
   case 1 :
/*
      ----------------------
      strictly lower entries
      ----------------------
*/
      for ( jcol = 0, col = mtx->entries, kk = 0 ; 
            jcol < ncol ; 
            jcol++, col += inc2 ) {
         istart = jcol + 1 ;
         iend   = nrow - 1 ;
         for ( irow = istart, ii = irow*inc1 ;
               irow <= iend ; 
               irow++, ii += inc1, kk++ ) {
            dvec[kk] = col[ii] ;
         }
      }
      break ;
   case 2 :
/*
      ------------------------------------
      lower entries including the diagonal
      ------------------------------------
*/
      for ( jcol = 0, col = mtx->entries, kk = 0 ; 
            jcol < ncol ; 
            jcol++, col += inc2 ) {
         istart = jcol ;
         iend   = nrow - 1 ;
         for ( irow = istart, ii = irow*inc1 ;
               irow <= iend ; 
               irow++, ii += inc1, kk++ ) {
            dvec[kk] = col[ii] ;
         }
      }
      break ;
   case 3 :
/*
      -----------------
      just the diagonal
      -----------------
*/
      for ( jcol = 0, col = mtx->entries, kk = 0 ; 
            jcol < ndiag ; 
            jcol++, col += inc2, kk++ ) {
         dvec[kk] = col[jcol*inc1] ;
      }
      break ;
   case 4 :
/*
      ------------------------------------
      upper entries including the diagonal
      ------------------------------------
*/
      for ( jcol = 0, col = mtx->entries, kk = 0 ; 
            jcol < ncol ; 
            jcol++, col += inc2 ) {
         istart = 0 ;
         iend   = (jcol < ndiag) ? jcol : ndiag - 1 ;
         for ( irow = istart, ii = irow*inc1 ;
               irow <= iend ; 
               irow++, ii += inc1, kk++ ) {
            dvec[kk] = col[ii] ;
         }
      }
      break ;
   case 5 :
/*
      --------------------------
      strictly the upper entries
      --------------------------
*/
      for ( jcol = 0, col = mtx->entries, kk = 0 ; 
            jcol < ncol ; 
            jcol++, col += inc2 ) {
         istart = 0 ;
         iend   = (jcol < ndiag) ? jcol - 1 : ndiag - 1 ;
         for ( irow = istart, ii = irow*inc1 ;
               irow <= iend ; 
               irow++, ii += inc1, kk++ ) {
            dvec[kk] = col[ii] ;
         }
      }
      break ;
   case 6 :
/*
      -----------
      all entries
      -----------
*/
      for ( jcol = 0, col = mtx->entries, kk = 0 ; 
            jcol < ncol ; 
            jcol++, col += inc2 ) {
         istart = 0 ;
         iend   = nrow - 1 ;
         for ( irow = istart, ii = 0 ;
               irow <= iend ; 
               irow++, ii += inc1, kk++ ) {
            dvec[kk] = col[ii] ;
         }
      }
      break ;
   default :
      break ;
   }
}
return(kk) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------- 
   subtract one matrix from another 

   A := A - B
   
   created -- 96may03, cca
   ----------------------------
*/
void
DA2_sub (
   DA2   *A,
   DA2   *B
) {
double   *entA, *entB ;
int      inc1A, inc1B, inc2A, inc2B, irow, jcol, 
         ncol, ncolA, ncolB, nrow, nrowA, nrowB ;
/*
   ---------------
   check the input
   ---------------
*/
if (  A == NULL
   || B == NULL
   || (nrowA = A->n1) <= 0
   || (ncolA = A->n2) <= 0
   || (inc1A = A->inc1) <= 0
   || (inc2A = A->inc2) <= 0
   || (nrowB = B->n1) <= 0
   || (ncolB = B->n2) <= 0
   || (inc1B = B->inc1) <= 0
   || (inc2B = B->inc2) <= 0 
   || (entA = A->entries) == NULL
   || (entB = B->entries) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_sub(%p,%p)"
           "\n bad input\n", A, B) ;
   if ( A != NULL ) {
      fprintf(stderr, "\n\n first DA2 object") ;
      DA2_writeStats(A, stderr) ;
   }
   if ( B != NULL ) {
      fprintf(stderr, "\n\n second DA2 object") ;
      DA2_writeStats(B, stderr) ;
   }
   exit(-1) ;
}
/*
fprintf(stdout, "\n debug : A") ;
DA2_writeForHumanEye(A, stdout) ;
fprintf(stdout, "\n debug : B") ;
DA2_writeForHumanEye(B, stdout) ;
*/
nrow = (nrowA <= nrowB) ? nrowA : nrowB ;
ncol = (ncolA <= ncolB) ? ncolA : ncolB ;
for ( irow = 0 ; irow < nrow ; irow++ ) {
   for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
      entA[irow*inc1A + jcol*inc2A] -= entB[irow*inc1B + jcol*inc2B] ;
   }
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------
   swap two rows of the matrix

   created -- 97mar08, cca
   ---------------------------
*/
void
DA2_swapRows (
   DA2   *a,
   int   irow1,
   int   irow2
) {
double   temp ;
double   *row1, *row2 ;
int      inc2, j, k, n2 ;
/*
   -----------
   check input
   -----------
*/
if (  a == NULL 
   || irow1 < 0 || irow1 >= a->n1
   || irow2 < 0 || irow2 >= a->n1 ) {
   fprintf(stderr, 
           "\n fatal error in DA2_swapRows(%p,%d,%d)"
           "\n bad input\n", a, irow1, irow2) ;
   exit(-1) ;
}
if (  a->n1   <= 0
   || a->inc1 <= 0
   || (n2 = a->n2) <= 0
   || (inc2 = a->inc2) <= 0
   || a->entries == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_swapRows(%p,%d,%d)"
           "\n bad structure\n", a, irow1, irow2) ;
   exit(-1) ;
}
if ( irow1 == irow2 ) {
   return ;
}
row1 = a->entries + irow1*a->inc1 ;
row2 = a->entries + irow2*a->inc1 ;
if ( inc2 == 1 ) {
   for ( j = 0 ; j < n2 ; j++ ) {
      temp    = row1[j] ;
      row1[j] = row2[j] ;
      row2[j] = temp    ;
   }
} else {
   for ( j = 0, k = 0 ; j < n2 ; j++, k += inc2 ) {
      temp    = row1[k] ;
      row1[k] = row2[k] ;
      row2[k] = temp    ;
   }
}
return ; }
 
/*--------------------------------------------------------------------*/
/*
   ------------------------------
   swap two columns of the matrix

   created -- 97mar08, cca
   ------------------------------
*/
void
DA2_swapColumns (
   DA2   *a,
   int   jcol1,
   int   jcol2
) {
double   temp ;
double   *col1, *col2 ;
int      i, inc1, k, n1 ;
/*
   -----------
   check input
   -----------
*/
if (  a == NULL
   || jcol1 < 0 || jcol1 >= a->n2
   || jcol2 < 0 || jcol2 >= a->n2 ) {
   fprintf(stderr,
           "\n fatal error in DA2_swapColumns(%p,%d,%d)"
           "\n bad input\n", a, jcol1, jcol2) ;
   exit(-1) ;
}
if (  (n1 = a->n1)   <= 0
   || (inc1 = a->inc1) <= 0
   || a->n2 <= 0
   || a->inc2 <= 0
   || a->entries == NULL ) {
   fprintf(stderr,
           "\n fatal error in DA2_swapColumns(%p,%d,%d)"
           "\n bad structure\n", a, jcol1, jcol2) ;
   exit(-1) ;
}
if ( jcol1 == jcol2 ) {
   return ;
}
col1 = a->entries + jcol1*a->inc2 ;
col2 = a->entries + jcol2*a->inc2 ;
if ( inc1 == 1 ) {
   for ( i = 0 ; i < n1 ; i++ ) {
      temp    = col1[i] ;
      col1[i] = col2[i] ;
      col2[i] = temp    ;
   }
} else {
   for ( i = 0, k = 0 ; i < n1 ; i++, k += inc1 ) {
      temp    = col1[k] ;
      col1[k] = col2[k] ;
      col2[k] = temp    ;
   }
}
return ; }
 
/*--------------------------------------------------------------------*/
