/*  sort.c  */

#include "../DA2.h"

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------
   permute the rows of the matrix
   A(*,*) = A(index(*),*)
   this method calls DA2_sortRowsUp
   but does not overwrite the index[] vector

   created -- 97nov17, cca
   -----------------------------------------
*/
void
DA2_permuteRows (
   DA2   *mtx,
   int   nrow,
   int   index[]
) {
int   *rowids ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || nrow < 0 || nrow > mtx->n1 || index == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_permuteRows(%p,%d,%p)"
           "\n bad input\n", mtx, nrow, index) ;
   exit(-1) ;
}
rowids = IVinit(nrow, -1) ;
IVcopy(nrow, rowids, index) ;
DA2_sortRowsUp(mtx, nrow, rowids) ;
IVfree(rowids) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------
   permute the columns of the matrix
   A(*,*) = A(*,index(*))
   this method calls DA2_sortColumnsUp
   but does not overwrite the index[] vector

   created -- 97nov17, cca
   -----------------------------------------
*/
void
DA2_permuteColumns (
   DA2   *mtx,
   int   ncol,
   int   index[]
) {
int   *colids ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || ncol < 0 || ncol > mtx->n2 || index == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_permuteColumns(%p,%d,%p)"
           "\n bad input\n", mtx, ncol, index) ;
   exit(-1) ;
}

colids = IVinit(ncol, -1) ;
IVcopy(ncol, colids, index) ;
DA2_sortColumnsUp(mtx, ncol, colids) ;
IVfree(colids) ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   sort the rows of the matrix in ascending order
   of the rowids[] vector. on return, rowids is
   in asending order. return value is the number
   of row swaps made.

   created -- 97jan20, cca
   ----------------------------------------------
*/
int
DA2_sortRowsUp (
   DA2   *mtx,
   int   nrow,
   int   rowids[]
) {
int   ii, minrow, minrowid, nswap, target ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || mtx->n1 < nrow || nrow < 0 || rowids == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_sortRowsUp(%p,%d,%p)"
           "\n bad input\n", mtx, nrow, rowids) ;
   if ( mtx != NULL ) {
      DA2_writeStats(mtx, stderr) ;
   }
   exit(-1) ;
}
nswap = 0 ;
if ( mtx->inc1 == 1 ) {
   double   *dvtmp ;
   int      jcol, ncol ;
   int      *ivtmp ;
/*
   ---------------------------------------------------
   matrix is stored by columns, so permute each column
   ---------------------------------------------------
*/
   ivtmp = IVinit(nrow, -1) ;
   dvtmp = DVinit(nrow, 0.0) ;
   IVramp(nrow, ivtmp, 0, 1) ;
   IV2qsortUp(nrow, rowids, ivtmp) ;
   ncol = mtx->n2 ;
   for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
      DVcopy(nrow, dvtmp, DA2_column(mtx, jcol)) ;
      DVgather(nrow, DA2_column(mtx, jcol), dvtmp, ivtmp) ;
   }
   IVfree(ivtmp) ;
   DVfree(dvtmp) ;
} else {
/*
   ----------------------------------------
   use a simple insertion sort to swap rows
   ----------------------------------------
*/
   for ( target = 0 ; target < nrow ; target++ ) {
      minrow   = target ;
      minrowid = rowids[target] ;
      for ( ii = target + 1 ; ii < nrow ; ii++ ) {
         if ( minrowid > rowids[ii] ) {
            minrow   = ii ;
            minrowid = rowids[ii] ;
         }
      }
      if ( minrow != target ) {
         rowids[minrow] = rowids[target] ;
         rowids[target] = minrowid ;
         DA2_swapRows(mtx, target, minrow) ;
         nswap++ ;
      }
   }
}

return(nswap) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------
   sort the columns of the matrix in ascending order
   of the colids[] vector. on return, colids is
   in asending order. return value is the number
   of column swaps made.

   created -- 97jan20, cca
   -------------------------------------------------
*/
int
DA2_sortColumnsUp (
   DA2   *mtx,
   int   ncol,
   int   colids[]
) {
int   ii, mincol, mincolid, nswap, target ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || mtx->n2 < ncol || ncol < 0 || colids == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_sortColumnsUp(%p,%d,%p)"
           "\n bad input\n", mtx, ncol, colids) ;
   if ( mtx != NULL ) {
      DA2_writeStats(mtx, stderr) ;
   }
   exit(-1) ;
}
nswap = 0 ;
if ( mtx->inc2 == 1 ) {
   double   *dvtmp ;
   int      irow, nrow ;
   int      *ivtmp ;
/*
   ---------------------------------------------------
   matrix is stored by rows, so permute each row
   ---------------------------------------------------
*/
   ivtmp = IVinit(ncol, -1) ;
   dvtmp = DVinit(ncol, 0.0) ;
   IVramp(ncol, ivtmp, 0, 1) ;
   IV2qsortUp(ncol, colids, ivtmp) ;
   nrow = mtx->n1 ;
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      DVcopy(ncol, dvtmp, DA2_row(mtx, irow)) ;
      DVgather(ncol, DA2_row(mtx, irow), dvtmp, ivtmp) ;
   }
   IVfree(ivtmp) ;
   DVfree(dvtmp) ;
} else {
/*
   ----------------------------------------
   use a simple insertion sort to swap cols
   ----------------------------------------
*/
   for ( target = 0 ; target < ncol ; target++ ) {
      mincol   = target ;
      mincolid = colids[target] ;
      for ( ii = target + 1 ; ii < ncol ; ii++ ) {
         if ( mincolid > colids[ii] ) {
            mincol   = ii ;
            mincolid = colids[ii] ;
         }
      }
      if ( mincol != target ) {
         colids[mincol] = colids[target] ;
         colids[target] = mincolid ;
         DA2_swapColumns(mtx, target, mincol) ;
         nswap++ ;
      }
   }
}
return(nswap) ; }

/*--------------------------------------------------------------------*/
