/*  search.c  */

#include "../DA2.h"

#define MYDEBUG 0
#define CAUTIOUS 1

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   return first q such that |a(irow,q)| = max_{j} |a(irow,j)|
   ----------------------------------------------------------
*/
int
DA2_findMaxAbsInRow (
   DA2      *a,
   int      irow,
   double   *pval
) {
double   maxval, val ;
double   *row ;
int      inc2, j, k, n2, q ;
/*
   ---------------
   check the input
   ---------------
*/
if ( a == NULL || irow < 0 || irow >= a->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_findMaxAbsInRow(%p,%d)"
           "\n bad input\n", a, irow) ;
   exit(-1) ;
}
if (  a->n1 <= 0
   || a->inc1 <= 0
   || (n2 = a->n2) <= 0
   || (inc2 = a->inc2) <= 0
   || a->entries == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_findMaxAbsInRow(%p,%d)"
           "\n bad structure\n", a, irow) ;
   exit(-1) ;
}
/*
   ----------------------------
   switch over the stride types
   ----------------------------
*/
row    = a->entries + irow*a->inc1 ;
q      = 0 ;
maxval = fabs(row[0]) ;
if ( inc2 == 1 ) {
   for ( j = 1 ; j < n2 ; j++ ) {
      if ( maxval < (val = fabs(row[j])) ) {
         maxval = val ; q = j ;
      }
   }
} else {
   for ( j = 1, k = inc2 ; j < n2 ; j++, k += inc2 ) {
      if ( maxval < (val = fabs(row[k])) ) {
         maxval = val ; q = j ;
      }
   }
}
*pval = maxval ;
return(q) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------------------
   return first q such that |a(q,jcol)| = max_{i} |a(i,jcol)|
   ----------------------------------------------------------
*/
int
DA2_findMaxAbsInColumn (
   DA2      *a,
   int      jcol,
   double   *pval
) {
double   maxval, val ;
double   *col ;
int      i, inc1, k, n1, q ;
/*
   ---------------
   check the input
   ---------------
*/
if ( a == NULL || jcol < 0 || jcol >= a->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_findMaxAbsInColumn(%p,%d)"
           "\n bad input", a, jcol) ;
   if ( a != NULL ) {
      fprintf(stderr, "\n a->n2 = %d\n", a->n2) ;
   }
   exit(-1) ;
}
if (  (n1   = a->n1)   <= 0
   || (inc1 = a->inc1) <= 0
   || a->n2   <= 0
   || a->inc2 <= 0
   || a->entries == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_findMaxAbsInColumn(%p,%d)"
           "\n bad structure\n", a, jcol) ;
   exit(-1) ;
}
/*
   ----------------------------
   switch over the stride types
   ----------------------------
*/
col    = a->entries + jcol*a->inc2 ;
q      = 0 ;
maxval = fabs(col[0]) ;
if ( inc1 == 1 ) {
   for ( i = 1 ; i < n1 ; i++ ) {
      if ( maxval < (val = fabs(col[i])) ) {
         maxval = val ; q = i ;
      }
   }
} else {
   for ( i = 1, k = inc1 ; i < n1 ; i++, k += inc1 ) {
      if ( maxval < (val = fabs(col[k])) ) {
         maxval = val ; q = i ;
      }
   }
}
*pval = maxval ;
return(q) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------
   find a local max entry

   if ( (istart,jstart) == (-1,-1) ) {
      search row 0 and column 0 and then continue
   } else if ( (istart,jstart) == (-1,jcol) ) {
      search column jcol and continue
   } else if ( (istart,jstart) == (irow,-1) ) {
      search row irow and continue
   } else {
      search row istart and column jstart and then continue
   }

   set *pirow to row
   set *picol to column
   set *pmaxval to local max value

   return value -- # of searches performed
   --------------------------------------------------------
*/
int
DA2_findLocalMax (
   DA2      *a,
   int      istart,
   int      jstart,
   int      *pirow,
   int      *pjcol,
   double   *pmaxval
) {
double   maxval, val, valcol, valrow ;
int      irow, jcol, nsearch, qcol, qrow ;
/*
   -----------
   check input
   -----------
*/
if (  a == NULL 
   || istart < -1 || istart >a->n1
   || jstart < -1 || jstart >a->n2
   || pirow == NULL 
   || pjcol == NULL 
   || pmaxval == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_findLocalMax(%p,%d,%d,%p,%p,%p)"
           "\n bad input\n", a, istart, jstart, pirow, pjcol, pmaxval) ;
   exit(-1) ;
}
if (  a->n1   <= 0
   || a->inc1 <= 0
   || a->n2   <= 0
   || a->inc2 <= 0
   || a->entries == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_findLocalMax(%p,%d,%d,%p,%p,%p)"
           "\n bad structure\n", 
           a, istart, jstart, pirow, pjcol, pmaxval) ;
   exit(-1) ;
}
/*
   ---------------------
   switch over the cases
   ---------------------
*/
if ( a->n1 == 1 ) {
   if ( a->n2 == 1 ) {
/*
      ---------------------------------------
      we have a 1 x 1 matrix, nothing to test
      ---------------------------------------
*/
      irow    = jcol = 0 ;
      maxval  = fabs(a->entries[0]) ;
      nsearch = 0 ;
   } else {
/*
      ----------------------
      matrix is a single row
      ----------------------
*/
      irow = 0 ;
      jcol = DA2_findMaxAbsInRow(a, 0, &maxval) ;
      nsearch = 1 ;
   }
} else if ( a->n2 == 1 ) {
/*
   -------------------------
   matrix is a single column
   -------------------------
*/
   irow    = DA2_findMaxAbsInColumn(a, 0, pmaxval) ;
   jcol    = 0 ;
   nsearch = 1 ;
} else if ( istart != -1 && jstart == -1 ) {
/*
   --------------------------
   start search in row istart
   --------------------------
*/
   irow    = istart ;
   jcol    = DA2_findMaxAbsInRow(a, irow, &maxval) ;
   nsearch = 1 ;
   while ( 1 ) {
      qrow = DA2_findMaxAbsInColumn(a, jcol, &val) ;
      nsearch++ ;
      if ( maxval >= val ) {
         break ;
      }
      irow = qrow ; maxval = val ;
      qcol = DA2_findMaxAbsInRow(a, irow, &val) ;
      nsearch++ ;
      if ( maxval >= val ) {
         break ;
      }
      jcol = qcol ; maxval = val ;
   }
} else if ( istart == -1 && jstart != -1 ) {
/*
   -----------------------------
   start search in column jstart
   -----------------------------
*/
   jcol = jstart ;
   irow = DA2_findMaxAbsInColumn(a, jcol, &maxval) ;
   nsearch = 1 ;
   while ( 1 ) {
      qcol = DA2_findMaxAbsInRow(a, irow, &val) ;
      nsearch++ ;
      if ( maxval >= val ) {
         break ;
      }
      jcol = qcol ; maxval = val ;
      qrow = DA2_findMaxAbsInColumn(a, jcol, &val) ;
      nsearch++ ;
      if ( maxval >= val ) {
         break ;
      }
      irow = qrow ; maxval = val ;
   }
} else {
/*
   -----------------------------------------------------------
   if (istart,jstart) == (-1,-1), set (istart,jstart) to (0,0) 
   -----------------------------------------------------------
*/
   if ( istart == -1 && jstart == -1 ) {
      istart = jstart = 0 ; 
      val    = fabs(a->entries[0]) ;
   } else {
      val = fabs(a->entries[istart*a->inc1 + jstart*a->inc2]) ;
   }
/*
   ------------------------------------------------
   search in column jstart for max entry
   search in row istart for max entry
   then start search off the max of the two entries
   ------------------------------------------------
*/
   jcol    = DA2_findMaxAbsInRow(a, istart, &valrow) ;
   irow    = DA2_findMaxAbsInColumn(a, jstart, &valcol) ;
   nsearch = 2 ;
   if ( valcol >= valrow ) {
      maxval = valrow ; 
      if ( valcol == val ) {
/*
         ------------------------------------
         entry (istart,jstart) is a local max
         ------------------------------------
*/
         irow = istart ;
         jcol = jstart ;
      } else {
/*
         -----------------------------------
         larger entry found in column jstart
         -----------------------------------
*/
         jcol = jstart ;
         while ( 1 ) {
            qcol = DA2_findMaxAbsInRow(a, irow, &val) ;
            nsearch++ ;
            if ( maxval >= val ) {
               break ;
            }
            jcol = qcol ; maxval = val ;
            qrow = DA2_findMaxAbsInColumn(a, jcol, &val) ;
            nsearch++ ;
            if ( maxval >= val ) {
               break ;
            }
            irow = qrow ; maxval = val ;
         }
      }
   } else {
/*
         --------------------------------
         larger entry found in row istart
         --------------------------------
*/
      irow = istart ;
      while ( 1 ) {
         qrow = DA2_findMaxAbsInColumn(a, jcol, &val) ;
         nsearch++ ;
         if ( maxval >= val ) {
            break ;
         }
         irow = qrow ; maxval = val ;
         qcol = DA2_findMaxAbsInRow(a, irow, &val) ;
         nsearch++ ;
         if ( maxval >= val ) {
            break ;
         }
         jcol = qcol ; maxval = val ;
      }
   }
}
*pirow = irow ;
*pjcol = jcol ;
*pmaxval = maxval ;

return(nsearch) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   find s local max elements

   rows[]  -- vector to be filled with row ids, -1 means no pivot
   cols[]  -- vector to be filled with column ids, -1 means no pivot
   vals[]  -- vector to be filled with pivot elements,
              if NULL, then ignored, otherwise filled
   rmark[] -- mark vector for rows, on entry rmark[*] != tag,
              on exit, if irow was searched, rmark[irow] = tag
   cmark[] -- mark vector for columns, on entry cmark[*] != tag
              on exit, if jcol was searched, cmark[jcol] = tag
   tag     -- tag for mark vector

   return value -- # of searches
   -----------------------------------------------------------------
*/
int
DA2_findLocalMaxes (
   DA2      *a,
   int      s,
   int      rows[],
   int      cols[],
   double   vals[],
   int      rmark[],
   int      cmark[],
   int      tag
) {
double   maxval, val ;
int      firstcol, firstrow, ierr, irow, jcol, nelem, nsearch, n1, n2, 
         qcol, qrow, success ;
/*
   -----------
   check input
   -----------
*/
if (  a == NULL 
   || s <= 0 
   || rows  == NULL 
   || cols  == NULL 
   || rmark == NULL 
   || cmark == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DA2_findLocalMaxes(%p,%d,%p,%p,%p,%p,%p,%d)"
        "\n bad input\n", a, s, rows, cols, vals, rmark, cmark, tag) ;
   exit(-1) ;
}
if (  (n1 = a->n1)   <= 0
   || a->inc1 <= 0
   || (n2 = a->n2) <= 0
   || a->inc2 <= 0
   || a->entries == NULL ) {
   fprintf(stderr, 
      "\n fatal error in DA2_findLocalMaxes(%p,%d,%p,%p,%p,%p,%p,%d)"
      "\n bad structure\n", a, s, rows, cols, vals, rmark, cmark, tag) ;
   exit(-1) ;
}
/*
   ---------------------------
   find the local max elements
   ---------------------------
*/
for ( nelem = 0 ; nelem < s ; nelem++ ) {
   rows[nelem] = cols[nelem] = -1 ;
}
#if MYDEBUG > 0
   fprintf(stdout, "\n\n s = %d, n1 = %d, n2 = %d", s, a->n1, n2) ;
#endif
nelem   = 0 ;
nsearch = 0 ;
if ( n1 >= n2 ) {
   for ( firstcol = 0 ; firstcol < n2 ; firstcol++ ) {
      jcol = firstcol ;
#if MYDEBUG > 0
      fprintf(stdout, "\n jcol = firstcol = %d", jcol) ;
      fprintf(stdout, ", cmark[%d] = %d", jcol, cmark[jcol]) ;
#endif
      if ( cmark[jcol] == tag ) {
         success = 0 ;
      } else {
         cmark[jcol] = tag ;
         irow = DA2_findMaxAbsInColumn(a, jcol, &maxval) ;
#if MYDEBUG > 0
         fprintf(stdout, "\n    irow = %d, maxval = %f", irow, maxval) ;
         fprintf(stdout, ", rmark[%d] = %d", irow, rmark[irow]) ;
#endif
         nsearch++ ;
         if ( rmark[irow] == tag ) {
            success = 0 ;
         } else {
            rmark[irow] = tag ;
            while ( 1 ) {
               qcol = DA2_findMaxAbsInRow(a, irow, &val) ;
#if MYDEBUG > 0
               fprintf(stdout, "\n    qcol = %d, val = %f", qcol, val) ;
               fprintf(stdout, ", cmark[%d] = %d", qcol, cmark[qcol]) ;
#endif
               nsearch++ ;
               if ( qcol == jcol ) {
                  success = 1 ; break ;
               } else if ( cmark[qcol] == tag ) {
                  success = 0 ; break ;
               } else {
                  cmark[qcol] = tag ;
                  if ( maxval >= val ) {
                     success = 1 ; break ;
                  } else {
                     jcol = qcol ; maxval = val ;
                  }
               }
               qrow = DA2_findMaxAbsInColumn(a, jcol, &val) ;
#if MYDEBUG > 0
               fprintf(stdout, "\n    qrow = %d, val = %f", qrow, val) ;
#endif
               nsearch++ ;
               if ( qrow == irow ) {
                  success = 1 ; break ;
               } else if ( rmark[qrow] == tag ) {
                  success = 0 ; break ;
               } else {
                  rmark[qrow] = tag ;
                  if ( maxval >= val ) {
                     success = 1 ; break ;
                  } else {
                     irow = qrow ; maxval = val ;
                  }
               }
            }
         }
         if ( success == 1 ) {
            rows[nelem] = irow   ;
            cols[nelem] = jcol   ;
            if ( vals != NULL ) {
               vals[nelem] = maxval ;
            }
#if MYDEBUG > 0
            fprintf(stdout, "\n    success, nelem = %d", nelem) ;
            fprintf(stdout, "\n    rows :") ;
            IVfp80(stdout, nelem+1, rows, 10, &ierr) ;
            fprintf(stdout, "\n    cols :") ;
            IVfp80(stdout, nelem+1, cols, 10, &ierr) ;
#endif
            if ( ++nelem == s ) {
               break ;
            }
         }
      }
   }
} else {
   for ( firstrow = 0 ; firstrow < n1 ; firstrow++ ) {
      irow = firstrow ;
#if MYDEBUG > 0
      fprintf(stdout, "\n irow = firstrow = %d", irow) ;
      fprintf(stdout, ", rmark[%d] = %d", irow, rmark[irow]) ;
#endif
      if ( rmark[irow] == tag ) {
         success = 0 ;
      } else {
         rmark[irow] = tag ;
         jcol = DA2_findMaxAbsInRow(a, irow, &maxval) ;
#if MYDEBUG > 0
         fprintf(stdout, "\n    jcol = %d, maxval = %f", jcol, maxval) ;
         fprintf(stdout, ", cmark[%d] = %d", jcol, cmark[jcol]) ;
#endif
         nsearch++ ;
         if ( cmark[jcol] == tag ) {
            success = 0 ;
         } else {
            cmark[jcol] = tag ;
            while ( 1 ) {
               qrow = DA2_findMaxAbsInColumn(a, jcol, &val) ;
#if MYDEBUG > 0
               fprintf(stdout, "\n    qrow = %d, val = %f", qrow, val) ;
               fprintf(stdout, ", rmark[%d] = %d", qrow, rmark[qrow]) ;
#endif
               nsearch++ ;
               if ( qrow == irow ) {
                  success = 1 ; break ;
               } else if ( rmark[qrow] == tag ) {
                  success = 0 ; break ;
               } else {
                  rmark[qrow] = tag ;
                  if ( maxval >= val ) {
                     success = 1 ; break ;
                  } else {
                     irow = qrow ; maxval = val ;
                  }
               }
               qcol = DA2_findMaxAbsInRow(a, irow, &val) ;
#if MYDEBUG > 0
               fprintf(stdout, "\n    qcol = %d, val = %f", qcol, val) ;
#endif
               nsearch++ ;
               if ( qcol == jcol ) {
                  success = 1 ; break ;
               } else if ( cmark[qcol] == tag ) {
                  success = 0 ; break ;
               } else {
                  cmark[qcol] = tag ;
                  if ( maxval >= val ) {
                     success = 1 ; break ;
                  } else {
                     jcol = qcol ; maxval = val ;
                  }
               }
            }
         }
         if ( success == 1 ) {
            rows[nelem] = irow   ;
            cols[nelem] = jcol   ;
            if ( vals != NULL ) {
               vals[nelem] = maxval ;
            }
#if MYDEBUG > 0
            fprintf(stdout, "\n    success, nelem = %d", nelem) ;
            fprintf(stdout, "\n    rows :") ;
            IVfp80(stdout, nelem+1, rows, 10, &ierr) ;
            fprintf(stdout, "\n    cols :") ;
            IVfp80(stdout, nelem+1, cols, 10, &ierr) ;
#endif
            if ( ++nelem == s ) {
               break ;
            }
         }
      }
   }
}
return(nsearch) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------
   purpose -- to find s quasimax entries

   examine only those rows and columns such that
   rowmark[irow] == tag and colmark[irow] == tag

   created -- 96sep07, cca
   ---------------------------------------------
*/
int
DA2_findQuasimaxes (
   DA2      *mtx,
   int      s,
   int      rowids[],
   int      colids[],
   double   maxvals[],
   int      rowmark[],
   int      colmark[],
   int      tag
) {
double   maxval ;
int      imax, irow, jcol ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || s < 0 || rowids == NULL || colids == NULL
   || rowmark == NULL || colmark == NULL ) {
   fprintf(stderr, 
         "\n fatal error in DA2_findQuasimaxes(%p,%d,%p,%p,%p,%p,%p,%d)"
         "\n bad input\n", 
         mtx, s, rowids, colids, maxvals, rowmark, colmark, tag) ;
   exit(-1) ;
}
for ( imax = 0 ; imax < s ; imax++ ) {
   DA2_quasimax(mtx, rowmark, colmark, tag, &irow, &jcol, &maxval) ;
   if ( irow == -1 ) {
      break ;
   }
   rowids[imax]  = irow ;
   rowmark[irow] = imax ;
   colids[imax]  = jcol ;
   colmark[jcol] = imax ;
   if ( maxvals != NULL ) {
      maxvals[imax] = maxval ;
   }
}
return(imax) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- return the location of a quasi-max entry,
              an entry whose magnitude is greater than or equal 
              to all entries in its row and column that belong 
              to tagged rows and columns.

   created -- 96sep07, cca
   ------------------------------------------------------------
*/
void
DA2_quasimax (
   DA2      *mtx,
   int      rowmark[],
   int      colmark[],
   int      tag,
   int      *pirow,
   int      *pjcol,
   double   *pmaxval
) {
double   maxval, val ;
double   *col, *entries, *row ;
int      ii, inc1, inc2, irow, jj, jcol, kk, ncol, nrow, qcol, qrow ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || rowmark == NULL || colmark == NULL 
   || pirow == NULL || pjcol == NULL || pmaxval == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_quasimax(%p,%p,%p,%d,%p,%p,%p)"
           "\n bad input\n", 
           mtx, rowmark, colmark, tag, pirow, pjcol, pmaxval) ;
   exit(-1) ;
}
nrow    = mtx->n1      ;
ncol    = mtx->n2      ;
inc1    = mtx->inc1    ;
inc2    = mtx->inc2    ;
entries = mtx->entries ;
/*
   ------------------------------
   find the first unmarked column
   ------------------------------
*/
qcol = -1 ;
for ( qcol = 0 ; qcol < ncol ; qcol++ ) {
   if ( colmark[qcol] == tag ) {
      break ;
   }
}
if ( qcol == -1 ) {
/*
   ----------------------------------------
   no unmarked columns, return empty-handed
   ----------------------------------------
*/
   *pirow = *pjcol = -1 ;
   *pmaxval = 0.0 ;
   return ;
}
/*
   -----------------------------------------
   find maxabs unmarked entry in this column
   -----------------------------------------
*/
col = entries + qcol * inc2 ;
maxval = 0.0 ; qrow = -1 ;
for ( ii = kk = 0 ; ii < nrow ; ii++, kk += inc1 ) {
   if (  rowmark[ii] == tag ) {
      val = fabs(col[kk]) ;
      if ( qrow == -1 || maxval < val ) {
         maxval = val ;
         qrow   = ii ;
#if MYDEBUG > 5
         fprintf(stdout, "\n 1. qrow = %d, qcol = %d, maxval = %f", 
                 qrow, qcol, maxval) ;
         fflush(stdout) ;
#endif
      }
   }
}
if ( qrow == -1 ) {
/*
   -------------------------------------
   no unmarked rows, return empty-handed
   -------------------------------------
*/
   *pirow = *pjcol = -1 ;
   *pmaxval = 0.0 ;
   return ;
}
irow = qrow ;
jcol = qcol ;
while ( 1 ) {
/*
   -----------------------------------
   look for a larger entry in this row
   -----------------------------------
*/
   row    = entries + irow * inc1 ;
   qcol   =  -1 ;
   for ( jj = kk = 0 ; jj < ncol ; jj++, kk += inc2 ) {
      if ( colmark[jj] == tag && maxval < (val = fabs(row[kk])) ) {
         maxval = val ;
         qcol   = jj  ;
#if MYDEBUG > 5
         fprintf(stdout, "\n 2. irow = %d, qcol = %d, maxval = %f", 
                 irow, qcol, maxval) ;
         fflush(stdout) ;
#endif
      }
   }
   if ( qcol == -1 ) {
/*
      -----------------------------------------------------------
      no larger entry found. set row, column and value and return
      -----------------------------------------------------------
*/
      *pirow   = irow   ;
      *pjcol   = jcol   ;
      *pmaxval = maxval ;
      return ;
   }
   jcol = qcol ;
/*
   --------------------------------------
   look for a larger entry in this column
   --------------------------------------
*/
   qrow = -1   ;
   col  = entries + jcol * inc2 ;
   for ( ii = kk = 0 ; ii < nrow ; ii++, kk += inc1 ) {
      if (  rowmark[ii] == tag && maxval < (val = fabs(col[kk])) ) {
         maxval = val ;
         qrow   = ii ;
#if MYDEBUG > 5
         fprintf(stdout, "\n 3. jcol = %d, qrow = %d, maxval = %f", 
                 jcol, qrow, maxval) ;
         fflush(stdout) ;
#endif
      }
   }
   if ( qrow == -1 ) {
/*
      -----------------------------------------------------------
      no larger entry found. set row, column and value and return
      -----------------------------------------------------------
*/
      *pirow   = irow   ;
      *pjcol   = jcol   ;
      *pmaxval = maxval ;
      return ;
   }
   irow = qrow ;
}
/*
   ------------------------------------
   set row, column and value and return
   ------------------------------------
*/
*pirow   = irow   ;
*pjcol   = jcol   ;
*pmaxval = maxval ;
#if CAUTIOUS > 0
{
double ijval = fabs(entries[irow*inc1 + jcol*inc2]) ;
col = entries + jcol*inc2 ;
for ( ii = 0 ; ii < nrow ; ii++ ) {
   if ( rowmark[ii] == tag && ijval < fabs(col[ii*inc1]) ) {
      fprintf(stderr, "\n\n HEY THERE in DA2_quasimax()"
              "\n irow = %d, jcol = %d, fabs(ent(%d,%d)) = %e"
              "\n ii = %d, jcol = %d, fabs(ent(%d,%d)) = %e",
              irow, jcol, irow, jcol, ijval,
              ii, jcol, ii, jcol, fabs(col[ii*inc1])) ;
      exit(-1) ;
   }
}
row = entries + irow*inc1 ;
for ( jj = 0 ; jj < ncol ; jj++ ) {
   if ( colmark[jj] == tag && ijval < fabs(row[jj*inc2]) ) {
      fprintf(stderr, "\n\n HEY THERE in DA2_quasimax()"
              "\n irow = %d, jcol = %d, fabs(ent(%d,%d)) = %e"
              "\n irow = %d, jj = %d, fabs(ent(%d,%d)) = %e",
              irow, jj, irow, jj, ijval,
              irow, jj, irow, jj, fabs(row[jj*inc2])) ;
      exit(-1) ;
   }
}
}
#endif

return ; }
   
/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- return the location of a quasi-max entry,
              an entry whose magnitude is greater than or equal 
              to all entries in its row and column that belong 
              to tagged rows and columns.

   note: the matrix is square and symmetric, with entries
         stored only in the upper triangle.

   created -- 96sep20, cca
   ------------------------------------------------------------
*/
void
DA2_symQuasimax (
   DA2      *mtx,
   int      mark[],
   int      tag,
   int      *pirow,
   int      *pjcol,
   double   *pmaxval
) {
double   maxval, val ;
double   *entries ;
int      first, ierr, ii, inc1, inc2, jj, 
         n, offset, qcol, qrow, second ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mark == NULL 
   || pirow == NULL || pjcol == NULL || pmaxval == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_symQuasimax(%p,%p,%d,%p,%p,%p)"
           "\n bad input\n", 
           mtx, mark, tag, pirow, pjcol, pmaxval) ;
   exit(-1) ;
}
if (  (n = mtx->n1) < 0 || n != mtx->n2
   || (inc1 = mtx->inc1) < 0
   || (inc2 = mtx->inc2) < 0
   || (entries = mtx->entries) == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_symQuasimax(%p,%p,%d,%p,%p,%p)"
           "\n bad structure\n", 
           mtx, mark, tag, pirow, pjcol, pmaxval) ;
   DA2_writeStats(mtx, stderr) ;
   exit(-1) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n n = %d, inc1 = %d, inc2 = %d", n, inc1, inc2) ;
DA2_writeForHumanEye(mtx, stdout) ;
fprintf(stdout, "\n mark :") ;
IVfp80(stdout, n, mark, 6, &ierr) ;
fflush(stdout) ;
#endif
/*
   --------------------------------------
   find the first unmarked row and column
   --------------------------------------
*/
for ( qcol = 0 ; qcol < n ; qcol++ ) {
   if ( mark[qcol] == tag ) {
      break ;
   }
}
if ( qcol == n ) {
   *pirow = *pjcol = -1 ;
   *pmaxval = 0.0 ;
   return ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n qcol = %d", qcol) ;
fflush(stdout) ;
#endif
/*
   ----------------------------------------------------------
   find the entry of maximum magnitude in this row and column
   ----------------------------------------------------------
*/
offset = qcol*inc2 ;
qrow   = -1 ;
maxval = 0.0 ;
for ( ii = 0 ; ii < qcol ; ii++ ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n    ii = %d, offset = %d, val = %12.4e",
           ii, offset, fabs(entries[offset])) ;
   fflush(stdout) ;
#endif
   if ( mark[ii] == tag && maxval < (val = fabs(entries[offset])) ) {
      maxval = val ;
      qrow   = ii  ;
   }
   offset += inc1 ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n    qcol = %d, offset = %d, val = %12.4e",
        qcol, offset, fabs(entries[offset])) ;
#endif
if ( maxval < (val = fabs(entries[offset])) ) {
   maxval = val  ;
   qrow   = qcol ;
}
offset += inc2 ;
for ( jj = qcol + 1 ; jj < n ; jj++ ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n    qcol = %d, offset = %d, val = %12.4e",
           qcol, offset, fabs(entries[offset])) ;
#endif
   if ( mark[jj] == tag && maxval < (val = fabs(entries[offset])) ) {
      maxval = val ;
      qrow   = jj  ;
   }
   offset += inc2 ;
}
if ( qrow == qcol ) {
   *pirow = *pjcol = qcol ;
   *pmaxval = maxval ;
   return ;
}
/*
   ---------------------------------------------
   loop, looking for entries of larger magnitude
   ---------------------------------------------
*/
first  = qcol ;
second = qrow ;
while ( 1 ) {
   qcol   = second ;
   offset = qcol * inc2 ;
   qrow   = -1 ;
   for ( ii = 0 ; ii < qcol ; ii++ ) {
      if ( mark[ii] == tag && maxval < (val = fabs(entries[offset])) ) {
         maxval = val ;
         qrow   = ii  ;
      }
      offset += inc1 ;
   }
   if ( maxval < (val = fabs(entries[offset])) ) {
      maxval = val  ;
      qrow   = qcol ;
   }
   offset += inc2 ;
   for ( jj = qcol + 1 ; jj < n ; jj++ ) {
      if ( mark[jj] == tag && maxval < (val = fabs(entries[offset])) ) {
         maxval = val ;
         qrow   = jj  ;
      }
      offset += inc2 ;
   }
   if ( qrow == qcol ) {
      *pirow   = *pjcol = qcol ;
      *pmaxval = maxval ;
      return ;
   } else if ( qrow == -1 ) {
      *pirow   = first  ;
      *pjcol   = second ;
      *pmaxval = maxval ;
      return ;
   }
   first  = second ;
   second = qrow ;
}
*pirow   = first  ;
*pjcol   = second ;
*pmaxval = maxval ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- return the location of the unmarked diagonal
      entry with the largest magnitude.

   created -- 97jan11, cca
   ------------------------------------------------------------
*/
int
DA2_maxDiagPivot (
   DA2      *mtx,
   int      mark[],
   int      tag
) {
double   maxdiag, val ;
double   *entries ;
int      ii, kk, loc, n, stride ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mark == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_maxDiagPivot(%p,%p,%d)"
           "\n bad input\n", mtx, mark, tag) ;
   exit(-1) ;
}
if (  (n = mtx->n1) < 0 || n != mtx->n2
   || mtx->inc1 < 0 || mtx->inc2 < 0
   || (entries = mtx->entries) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_maxDiagPivot(%p,%p,%d)"
        "\n bad structure\n", mtx, mark, tag) ;
   DA2_writeStats(mtx, stderr) ;
   exit(-1) ;
}
/*
   -----------------------------------------------------
   find the unmarked diagonal entry of maximum magnitude
   -----------------------------------------------------
*/
loc     = -1 ;
maxdiag = 0.0 ;
stride  = mtx->inc1 + mtx->inc2 ;
for ( ii = 0, kk = 0 ; ii < n ; ii++, kk += stride ) {
   if (  mark[ii] == tag ) {
     val = fabs(entries[kk]) ;
     if  ( loc == -1 || maxdiag < val ) {
         loc = ii ;
         maxdiag = val ;
      }
   }
}
return(loc) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------------------
   purpose -- find a 1x1 or 2x2 pivot using 
              the fast Bunch-Parlett algorithm

   note: the matrix is square and symmetric, with entries
         stored only in the upper triangle.

   created -- 96sep20, cca
   ------------------------------------------------------------
*/
void
DA2_fastBunchParlettPivot (
   DA2      *mtx,
   int      mark[],
   int      tag,
   int      *pirow,
   int      *pjcol
) {
double   maxdiag, gamma_r, gamma_s, val ;
double   *entries ;
int      ierr, ii, inc1, inc2, kk, n, r, s, t ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mark == NULL || pirow == NULL || pjcol == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DA2_fastBunchParlettPivot(%p,%p,%d,%p,%p)"
           "\n bad input\n", 
           mtx, mark, tag, pirow, pjcol) ;
   exit(-1) ;
}
if (  (n = mtx->n1) < 0 || n != mtx->n2
   || (inc1 = mtx->inc1) < 0
   || (inc2 = mtx->inc2) < 0
   || (entries = mtx->entries) == NULL ) {
   fprintf(stderr, 
        "\n fatal error in DA2_fastBunchParlettPivot(%p,%p,%d,%p,%p)"
        "\n bad structure\n", 
        mtx, mark, tag, pirow, pjcol) ;
   DA2_writeStats(mtx, stderr) ;
   exit(-1) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n n = %d, inc1 = %d, inc2 = %d", n, inc1, inc2) ;
DA2_writeForHumanEye(mtx, stdout) ;
fprintf(stdout, "\n mark :") ;
IVfp80(stdout, n, mark, 6, &ierr) ;
fflush(stdout) ;
#endif
/*
   -----------------------------------------------------
   find the unmarked diagonal entry of maximum magnitude
   -----------------------------------------------------
*/
r = -1 ;
maxdiag = 0.0 ;
for ( ii = 0, kk = 0 ; ii < n ; ii++, kk += inc1 + inc2 ) {
   if (  mark[ii] == tag ) {
     val = fabs(entries[kk]) ;
     if  ( r == -1 || maxdiag < val ) {
         r = ii ;
         maxdiag = val ;
      }
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n r = %d, maxdiag = %e", r, maxdiag) ;
#endif
if ( r == -1 ) {
/*
   ----------------------------
   no unmarked rows and columns
   ----------------------------
*/
   *pirow = *pjcol = -1 ;
   return ; 
}
/*
   -------------------------------------------------------------------
   find the offdiagonal entry of maximum magnitude in row and column r
   -------------------------------------------------------------------
*/
s = -1 ;
gamma_r = 0.0 ;
#if MYDEBUG > 0
fprintf(stdout, "\n inc1 = %d, inc2 = %d", inc1, inc2) ;
#endif
for ( ii = 0, kk = r*inc2 ; ii < r ; ii++, kk += inc1 ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n 1. entries[%d] = %e", kk, entries[kk]) ;
   fprintf(stdout, ", r = %d, inc2 = %d, kk = %d", r, inc2, kk) ;
#endif
   if ( mark[ii] == tag ) {
      val = fabs(entries[kk]) ;
      if ( s == -1 || gamma_r < val ) {
         s = ii ;
         gamma_r = val ; 
      }
   }
}
for ( ii = r + 1, kk = r*inc1 + (r+1)*inc2 ; 
      ii < n ; 
      ii++, kk += inc2 ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n 2. entries[%d] = %e", kk, entries[kk]) ;
   fprintf(stdout, ", r = %d, inc2 = %d, kk = %d", r, inc2, kk) ;
#endif
   if ( mark[ii] == tag ) {
      val = fabs(entries[kk]) ;
      if ( s == -1 || gamma_r < val ) {
         s = ii ;
         gamma_r = val ; 
      }
   }
}
#if MYDEBUG > 0
fprintf(stdout, "\n s = %d, gamma_r = %e", s, gamma_r) ;
#endif
if ( s == -1 ) {
/*
   -------------------------------------
   the only unmarked row and column is r
   -------------------------------------
*/
   *pirow = *pjcol = r ;
   return ; 
}
/*
   ---------------
   loop until done
   ---------------
*/
if ( maxdiag >= 0.6404*gamma_r ) {
/*
   -----------
   1 x 1 pivot
   -----------
*/
#if MYDEBUG > 0
   fprintf(stdout, "\n accepting 1x1 pivot") ;
#endif
   *pirow = *pjcol = r ;
   return ; 
} else {
   while ( 1 ) {
/*
   ----------------------------------------------
   find
      t = index of max off diag entry in column s
      gamma_s is its magnitude
   ----------------------------------------------
*/
      t = -1 ;
      gamma_s = 0.0 ;
      for ( ii = 0, kk = s*inc2 ; ii < s ; ii++, kk += inc1 ) {
#if MYDEBUG > 0
   fprintf(stdout, "\n 3. entries[%d] = %e", kk, entries[kk]) ;
   fprintf(stdout, ", s = %d, inc2 = %d, kk = %d", s, inc2, kk) ;
#endif
         if ( mark[ii] == tag ) {
            val = fabs(entries[kk]) ;
            if ( t == -1 || gamma_s < val ) {
               t = ii ;
               gamma_s = val ; 
            }
         }
      }
      for ( ii = s + 1, kk = s*inc1 + (s+1)*inc2 ; 
            ii < n ; 
            ii++, kk += inc2 ){
#if MYDEBUG > 0
   fprintf(stdout, "\n 4. entries[%d] = %e", kk, entries[kk]) ;
   fprintf(stdout, ", s = %d, inc2 = %d, kk = %d", s, inc2, kk) ;
#endif
         if ( mark[ii] == tag ) {
            val = fabs(entries[kk]) ;
            if ( t == -1 || gamma_s < val ) {
               t = ii ;
               gamma_s = val ; 
            }
         }
      }
#if MYDEBUG > 0
      fprintf(stdout, "\n t = %d, gamma_s = %e", t, gamma_s) ;
#endif
      if ( t == r || gamma_s == gamma_r ) {
/*
         ----------------------
         return the 2 x 2 pivot
         ----------------------
*/
#if MYDEBUG > 0
         fprintf(stdout, "\n accepting 2x2 pivot") ;
#endif
         if ( r < s ) {
            *pirow = r ;
            *pjcol = s ;
         } else {
            *pirow = s ;
            *pjcol = r ;
         }
         return ;
      } else {
/*
         --------------------------------
         keep looking for a local maximum
         --------------------------------
*/
         r = s ;
         gamma_r = gamma_s ;
         s = t ;
      }
   }
}
return ; }

/*--------------------------------------------------------------------*/
