/*  scatter.c  */

#include "../DA2.h"

#define CAUTIOUS 0

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   scatter entries of A into B
      B(rowids, colids) = A
   rowids -- ids of rows to be scattered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be scattered,
      must be not NULL if B->n2 != A->n2

   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_scatter (
   DA2   *B,
   int   *rowids,
   int   *colids,
   DA2   *A
) {
int      ierr, ii, inc1A, inc1B, inc2A, inc2B, irow, jj, jcol, 
         n1A, n1B, n2A, n2B, rc ;
double   *entA, *entB ;
/*
   ---------------
   check the input
   ---------------
*/
if (  B == NULL || A == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n B and/orA is NULL\n", A, rowids, colids, B) ;
   exit(-1) ;
}
n1B   = B->n1      ;
n2B   = B->n2      ;
inc1B = B->inc1    ;
inc2B = B->inc2    ;
entB  = B->entries ;
n1A   = A->n1      ;
n2A   = A->n2      ;
inc1A = A->inc1    ;
inc2A = A->inc2    ;
entA  = A->entries ;
#if CAUTIOUS
if ( n1B <= 0 || n2B <= 0 || inc1B <= 0 || inc2B <= 0 || entB == NULL ){
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n B is invalid\n", A, rowids, colids, B) ;
   DA2_writeStats(B, stderr) ;
   exit(-1) ;
}
if ( n1A <= 0 || n2A <= 0 || inc1A <= 0 || inc2A <= 0 || entA == NULL ){
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n A is invalid\n", A, rowids, colids, B) ;
   DA2_writeStats(A, stderr) ;
   exit(-1) ;
}
if ( n1B != n1A ) {
/*
   ---------------------------
   check out the rowids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n1B < n1A || rowids == NULL ) {
      goto rowError ;
   } else {
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         irow = rowids[ii] ;
         if ( irow < 0 || irow >= n1B ) {
            goto rowError ;
         }
      }
   }
}
if ( n2B != n2A ) {
/*
   ---------------------------
   check out the colids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n2B < n2A || colids == NULL ) {
      goto columnError ;
      rc = 1 ;
   } else {
      for ( jj = 0 ; jj < n2A ; jj++ ) {
         jcol = colids[jj] ;
         if ( jcol < 0 || jcol >= n2B ) {
            goto columnError ;
         }
      }
   }
}
#endif
/*
   --------------------------
   switch over the four cases
   --------------------------
*/
if ( colids != NULL && rowids != NULL ) {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         irow = rowids[ii] ;
         entB[irow*inc1B + jcol*inc2B] = entA[ii*inc1A + jj*inc2A] ;
      }
   }
} else if ( colids == NULL && rowids != NULL ) {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         irow = rowids[ii] ;
         entB[irow*inc1B + jj*inc2B] = entA[ii*inc1A + jj*inc2A] ;
      }
   }
} else if ( colids != NULL && rowids == NULL ) {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         entB[ii*inc1B + jcol*inc2B] = entA[ii*inc1A + jj*inc2A] ;
      }
   }
} else {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         entB[ii*inc1B + jj*inc2B] = entA[ii*inc1A + jj*inc2A] ;
      }
   }
}
#if CAUTIOUS
/*
   -----------
   error jumps
   -----------
*/
rowError :
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n bad rowids vector\n", A, rowids, colids, B) ;
   if ( n1A > 0 && rowids != NULL ) {
      fprintf(stderr, "\n rowids") ;
      IVfp80(stderr, n1A, rowids, 9, &ierr) ;
   }
   exit(-1) ;
columnError :
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n bad colids vector\n", A, rowids, colids, B) ;
   if ( n2A > 0 && colids != NULL ) {
      fprintf(stderr, "\n colids") ;
      IVfp80(stderr, n2A, colids, 9, &ierr) ;
   }
   exit(-1) ;
#endif

return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   scatter/add entries of A into B
      B(rowids, colids) += A
   rowids -- ids of rows to be scattered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be scattered,
      must be not NULL if B->n2 != A->n2

   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_scatterAdd (
   DA2   *B,
   int   *rowids,
   int   *colids,
   DA2   *A
) {
int      ierr, ii, inc1A, inc1B, inc2A, inc2B, irow, jj, jcol, 
         n1A, n1B, n2A, n2B, rc ;
double   *entA, *entB ;
/*
   ---------------
   check the input
   ---------------
*/
if (  B == NULL || A == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n B and/orA is NULL\n", A, rowids, colids, B) ;
   exit(-1) ;
}
n1B   = B->n1      ;
n2B   = B->n2      ;
inc1B = B->inc1    ;
inc2B = B->inc2    ;
entB  = B->entries ;
n1A   = A->n1      ;
n2A   = A->n2      ;
inc1A = A->inc1    ;
inc2A = A->inc2    ;
entA  = A->entries ;
#if CAUTIOUS
if ( n1B <= 0 || n2B <= 0 || inc1B <= 0 || inc2B <= 0 || entB == NULL ){
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n B is invalid\n", A, rowids, colids, B) ;
   DA2_writeStats(B, stderr) ;
   exit(-1) ;
}
if ( n1A <= 0 || n2A <= 0 || inc1A <= 0 || inc2A <= 0 || entA == NULL ){
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n A is invalid\n", A, rowids, colids, B) ;
   DA2_writeStats(A, stderr) ;
   exit(-1) ;
}
if ( n1B != n1A ) {
/*
   ---------------------------
   check out the rowids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n1B < n1A || rowids == NULL ) {
      goto rowError ;
   } else {
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         irow = rowids[ii] ;
         if ( irow < 0 || irow >= n1B ) {
            goto rowError ;
         }
      }
   }
}
if ( n2B != n2A ) {
/*
   ---------------------------
   check out the colids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n2B < n2A || colids == NULL ) {
      goto columnError ;
      rc = 1 ;
   } else {
      for ( jj = 0 ; jj < n2A ; jj++ ) {
         jcol = colids[jj] ;
         if ( jcol < 0 || jcol >= n2B ) {
            goto columnError ;
         }
      }
   }
}
#endif
/*
   --------------------------
   switch over the four cases
   --------------------------
*/
if ( colids != NULL && rowids != NULL ) {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         irow = rowids[ii] ;
         entB[irow*inc1B + jcol*inc2B] += entA[ii*inc1A + jj*inc2A] ;
      }
   }
} else if ( colids == NULL && rowids != NULL ) {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         irow = rowids[ii] ;
         entB[irow*inc1B + jj*inc2B] += entA[ii*inc1A + jj*inc2A] ;
      }
   }
} else if ( colids != NULL && rowids == NULL ) {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         entB[ii*inc1B + jcol*inc2B] += entA[ii*inc1A + jj*inc2A] ;
      }
   }
} else {
   for ( jj = 0 ; jj < n2A ; jj++ ) {
      for ( ii = 0 ; ii < n1A ; ii++ ) {
         entB[ii*inc1B + jj*inc2B] += entA[ii*inc1A + jj*inc2A] ;
      }
   }
}
#if CAUTIOUS
/*
   -----------
   error jumps
   -----------
*/
rowError :
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n bad rowids vector\n", A, rowids, colids, B) ;
   if ( n1A > 0 && rowids != NULL ) {
      fprintf(stderr, "\n rowids") ;
      IVfp80(stderr, n1A, rowids, 9, &ierr) ;
   }
   exit(-1) ;
columnError :
   fprintf(stderr, "\n fatal error in DA2_scatter(%p,%p,%p,%p)"
           "\n bad colids vector\n", A, rowids, colids, B) ;
   if ( n2A > 0 && colids != NULL ) {
      fprintf(stderr, "\n colids") ;
      IVfp80(stderr, n2A, colids, 9, &ierr) ;
   }
   exit(-1) ;
#endif

return ; }

/*--------------------------------------------------------------------*/
