/*  norms.c  */

#include "../DA2.h"

#define CAUTIOUS 1

/*--------------------------------------------------------------------*/
/*
   -------------------------------------
   return the entry of maximum magnitude
 
   created -- 96sep15, cca
   -------------------------------------
*/
double
DA2_maxabs (
   DA2   *a
) {
double   maxval, val ;
double   *entries, *row ;
int      inc1, inc2, irow, jcol, kk, n1, n2 ;
/*
   ---------------
   check the input
   ---------------
*/
if (  a == NULL 
   || (n1 = a->n1) < 0
   || (n2 = a->n2) < 0
   || (inc1 = a->inc1) < 0
   || (inc2 = a->inc2) < 0 ) {
   fprintf(stderr, 
           "\n fatal error in DA2_maxabs(%p)"
           "\n bad input\n", a ) ;
   exit(-1) ;
}
entries = a->entries ;
maxval = 0.0 ;
row = entries ;
for ( irow = 0 ; irow < n1 ; irow++ ) {
   for ( jcol = 0, kk = 0 ; jcol < n2 ; jcol++, kk += inc2 ) {
      if ( maxval < (val = fabs(row[kk])) ) {
         maxval = val ;
      }
   }
   row += inc1 ;
}
return(maxval) ; }
   
/*--------------------------------------------------------------------*/
/*
   ---------------------------------------
   return the frobenius norm of the matrix

   created -- 96may03, cca
   ---------------------------------------
*/
double
DA2_frobNorm (
   DA2   *mtx
) {
double   norm ;
int      ncol, nrow ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL ) {
   fprintf(stderr, 
           "\n fatal error in DA2_frobNorm(%p) "
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if ( (nrow = mtx->n1) <= 0 || (ncol = mtx->n2) <= 0 ) {
   return(0.0) ;
}
norm  = 0 ;
if ( mtx->inc1 == 1 ) {
   double   *col ;
   int      inc2 = mtx->inc2, irow, jcol ;

   for ( jcol = 0, col = mtx->entries ; 
         jcol < ncol ; 
         jcol++, col += inc2 ) {
      for ( irow = 0 ; irow < nrow ; irow++ ) {
         norm += col[irow] * col[irow] ;
      }
   }
} else if ( mtx->inc2 == 1 ) {
   double   *row ;
   int      inc1 = mtx->inc1, irow, jcol ;

   for ( irow = 0, row = mtx->entries ; 
         irow < nrow ; 
         irow++, row += inc1 ) {
      for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
         norm += row[jcol] * row[jcol] ;
      }
   }
} else {
   double   val ;
   double   *entries = mtx->entries ;
   int      inc1 = mtx->inc1, inc2 = mtx->inc2, irow, jcol ;
   
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
         val = entries[irow*inc1 + jcol*inc2] ;
         norm += val*val ;
      }
   }
}
norm = sqrt(norm) ;

return(norm) ; }
   
/*--------------------------------------------------------------------*/
/*
   ---------------------------------
   return the one-norm of the matrix

   created -- 96may03, cca
   ---------------------------------
*/
double
DA2_oneNorm (
   DA2   *mtx
) {
double   norm ;
int      ncol, nrow ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_oneNorm(%p) "
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if ( (nrow = mtx->n1) <= 0 || (ncol = mtx->n2) <= 0 ) {
   return(0.0) ;
}
norm = 0.0 ;
if ( mtx->inc1 == 1 ) {
   double   sum, val ;
   double   *col ;
   int      inc2 = mtx->inc2, irow, jcol ;

   for ( jcol = 0, col = mtx->entries ; 
         jcol < ncol ; 
         jcol++, col += inc2 ) {
      sum = 0.0 ;
      for ( irow = 0 ; irow < nrow ; irow++ ) {
         if ( (val = col[irow]) < 0 ) {
            sum -= val ;
         } else if ( val > 0 ) {
            sum += val ;
         }
      }
      if ( norm < sum ) {
         norm = sum ;
      }
   }
} else {
   double   sum, val ;
   double   *col ;
   int      inc1 = mtx->inc1, inc2 = mtx->inc2, irow, jcol, kk ;

   for ( jcol = 0, col = mtx->entries ; 
         jcol < ncol ; 
         jcol++, col += inc2 ) {
      sum = 0.0 ;
      for ( irow = 0, kk = 0 ; irow < nrow ; irow++, kk += inc1 ) {
         if ( (val = col[kk]) < 0 ) {
            sum -= val ;
         } else if ( val > 0 ) {
            sum += val ;
         }
      }
      if ( norm < sum ) {
         norm = sum ;
      }
   }
}

return(norm) ; }
   
/*--------------------------------------------------------------------*/
/*
   --------------------------------------
   return the infinity-norm of the matrix

   created -- 96may03, cca
   --------------------------------------
*/
double
DA2_infinityNorm (
   DA2   *mtx
) {
double   norm ;
int      ncol, nrow ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_infinityNorm(%p) "
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if ( (nrow = mtx->n1) <= 0 || (ncol = mtx->n2) <= 0 ) {
   return(0.0) ;
}
norm = 0.0 ;
if ( mtx->inc2 == 1 ) {
   double   sum, val ;
   double   *row ;
   int      inc1 = mtx->inc1, irow, jcol ;

   for ( irow = 0, row = mtx->entries ; 
         irow < nrow ; 
         irow++, row += inc1 ) {
      sum = 0.0 ;
      for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
         if ( (val = row[jcol]) < 0 ) {
            sum -= val ;
         } else if ( val > 0 ) {
            sum += val ;
         }
      }
      if ( norm < sum ) {
         norm = sum ;
      }
   }
} else {
   double   sum, val ;
   double   *row ;
   int      inc1 = mtx->inc1, inc2 = mtx->inc2, irow, jcol, kk ;

   for ( irow = 0, row = mtx->entries ; 
         irow < nrow ; 
         irow++, row += inc1 ) {
      sum = 0.0 ;
      for ( jcol = 0, kk = 0 ; jcol < ncol ; jcol++, kk += inc2 ) {
         if ( (val = row[kk]) < 0 ) {
            sum -= val ;
         } else if ( val > 0 ) {
            sum += val ;
         }
      }
      if ( norm < sum ) {
         norm = sum ;
      }
   }
}

return(norm) ; }
   
/*--------------------------------------------------------------------*/
/*
   ----------------------------------
   return the one-norm of column jcol

   created -- 96nov27, cca
   ----------------------------------
*/
double
DA2_oneNormOfColumn (
   DA2   *mtx,
   int   jcol 
) {
double   sum, val ;
double   *col ;
int      inc1, irow, kk, nrow ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL 
   || jcol < 0 || jcol > mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_oneNormOfColumn(%p,%d)"
           "\n bad input\n", mtx, jcol) ;
   exit(-1) ;
}
col  = mtx->entries + jcol*mtx->inc2 ;
nrow = mtx->n1 ;
sum  = 0.0 ;
if ( (inc1 = mtx->inc1) == 1 ) {
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      if ( (val = col[irow]) < 0.0 ) {
         sum -= val ;
      } else if ( val > 0.0 ) {
         sum += val ;
      }
   }
} else {
   for ( irow = 0, kk = 0 ; irow < nrow ; irow++, kk += inc1 ) {
      if ( (val = col[kk]) < 0.0 ) {
         sum -= val ;
      } else if ( val > 0.0 ) {
         sum += val ;
      }
   }
}
return(sum) ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------
   return the two-norm of column jcol

   created -- 96nov27, cca
   ----------------------------------
*/
double
DA2_twoNormOfColumn (
   DA2   *mtx,
   int   jcol 
) {
double   sum ;
double   *col ;
int      inc1, irow, kk, nrow ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL 
   || jcol < 0 || jcol > mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_twoNormOfColumn(%p,%d)"
           "\n bad input\n", mtx, jcol) ;
   exit(-1) ;
}
col  = mtx->entries + jcol*mtx->inc2 ;
nrow = mtx->n1 ;
sum  = 0.0 ;
if ( (inc1 = mtx->inc1) == 1 ) {
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      sum += col[irow] * col[irow] ;
   }
} else {
   for ( irow = 0, kk = 0 ; irow < nrow ; irow++, kk += inc1 ) {
      sum += col[kk] * col[kk] ;
   }
}
sum = sqrt(sum) ;

return(sum) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------
   return the infinity-norm of column jcol

   created -- 96nov27, cca
   ---------------------------------------
*/
double
DA2_infinityNormOfColumn (
   DA2   *mtx,
   int   jcol 
) {
double   norm, val ;
double   *col ;
int      inc1, irow, kk, nrow ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL 
   || jcol < 0 || jcol > mtx->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_infinityNormOfColumn(%p,%d)"
           "\n bad input\n", mtx, jcol) ;
   exit(-1) ;
}
col  = mtx->entries + jcol*mtx->inc2 ;
nrow = mtx->n1 ;
norm = 0.0 ;
if ( (inc1 = mtx->inc1) == 1 ) {
   for ( irow = 0 ; irow < nrow ; irow++ ) {
      if ( norm < (val = fabs(col[irow])) ) {
         norm = val ;
      }
   }
} else {
   for ( irow = 0, kk = 0 ; irow < nrow ; irow++, kk += inc1 ) {
      if ( norm < (val = fabs(col[kk])) ) {
         norm = val ;
      }
   }
}
return(norm) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------
   return the one-norm of row irow

   created -- 96nov27, cca
   -------------------------------
*/
double
DA2_oneNormOfRow (
   DA2   *mtx,
   int   irow 
) {
double   sum, val ;
double   *row ;
int      inc2, jcol, kk, ncol ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL 
   || irow < 0 || irow > mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_oneNormOfRow(%p,%d)"
           "\n bad input\n", mtx, irow) ;
   exit(-1) ;
}
row  = mtx->entries + irow*mtx->inc1 ;
ncol = mtx->n2 ;
sum  = 0.0 ;
if ( (inc2 = mtx->inc2) == 1 ) {
   for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
      if ( (val = row[jcol]) < 0.0 ) {
         sum -= val ;
      } else if ( val > 0.0 ) {
         sum += val ;
      }
   }
} else {
   for ( jcol = 0, kk = 0 ; jcol < ncol ; jcol++, kk += inc2 ) {
      if ( (val = row[kk]) < 0.0 ) {
         sum -= val ;
      } else if ( val > 0.0 ) {
         sum += val ;
      }
   }
}
return(sum) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------
   return the two-norm of row irow

   created -- 96nov27, cca
   -------------------------------
*/
double
DA2_twoNormOfRow (
   DA2   *mtx,
   int   irow 
) {
double   sum ;
double   *row ;
int      inc2, jcol, kk, ncol ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL 
   || irow < 0 || irow > mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_twoNormOfRow(%p,%d)"
           "\n bad input\n", mtx, irow) ;
   exit(-1) ;
}
row  = mtx->entries + irow*mtx->inc1 ;
ncol = mtx->n2 ;
sum  = 0.0 ;
if ( (inc2 = mtx->inc2) == 1 ) {
   for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
      sum += row[jcol] * row[jcol] ;
   }
} else {
   for ( jcol = 0, kk = 0 ; jcol < ncol ; jcol++, kk += inc2 ) {
      sum += row[kk] * row[kk] ;
   }
}
sum = sqrt(sum) ;

return(sum) ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------
   return the infinity-norm of row irow

   created -- 96nov27, cca
   ------------------------------------
*/
double
DA2_infinityNormOfRow (
   DA2   *mtx,
   int   irow 
) {
double   norm, val ;
double   *row ;
int      inc2, jcol, kk, ncol ;
/*
   ---------------
   check the input
   ---------------
*/
if (  mtx == NULL || mtx->entries == NULL 
   || irow < 0 || irow > mtx->n1 ) {
   fprintf(stderr, "\n fatal error in DA2_infinityNormOfRow(%p,%d)"
           "\n bad input\n", mtx, irow) ;
   exit(-1) ;
}
row  = mtx->entries + irow*mtx->inc1 ;
ncol = mtx->n2 ;
norm = 0.0 ;
if ( (inc2 = mtx->inc2) == 1 ) {
   for ( jcol = 0 ; jcol < ncol ; jcol++ ) {
      if ( norm < (val = fabs(row[jcol])) ) {
         norm = val ;
      }
   }
} else {
   for ( jcol = 0, kk = 0 ; jcol < ncol ; jcol++, kk += inc2 ) {
      if ( norm < (val = fabs(row[kk])) ) {
         norm = val ;
      }
   }
}
return(norm) ; }

/*--------------------------------------------------------------------*/
