/*  inverse.c  */

#include "../DA2.h"

#define MYDEBUG 0

/*--------------------------------------------------------------------*/
/*
   ---------------------------------
   on input,  
      A is the matrix to be inverted
      B contains the identity matrix
   on output, 
      A contains the identity matrix
      B contains A^{-1}

   note: presently, 
      A and B must be column major

   return value -- 
      1 if A is nonsingular
      0 if A is singular

   created -- 96jun29, cca
   ---------------------------------
*/
int
DA2_invert (
   DA2   *A,
   DA2   *B
) {
int      icol, ii, inc2A, inc2B, jcol, n ;
double   beta, fac, s, sum, twonorm, val ;
double   *Aent, *Bent, *colAi, *colAj, *colBj ;
/*
   ---------------
   check the input
   ---------------
*/
if ( A == NULL || B == NULL 
   || (n = A->n1) <= 0 || n != A->n2 
   || A->inc1 != 1
   || (inc2A = A->inc2) <= 0 
   || (Aent = A->entries) == NULL
   || n != B->n1 || n != B->n2
   || B->inc1 != 1
   || (inc2B = B->inc2) <= 0 
   || (Bent = B->entries) == NULL 
) {
   fprintf(stderr, "\n fatal error in DA2_invert(%p,%p)"
           "\n bad input", A, B) ;
   if ( A->inc1 != 1 ) {
      fprintf(stderr, "\n sorry, A->inc1 = %d, must be 1", A->inc1) ;
   }
   if ( B->inc1 != 1 ) {
      fprintf(stderr, "\n sorry, B->inc1 = %d, must be 1", B->inc1) ;
   }
   if ( A != NULL ) {
      fprintf(stderr, "\n A") ;
      DA2_writeForHumanEye(A, stderr) ;
   }
   if ( B != NULL ) {
      fprintf(stderr, "\n B") ;
      DA2_writeForHumanEye(B, stderr) ;
   }
   exit(-1) ;
}
/*
   --------------------------------------------------
   only column major supported
      A = QR --- overwrite A with R and B with Q^T
   --------------------------------------------------
*/
colAi = Aent ;
for ( icol = 0 ; icol < n - 1 ; icol++ ) {
/*
   ----------------------------------------------
   compute the householder vector for this column
   ----------------------------------------------
*/
   twonorm = 0.0 ;
   for ( ii = icol ; ii < n ; ii++ ) {
      val = colAi[ii] ;
      twonorm += val*val ;
   }
   twonorm = sqrt(twonorm) ;
   if ( twonorm == 0.0 ) {
      return(0) ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n column %d, twonorm = %12.4e", icol, twonorm) ;
#endif
   if ( colAi[icol] < 0 ) {
      colAi[icol] -= twonorm ;
   } else {
      colAi[icol] += twonorm ;
   }
   sum = 0.0 ;
   for ( ii = icol ; ii < n ; ii++ ) {
      val = colAi[ii] ;
      sum += val*val ;
   }
   beta = 2./sum ;
#if MYDEBUG > 0
   fprintf(stdout, "\n column %d, beta = %12.4e", icol, beta) ;
#endif
/*
   ----------------------------
   update the following columns
   ----------------------------
*/
   colAj = colAi + inc2A ;
   for ( jcol = icol + 1 ; jcol < n ; jcol++ ) {
      s = 0.0 ;
      for ( ii = icol ; ii < n ; ii++ ) {
         s += colAi[ii] * colAj[ii] ;
      }
      s *= beta ;
      for ( ii = icol ; ii < n ; ii++ ) {
         colAj[ii] -= s * colAi[ii] ;
      }
      colAj += inc2A ;
   }
#if MYDEBUG > 0
   fprintf(stdout, "\n A : after step %d", icol) ;
   DA2_writeForHumanEye(A, stdout) ;
#endif
   colBj = Bent ;
   for ( jcol = 0 ; jcol < n ; jcol++ ) {
      s = 0.0 ;
      for ( ii = icol ; ii < n ; ii++ ) {
         s += colAi[ii] * colBj[ii] ;
      }
      s *= beta ;
      for ( ii = icol ; ii < n ; ii++ ) {
         colBj[ii] -= s * colAi[ii] ;
      }
      colBj += inc2B ;
   }
/*
   --------------------
   overwrite the column
   --------------------
*/
   if ( colAi[icol] >= 0 ) {
      colAi[icol] = -twonorm ;
   } else {
      colAi[icol] =  twonorm ;
   }
   for ( ii = icol + 1 ; ii < n ; ii++ ) {
      colAi[ii] = 0.0 ;
   }
   colAi += inc2A ;
}
if ( colAi[n-1] == 0.0 ) {
   return(0) ;
}
#if MYDEBUG > 0
fprintf(stdout, "\n A : after first step") ;
DA2_writeForHumanEye(A, stdout) ;
fprintf(stdout, "\n B : after first step") ;
DA2_writeForHumanEye(B, stdout) ;
for ( icol  = 0 ; icol < n ; icol++ ) {
   fprintf(stdout, "\n 2-norm of column %d of Q^T = %12.5e",
           icol, DA2_twoNormOfColumn(B, icol)) ;
}
#endif
/*
   -------------
   now backsolve
   -------------
*/
colAi = Aent + (n-1)*inc2A ;
for ( icol = n - 1 ; icol >= 0 ; icol-- ) {
   colBj = Bent ;
   for ( jcol = 0 ; jcol < n ; jcol++ ) {
      fac = colBj[icol] / colAi[icol] ;
      for ( ii = 0 ; ii < icol ; ii++ ) {
         colBj[ii] -= fac * colAi[ii] ;
      }
      colBj[icol] = fac ;
      colBj += inc2B ;
   }
   for ( ii = 0 ; ii < icol ; ii++ ) {
      colAi[ii] = 0.0 ;
   }
   colAi[icol] = 1.0 ;
   colAi -= inc2A ;
#if MYDEBUG > 0
   fprintf(stdout, "\n after backsolve step %d, B", icol) ;
   DA2_writeForHumanEye(B, stdout) ;
#endif
}
#if MYDEBUG > 0
fprintf(stdout, "\n I matrix") ;
DA2_writeForHumanEye(A, stdout) ;
fprintf(stdout, "\n R^{-1}Q^T matrix") ;
DA2_writeForHumanEye(B, stdout) ;
#endif
return(1) ; }
 
/*--------------------------------------------------------------------*/
