/*  init.c  */

#include "../DA2.h"

/*====================================================================*/
/*
   ------------------------------------------------------------------
   initializer. sets the n1, n2, inc1 and inc2 fields.
   must have
      mtx != NULL
      n1, n2, inc1, inc2 > 0
      (inc1 = 1 and inc2 = nrow) or (inc1 = ncol and inc2 = 1)

   if entries is NULL then
      entries are allocated and zeroed.
   else
      mtx->nowned is set to 0
      mtx->entries is set to entries.
   endif
   n1, n2, inc1, inc2 set

   created -- 96apr19, cca
   ------------------------------------------------------------------
*/
void
DA2_init ( 
   DA2      *mtx, 
   int      n1, 
   int      n2, 
   int      inc1, 
   int      inc2, 
   double   *entries 
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || n1 <= 0 || n2 <= 0 || inc1 <= 0 || inc2 <= 0 ) {
   fprintf(stderr, "\n fatal error in DA2_init(%p,%d,%d,%d,%d,%p)"
           "\n bad input\n", mtx, n1, n2, inc1, inc2, entries) ;
   exit(-1) ;
}
if ( entries == NULL
   && !( (inc1 == 1 && inc2 == n1) || (inc1 == n2 && inc2 == 1) ) ) {
/*
   ---------------------------------------------------------------
   whoa, when we own the data we can only initialize a full matrix
   ---------------------------------------------------------------
*/
   fprintf(stderr, "\n fatal error in DA2_init(%p,%d,%d,%d,%d,%p)"
           "\n entries is not NULL and we have bad increments"
           "\n inc1 = %d, inc2 = %d, nrow = %d, ncol = %d\n", 
           mtx, n1, n2, inc1, inc2, entries, 
           inc1, inc2, n1, n2) ;
   exit(-1) ;
}
mtx->n1   = n1   ;
mtx->n2   = n2   ;
mtx->inc1 = inc1 ;
mtx->inc2 = inc2 ;
if ( entries != NULL ) {
/*
   ----------------------------
   set pointer to these entries
   ----------------------------
*/
   mtx->nowned  = 0 ;
   mtx->entries = entries ;
} else {
/*
   ----------------------------
   allocate and own the entries
   ----------------------------
*/
   mtx->nowned  = n1 * n2 ;
   mtx->entries = DVinit(mtx->nowned, 0.0) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------
   submatrix initializer 

   A(0:lastrow-firstrow,0:lastcol-firstcol) 
              = B(firstrow:lastrow, firstcol:lastcol)

   created -- 96aug18, cca
   --------------------------------------------------
*/
void
DA2_subDA2 (
   DA2   *mtxA,
   DA2   *mtxB,
   int   firstrow,
   int   lastrow,
   int   firstcol,
   int   lastcol
) {
/*
   ---------------
   check the input
   ---------------
*/
if (  mtxA == NULL || mtxB == NULL 
   || firstrow < 0 || lastrow >= mtxB->n1 
   || firstcol < 0 || lastcol >= mtxB->n2 ) {
   fprintf(stderr, "\n fatal error in DA2_subDA2(%p,%p,%d,%d,%d,%d)"
           "\n bad input\n", 
           mtxA, mtxB, firstrow, lastrow, firstcol, lastcol) ;
   if ( mtxA != NULL ) {
      fprintf(stderr, "\n first DA2") ;
      DA2_writeForHumanEye(mtxA, stderr) ;
   }
   if ( mtxB != NULL ) {
      fprintf(stderr, "\n second DA2") ;
      DA2_writeForHumanEye(mtxB, stderr) ;
   }
   exit(-1) ;
}
mtxA->inc1    = mtxB->inc1 ;
mtxA->inc2    = mtxB->inc2 ;
mtxA->n1      = lastrow - firstrow + 1 ;
mtxA->n2      = lastcol - firstcol + 1 ;
mtxA->entries = mtxB->entries 
              + firstrow*mtxB->inc1 + firstcol*mtxB->inc2 ;
mtxA->nowned  = 0 ;

return ; }

/*--------------------------------------------------------------------*/
/*
   ------------------------------------------------
   resize the object

   if the object does not own its data then
      if entries are present then
         if not enough entries are present
            write error message and exit
         endif 
      else 
         allocate and own new entries
      endif
   else if not enough entries present then
      if old entries are present then
         free old entries
      endif
      allocate and own new entries
   endif
   set dimensions and increments

   created  -- 96nov16, cca
   ------------------------------------------------
*/
void
DA2_resize (
   DA2   *mtx,
   int   nrow,
   int   ncol,
   int   inc1,
   int   inc2
) {
int   nent ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || nrow <= 0 || ncol <= 0 || inc1 <= 0 || inc2 <= 0 ) {
   fprintf(stderr, "\n fatal error in DA2_resize(%p,%d,%d,%d,%d)"
           "\n bad input\n", mtx, nrow, ncol, inc1, inc2) ;
   exit(-1) ;
}
if ( !( (inc1 == 1 && inc2 == nrow) || (inc1 == ncol && inc2 == 1) ) ) {
/*
   -----------------------------------------
   whoa, we can only resize to a full matrix
   -----------------------------------------
*/
   fprintf(stderr, "\n fatal error in DA2_resize(%p,%d,%d,%d,%d)"
           "\n bad increments"
           "\n inc1 = %d, inc2 = %d, nrow = %d, ncol = %d\n", 
           mtx, nrow, ncol, inc1, inc2, inc1, inc2, nrow, ncol) ;
   exit(-1) ;
}
nent = nrow * ncol ;
if ( mtx->nowned == 0 ) {
   if ( mtx->n1 * mtx->n2 > 0 ) {
/*
      ---------------------------------------
      entries are not owned by this object,
      size can only remain the same or shrink
      ---------------------------------------
*/
      if ( nent > mtx->n1 * mtx->n2 ) {
         fprintf(stderr, "\n fatal error in DA2_resize(%p,%d,%d,%d,%d)"
                 "\n a->nowned = %d, mtx->n1*mtx->n2 = %d"
                 "\n  nrow = %d, ncol = %d\n",
                  mtx, nrow, ncol, inc1, inc2, mtx->nowned, 
                  mtx->n1*mtx->n2, nrow, ncol) ;
         exit(-1) ;
      }
   } else {
/*
      --------------------------------
      no entries at all, allocate more
      --------------------------------
*/
      mtx->entries = DVinit(nent, 0.0) ;
      mtx->nowned  = nent ;
   }
} else if ( mtx->nowned < nent ) {
/*
   ---------------------------------------------
   object owns its entries but must grow in size
   ---------------------------------------------
*/
   if ( mtx->entries != NULL ) {
      DVfree(mtx->entries) ;
   }
   mtx->entries = DVinit(nent, 0.0) ;
   mtx->nowned  = nent ;
}
/*
   --------------------------
   set dimensions and strides
   --------------------------
*/
mtx->n1   = nrow ;
mtx->n2   = ncol ;
mtx->inc1 = inc1 ;
mtx->inc2 = inc2 ;

return ; }
   
/*--------------------------------------------------------------------*/
