/*  gather.c  */

#include "../DA2.h"

#define CAUTIOUS 1

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   gather entries
      B = A(rowids, colids)
   rowids -- ids of rows to be gathered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be gathered,
      must be not NULL if B->n2 != A->n2

   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_gather (
   DA2   *B,
   DA2   *A,
   int   *rowids,
   int   *colids
) {
int      ierr, ii, inc1A, inc1B, inc2A, inc2B, irow, jj, jcol, 
         n1A, n1B, n2A, n2B, rc ;
double   *entA, *entB ;
/*
   ---------------
   check the input
   ---------------
*/
if (  B == NULL || A == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_gather(%p,%p,%p,%p)"
           "\n B and/or A is NULL\n", B, A, rowids, colids) ;
   exit(-1) ;
}
n1B   = B->n1      ;
n2B   = B->n2      ;
inc1B = B->inc1    ;
inc2B = B->inc2    ;
entB  = B->entries ;
n1A   = A->n1      ;
n2A   = A->n2      ;
inc1A = A->inc1    ;
inc2A = A->inc2    ;
entA  = A->entries ;
#if CAUTIOUS
if ( n1B <= 0 || n2B <= 0 || inc1B <= 0 || inc2B <= 0 || entB == NULL ){
   fprintf(stderr, "\n fatal error in DA2_gather(%p,%p,%p,%p)"
           "\n B is invalid\n", B, A, rowids, colids) ;
   DA2_writeStats(B, stderr) ;
   exit(-1) ;
}
if ( n1A <= 0 || n2A <= 0 || inc1A <= 0 || inc2A <= 0 || entA == NULL ){
   fprintf(stderr, "\n fatal error in DA2_gather(%p,%p,%p,%p)"
           "\n A is invalid\n", B, A, rowids, colids) ;
   DA2_writeStats(A, stderr) ;
   exit(-1) ;
}
if ( n1B != n1A ) {
/*
   ---------------------------
   check out the rowids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n1B > n1A || rowids == NULL ) {
      rc = 1 ;
   } else {
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         if ( (irow = rowids[ii]) < 0 || irow >= n1A ) {
            rc = 1 ;
            break ;
         }
      }
   }
   if ( rc == 1 ) {
      fprintf(stderr, "\n fatal error in DA2_gather(%p,%p,%p,%p)"
              "\n bad rowids vector\n", B, A, rowids, colids) ;
      fprintf(stderr, "\n\n A") ;
      DA2_writeStats(A, stderr) ;
      fprintf(stderr, "\n\n B") ;
      DA2_writeStats(B, stderr) ;
      if ( n1B > 0 && rowids != NULL ) {
         fprintf(stderr, "\n rowids") ;
         IVfp80(stderr, n1B, rowids, 9, &ierr) ;
      }
      exit(-1) ;
   }
}
if ( n2B != n2A ) {
/*
   ---------------------------
   check out the colids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n2B > n2A || colids == NULL ) {
      rc = 1 ;
   } else {
      for ( jj = 0 ; jj < n2B ; jj++ ) {
         if ( (jcol = colids[jj]) < 0 || jcol >= n2A ) {
            rc = 1 ;
            break ;
         }
      }
   }
   if ( rc == 1 ) {
      fprintf(stderr, "\n fatal error in DA2_gather(%p,%p,%p,%p)"
              "\n bad colids vector\n", B, A, rowids, colids) ;
      if ( n2B > 0 && colids != NULL ) {
         fprintf(stderr, "\n colids") ;
         IVfp80(stderr, n2B, colids, 9, &ierr) ;
      }
      exit(-1) ;
   }
}
#endif
/*
   --------------------------
   switch over the four cases
   --------------------------
*/
if ( n1B != n1A && n2B != n2A ) {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         irow = rowids[ii] ;
         entB[ii*inc1B + jj*inc2B] = entA[irow*inc1A + jcol*inc2A] ;
      }
   }
} else if ( n1B != n1A ) {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         irow = rowids[ii] ;
         entB[ii*inc1B + jj*inc2B] = entA[irow*inc1A + jj*inc2A] ;
      }
   }
} else if ( n2B != n2A ) {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         entB[ii*inc1B + jj*inc2B] = entA[ii*inc1A + jcol*inc2A] ;
      }
   }
} else {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         entB[ii*inc1B + jj*inc2B] = entA[ii*inc1A + jj*inc2A] ;
      }
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   gather entries from A into B 
      B = A(rowids, colids)
   and zero entries in A
   rowids -- ids of rows to be gathered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be gathered,
      must be not NULL if B->n2 != A->n2

   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_gatherZero (
   DA2   *B,
   DA2   *A,
   int   *rowids,
   int   *colids
) {
int      ierr, ii, inc1A, inc1B, inc2A, inc2B, irow, jj, jcol, 
         n1A, n1B, n2A, n2B, rc ;
double   *entA, *entB ;
/*
   ---------------
   check the input
   ---------------
*/
if (  B == NULL || A == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_gatherZero(%p,%p,%p,%p)"
           "\n B and/or A is NULL\n", B, A, rowids, colids) ;
   exit(-1) ;
}
n1B   = B->n1      ;
n2B   = B->n2      ;
inc1B = B->inc1    ;
inc2B = B->inc2    ;
entB  = B->entries ;
n1A   = A->n1      ;
n2A   = A->n2      ;
inc1A = A->inc1    ;
inc2A = A->inc2    ;
entA  = A->entries ;
#if CAUTIOUS
if ( n1B <= 0 || n2B <= 0 || inc1B <= 0 || inc2B <= 0 || entB == NULL ){
   fprintf(stderr, "\n fatal error in DA2_gatherZero(%p,%p,%p,%p)"
           "\n B is invalid\n", B, A, rowids, colids) ;
   DA2_writeStats(B, stderr) ;
   exit(-1) ;
}
if ( n1A <= 0 || n2A <= 0 || inc1A <= 0 || inc2A <= 0 || entA == NULL ){
   fprintf(stderr, "\n fatal error in DA2_gatherZero(%p,%p,%p,%p)"
           "\n A is invalid\n", B, A, rowids, colids) ;
   DA2_writeStats(A, stderr) ;
   exit(-1) ;
}
if ( n1B != n1A ) {
/*
   ---------------------------
   check out the rowids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n1B > n1A || rowids == NULL ) {
      rc = 1 ;
   } else {
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         if ( (irow = rowids[ii]) < 0 || irow >= n1A ) {
            rc = 1 ;
            break ;
         }
      }
   }
   if ( rc == 1 ) {
      fprintf(stderr, "\n fatal error in DA2_gatherZero(%p,%p,%p,%p)"
              "\n bad rowids vector\n", B, A, rowids, colids) ;
      fprintf(stderr, "\n\n A") ;
      DA2_writeStats(A, stderr) ;
      fprintf(stderr, "\n\n B") ;
      DA2_writeStats(B, stderr) ;
      if ( n1B > 0 && rowids != NULL ) {
         fprintf(stderr, "\n rowids") ;
         IVfp80(stderr, n1B, rowids, 9, &ierr) ;
      }
      exit(-1) ;
   }
}
if ( n2B != n2A ) {
/*
   ---------------------------
   check out the colids vector
   ---------------------------
*/
   rc = 0 ;
   if ( n2B > n2A || colids == NULL ) {
      rc = 1 ;
   } else {
      for ( jj = 0 ; jj < n2B ; jj++ ) {
         if ( (jcol = colids[jj]) < 0 || jcol >= n2A ) {
            rc = 1 ;
            break ;
         }
      }
   }
   if ( rc == 1 ) {
      fprintf(stderr, "\n fatal error in DA2_gatherZero(%p,%p,%p,%p)"
              "\n bad colids vector\n", B, A, rowids, colids) ;
      if ( n2B > 0 && colids != NULL ) {
         fprintf(stderr, "\n colids") ;
         IVfp80(stderr, n2B, colids, 9, &ierr) ;
      }
      exit(-1) ;
   }
}
#endif
/*
   --------------------------
   switch over the four cases
   --------------------------
*/
if ( n1B != n1A && n2B != n2A ) {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         irow = rowids[ii] ;
         entB[ii*inc1B + jj*inc2B] = entA[irow*inc1A + jcol*inc2A] ;
         entA[irow*inc1A + jcol*inc2A] = 0 ;
      }
   }
} else if ( n1B != n1A ) {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         irow = rowids[ii] ;
         entB[ii*inc1B + jj*inc2B] = entA[irow*inc1A + jj*inc2A] ;
         entA[irow*inc1A + jj*inc2A] = 0 ;
      }
   }
} else if ( n2B != n2A ) {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      jcol = colids[jj] ;
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         entB[ii*inc1B + jj*inc2B] = entA[ii*inc1A + jcol*inc2A] ;
         entA[ii*inc1A + jcol*inc2A] = 0 ;
      }
   }
} else {
   for ( jj = 0 ; jj < n2B ; jj++ ) {
      for ( ii = 0 ; ii < n1B ; ii++ ) {
         entB[ii*inc1B + jj*inc2B] = entA[ii*inc1A + jj*inc2A] ;
         entA[ii*inc1A + jj*inc2A] = 0 ;
      }
   }
}
return ; }

/*--------------------------------------------------------------------*/
