/*  QRfactor.c  */

#include "../DA2.h"

#define MYDEBUG 1

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------------
   purpose ---
      given a matrix A, convert to an upper triangular or
      upper trapezoidal R using householder transformations.
      the transformations are stored in the lower triangle
      or lower trapezoid of A, their coefficients are stored
      in the betaDV object

      this code follows closely page 210 from golub and van loan,
      "matrix computations", 3rd edition.
 
      note: presently the matrix must be column major

   created -- 97mar29, cca
   --------------------------------------------------------------
*/
void
DA2_QRfactor (
  DA2   *mtx,
  DV    *betaDV
) {
double   beta, fac, mu, sigma, v0 ;
double   *colj, *colk ;
int      irow, jcol, kcol, ncol, nrow, nstep ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || betaDV == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_QRfactor(%p)"
           "\n bad input\n", mtx) ;
   exit(-1) ;
}
if (  (nrow = DA2_nrow(mtx)) <= 0
   || (ncol = DA2_ncol(mtx)) <= 0
   || DA2_inc1(mtx) != 1
   || DA2_entries(mtx) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_QRfactor(%p)"
           "\n bad object\n", mtx) ;
   DA2_writeStats(mtx, stderr) ;
   exit(-1) ;
}
/*
   ----------------------------------------------
   determine the number of steps = min(nrow,ncol)
   and resize the coefficient DV object
   ----------------------------------------------
*/
nstep = (ncol < nrow) ? ncol : nrow - 1 ;
DV_setSize(betaDV, nstep) ;
/*
   -------------------
   loop over the steps
   -------------------
*/
for ( jcol = 0 ; jcol < nstep ; jcol++ ) {
/*
   -----------------------------------------
   compute the pointer to the present column
   -----------------------------------------
*/
   colj  = DA2_column(mtx, jcol) ;
/*
   -----------------------------------
   compute ||colj(jcol+1:nrow-1)||_2^2
   -----------------------------------
*/
   sigma = 0.0 ;
   for ( irow = jcol + 1 ; irow < nrow ; irow++ ) {
      sigma += colj[irow]*colj[irow] ;
   }
   if ( sigma != 0.0 ) {
/*
      --------------------------------------------
      there are zero entries below the diagonal
      compute mu = compute ||colj(jcol:nrow-1)||_2
      --------------------------------------------
*/
      mu = sqrt(sigma + colj[jcol]*colj[jcol]) ;
/*
      -----------------------------------------------
      set the first element of the householder vector
      -----------------------------------------------
*/
      if ( colj[jcol] <= 0 ) {
         v0 = colj[jcol] - mu ;
      } else {
         v0 = -sigma/(colj[jcol] + mu) ;
      }
/*
      -------------------
      get the coefficient
      -------------------
*/
      beta = 2*v0*v0/(sigma + v0*v0) ;
/*
      ---------------------------------------------
      set the first entry of the transformed column
      ---------------------------------------------
*/
      colj[jcol] = mu ;
/*
      -----------------------------------------------------------
      scale the householder vector so that its first entry is 1.0
      -----------------------------------------------------------
*/
      for ( irow = jcol + 1 ; irow < nrow ; irow++ ) {
         colj[irow] = colj[irow] / v0 ;
      }
/*
      ------------------------------------------------------------
      loop over the following columns and apply the transformation
      ------------------------------------------------------------
*/
      for ( kcol = jcol + 1 ; kcol < ncol ; kcol++ ) {
         colk = DA2_column(mtx, kcol) ;
/*
         ---------------------------------------------
         compute the dot product between colj and colk
         ---------------------------------------------
*/
         for ( irow = jcol + 1, fac = colk[jcol] ; 
               irow < nrow ; 
               irow++ ) {
            fac += colk[irow]*colj[irow] ;
         }
/*
         --------------------------
         compute the scaling factor
         --------------------------
*/
         fac = -beta*fac ;
/*
         ---------------------------------------
         apply the transformation to column colk
         ---------------------------------------
*/
         colk[jcol] += fac ;
         for ( irow = jcol + 1 ; irow < nrow ; irow++ ) {
            colk[irow] += fac * colj[irow] ;
         }
      }
   } else {
      beta = 0.0 ;
   }
/*
   -------------------------
   set the coefficient entry
   -------------------------
*/
   DV_setEntry(betaDV, jcol, beta) ;
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------------
   apply a sequence of householder transformations to a matrix B

   B = \prod_{i=1}^{nstep} (I - beta*h*h^T) B

   the householder vectors are stored in in the lower triangle of H,
   the first is size nrow(H), the second has size nrow(H)-1, etc.
   the first nonzero component of each vector is 1.
   the beta coefficients are found in betaDV.

   created -- 97mar29, cca
   -----------------------------------------------------------------
*/
void
DA2_applyHouseholders (
   DA2   *H,
   DV    *betaDV,
   DA2   *B
) {
double   beta, fac ;
double   *colk, *v ;
int      irow, istep, kcol, ncol, nrow, nstep ;
/*
   ---------------
   check the input
   ---------------
*/
if ( H == NULL || betaDV == NULL || B == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_applyHouseholders(%p,%p,%p)"
           "\n bad input\n", H, betaDV, B) ;
   exit(-1) ;
}
if (  (nrow  = DA2_nrow(H)) <= 0
   || (nstep = DA2_ncol(H)) <= 0
   || DA2_inc1(H) != 1
   || DA2_entries(H) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_applyHouseholders(%p,%p,%p)"
           "\n bad object H\n", H, betaDV, B) ;
   DA2_writeStats(H, stderr) ;
   exit(-1) ;
}
if (  nstep != DV_size(betaDV)
   || DV_entries(betaDV) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_applyHouseholders(%p,%p,%p)"
           "\n bad object betaDV\n", H, betaDV, B) ;
   DV_writeStats(betaDV, stderr) ;
   exit(-1) ;
}
if (  nrow != DA2_nrow(B)
   || (ncol = DA2_ncol(B)) <= 0
   || DA2_inc1(B) != 1
   || DA2_entries(B) == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_applyHouseholders(%p,%p,%p)"
           "\n bad object B \n", H, betaDV, B) ;
   DA2_writeStats(B, stderr) ;
   exit(-1) ;
}
/*
   ---------------------------------
   loop over the householder vectors
   ---------------------------------
*/
for ( istep = 0 ; istep < nstep ; istep++ ) {
   v    = DA2_column(H, istep) ;
   beta = DV_entry(betaDV, istep) ;
   for ( kcol = 0 ; kcol < ncol ; kcol++ ) {
      colk = DA2_column(B, kcol) ;
      for ( irow = istep + 1, fac = colk[istep] ;
            irow < nrow ; 
            irow++ ) {
         fac += colk[irow] * v[irow] ;
      }
      fac = -beta * fac ;
      colk[istep] += fac ;
      for ( irow = istep + 1 ; irow < nrow ; irow++ ) {
         colk[irow] += fac * v[irow] ;
      }
   }
}
return ; }

/*--------------------------------------------------------------------*/
