/*  IO.c  */

#include "../DA2.h"

static const char *suffixb = ".da2b" ;
static const char *suffixf = ".da2f" ;

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------------
   purpose -- to read in the object from a file

   input --

      fn -- filename, must be *.da2b or *.da2f

   return value -- 1 if success, 0 if failure
   -----------------------------------------------------------
*/
int
DA2_readFromFile ( 
   DA2    *mtx, 
   char   *fn 
) {
FILE   *fp ;
int    fnlength, rc = 0, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || fn == NULL ) {
   fprintf(stderr, 
    "\n error in DA2_readFromFile(%p,%s), file %s, line %d"
    "\n bad input", mtx, fn, __FILE__, __LINE__) ;
   return(0) ;
}
/*
   -------------
   read the file
   -------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "rb")) == NULL ) {
         fprintf(stderr, 
                 "\n error in DA2_readFromFile(%s), file %s, line %d"
                 "\n unable to open file %s", 
                 fn, __FILE__, __LINE__, fn) ;
      } else {
         rc = DA2_readFromBinaryFile(mtx, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "r")) == NULL ) {
         fprintf(stderr, 
                 "\n error in DA2_readFromFile(%s), file %s, line %d"
                 "\n unable to open file %s", 
                 fn, __FILE__, __LINE__, fn) ;
      } else {
         rc = DA2_readFromFormattedFile(mtx, fp) ;
         fclose(fp) ;
      }
   } else {
      fprintf(stderr, 
              "\n error in DA2_readFromFile(%s), file %s, line %d"
              "\n bad DA2 file name %s,"
              "\n must end in %s (binary) or %s (formatted)\n",
              fn, __FILE__, __LINE__, fn, suffixb, suffixf) ;
      rc = 0 ;
   }
} else {
   fprintf(stderr, 
       "\n error in DA2_readFromFile(%s), file %s, line %d"
       "\n bad DA2 file name %s,"
       "\n must end in %s (binary) or %s (formatted)\n",
       fn, __FILE__, __LINE__, fn, suffixb, suffixf) ;
   rc = 0 ;
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------
   purpose -- to read an object from a formatted file

   return value -- 1 if success, 0 if failure
   --------------------------------------------------
*/
int
DA2_readFromFormattedFile ( 
   DA2    *mtx, 
   FILE   *fp 
) {
int   rc, size ;
int   itemp[4] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || fp == NULL ) {
   fprintf(stderr, 
  "\n error in DA2_readFromFormattedFile(%p,%p), file %s, line %d"
  "\n bad input", mtx, fp, __FILE__, __LINE__) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DA2_clearData(mtx) ;
/*
   ------------------------------------------------------
   read in the four scalar parameters: n1, n2, inc1, inc2
   ------------------------------------------------------
*/
if ( (rc = IVfscanf(fp, 4, itemp)) != 4 ) {
   fprintf(stderr, 
           "\n error in DA2_readFromFormattedFile(), file %s, line %d"
           "\n %d items of %d read\n",
           __FILE__, __LINE__, rc, 4) ;
   return(0) ;
}
/*
   ---------------------
   initialize the object
   ---------------------
*/
DA2_init(mtx, itemp[0], itemp[1], itemp[2], itemp[3], NULL) ;
/*
   ----------------------------
   read in the entries[] vector
   ----------------------------
*/
if ( (size = 1 + (mtx->n1-1)*mtx->inc1 + (mtx->n2-1)*mtx->inc2) > 0 ) {
   if ( (rc = DVfscanf(fp, size, mtx->entries)) != size ) {
      fprintf(stderr, 
              "\n error in DA2_readFromFormattedFile, file %s, line %d"
              "\n %d items of %d read\n", 
              __FILE__, __LINE__, rc, size) ;
      return(0) ;
   }
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------
   purpose -- to read an object from a binary file

   return value -- 1 if success, 0  if failure
   -----------------------------------------------
*/
int
DA2_readFromBinaryFile ( 
   DA2    *mtx, 
   FILE   *fp 
) {
int   rc, size ;
int   itemp[4] ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || fp == NULL ) {
   fprintf(stderr, 
    "\n fatal error in DA2_readFromBinaryFile(%p,%p), file %s, line %d"
    "\n bad input", mtx, fp, __FILE__, __LINE__) ;
   return(0) ;
}
/*
   ---------------------
   clear the data fields
   ---------------------
*/
DA2_clearData(mtx) ;
/*
   ------------------------------------------------------
   read in the four scalar parameters: n1, n2, inc1, inc2
   ------------------------------------------------------
*/
if ( (rc = fread((char *) itemp, sizeof(int), 4, fp)) != 4 ) {
   fprintf(stderr, 
           "\n error in DA2_readFromBinaryFile, file %s, line %d"
           "\n %d items of %d read\n", 
           __FILE__, __LINE__, rc, 4) ;
   return(0) ;
}
fprintf(stdout, "\n itemp = {%d, %d, %d, %d}", 
        itemp[0], itemp[1], itemp[2], itemp[3]) ;
fflush(stdout) ;
/*
   ---------------------
   initialize the object
   ---------------------
*/
DA2_init(mtx, itemp[0], itemp[1], itemp[2], itemp[3], NULL) ;
/*
   ----------------------------
   read in the entries[] vector
   ----------------------------
*/
if ( (size = 1 + (mtx->n1-1)*mtx->inc1 + (mtx->n2-1)*mtx->inc2) > 0 ) {
   if ( (rc = fread(mtx->entries, sizeof(double), size, fp)) != size ) {
      fprintf(stderr, 
              "\n error in DA2_readFromBinaryFile, file %s, line %d"
              "\n %d items of %d read\n", 
              __FILE__, __LINE__, rc, size) ;
      return(0) ;
   }
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   purpose -- to write an adjacency graph object to a file

   input --

      fn -- filename
        *.da2b -- binary
        *.da2f -- formatted
        anything else -- for human eye
   ---------------------------------------------------------
*/
void
DA2_writeToFile ( 
   DA2    *mtx, 
   char   *fn 
) {
FILE   *fp ;
int    fnlength, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( mtx == NULL || fn == NULL ) {
   fprintf(stderr, 
    "\n fatal error in DA2_writeToFile(%p,%s), file %s, line %d"
    "\n bad input", mtx, fn, __FILE__, __LINE__) ;
}
/*
   ------------------
   write out the file
   ------------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "wb")) == NULL ) {
         fprintf(stderr, 
                 "\n error in DA2_writeToFile(), file %s, line %d"
                 "\n unable to open file %s", __FILE__, __LINE__, fn) ;
      } else {
         DA2_writeToBinaryFile(mtx, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "w")) == NULL ) {
         fprintf(stderr, 
                 "\n error in DA2_writeToFile(), file %s, line %d"
                 "\n unable to open file %s", __FILE__, __LINE__, fn) ;
      } else {
         DA2_writeToFormattedFile(mtx, fp) ;
         fclose(fp) ;
      }
   } else {
      if ( (fp = fopen(fn, "a")) == NULL ) {
         fprintf(stderr, 
                 "\n error in DA2_writeToFile(), file %s, line %d"
                 "\n unable to open file %s", __FILE__, __LINE__, fn) ;
      } else {
         DA2_writeForHumanEye(mtx, fp) ;
         fclose(fp) ;
      }
   }
} else {
   if ( (fp = fopen(fn, "a")) == NULL ) {
      fprintf(stderr, 
              "\n error in DA2_writeToFile, file %s, line %d"
              "\n unable to open file %s", __FILE__, __LINE__, fn) ;
   } else {
      DA2_writeForHumanEye(mtx, fp) ;
      fclose(fp) ;
   }
}
return ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------
   purpose -- to write an object to a formatted file
   -------------------------------------------------
*/
void
DA2_writeToFormattedFile ( 
   DA2    *mtx, 
   FILE   *fp 
) {
int   size ;

if ( mtx == NULL || fp == NULL ) {
   return ;
}
fprintf(fp, "\n %d %d %d %d", mtx->n1, mtx->n2, mtx->inc1, mtx->inc2) ;
if ( (size = 1 + (mtx->n1-1)*mtx->inc1 + (mtx->n2-1)*mtx->inc2) > 0
     && mtx->entries != NULL ) {
   DVfprintf(fp, size, mtx->entries) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------------
   purpose -- to write an adjacency object to a binary file
   --------------------------------------------------------
*/
void
DA2_writeToBinaryFile ( 
   DA2    *mtx, 
   FILE   *fp 
) {
int   size ;

if ( fp == NULL ) {
   return ;
}
fwrite((void *) &mtx->n1,   sizeof(int), 1, fp) ;
fwrite((void *) &mtx->n2,   sizeof(int), 1, fp) ;
fwrite((void *) &mtx->inc1, sizeof(int), 1, fp) ;
fwrite((void *) &mtx->inc2, sizeof(int), 1, fp) ;
if ( (size = 1 + (mtx->n1-1)*mtx->inc1 + (mtx->n2-1)*mtx->inc2) > 0
     && mtx->entries != NULL ) {
   fwrite((void *) &mtx->entries, sizeof(double), size, fp) ;
}

return ; }

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   purpose -- to write the object for a human eye
   ----------------------------------------------
*/
void
DA2_writeForHumanEye ( 
   DA2    *mtx, 
   FILE   *fp 
) {
int   i, j, jfirst, jlast ;

if ( mtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_writeForHumanEye(%p,%p)"
           "\n bad input\n", mtx, fp) ;
   exit(-1) ;
}
DA2_writeStats(mtx, fp) ;
for ( jfirst = 0 ; jfirst < mtx->n2 ; jfirst += 4 ) {
   jlast = jfirst + 3 ;
   if ( jlast >= mtx->n2 ) {
      jlast = mtx->n2 - 1 ;
   }
   fprintf(fp, "\n    ") ;
   for ( j = jfirst ; j <= jlast ; j++ ) {
      fprintf(fp, "%19d", j) ;
   }
   for ( i = 0 ; i < mtx->n1 ; i++ ) {
      fprintf(fp, "\n%4d", i) ;
      for ( j = jfirst ; j <= jlast ; j++ ) {
         fprintf(fp, "%19.11e", 
                  mtx->entries[i*mtx->inc1 + j*mtx->inc2]) ;
      }
   }
}

return ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------
   purpose -- to write out the statistics
   --------------------------------------
*/
void
DA2_writeStats (
   DA2    *mtx, 
   FILE   *fp 
) {
if ( mtx == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in DA2_writeStats(%p,%p)"
           "\n bad input\n", mtx, fp) ;
   exit(-1) ;
}
fprintf(fp, "\n DA2 : double 2D array object :") ;
fprintf(fp, 
        "\n %d x %d matrix, inc1 = %d, inc2 = %d,"
        "\n nowned = %d, %d entries allocated, occupies %d bytes" 
        "\n entries = %p, maxabs = %20.12e",
        mtx->n1, mtx->n2, mtx->inc1, mtx->inc2, mtx->nowned,
        mtx->n1*mtx->n2,
        DA2_sizeOf(mtx), mtx->entries, DA2_maxabs(mtx)) ;

return ; }

/*--------------------------------------------------------------------*/
