/*  test_ndUpd.c  */

#include "../DA2.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ---------------------------------------
   test the dense update routine DA2_ndUpd

   A -= B * C

   created -- 96nov18, cca
   ---------------------------------------
*/
{
DA2      *a, *b, *c, *d ;
double   ops, t1, t2, value ;
FILE     *msgFile ;
int      ierr, ii, jj, majorA, majorB, majorC,
         msglvl, nrowA, ncolA, ncolB, seed ;

if ( argc != 10 ) {
   fprintf(stdout, 
"\n\n usage : %s msglvl msgFile majorA majorB majorC "
"\n         nrowA ncolA ncolB seed "
"\n    msglvl  -- message level"
"\n    msgFile -- message file"
"\n    majorA  -- 0 for A row major, 1 for A column major"
"\n    majorB  -- 0 for B row major, 1 for B column major"
"\n    majorC  -- 0 for C row major, 1 for C column major"
"\n    nrowA   -- # of rows in A"
"\n    ncolA   -- # of columns in A"
"\n    ncolB   -- # of columns in B"
"\n    seed    -- random number seed"
"\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
majorA = atoi(argv[3]) ;
majorB = atoi(argv[4]) ;
majorC = atoi(argv[5]) ;
nrowA  = atoi(argv[6]) ;
ncolA  = atoi(argv[7]) ;
ncolB  = atoi(argv[8]) ;
if (  nrowA <= 0 || ncolA <= 0 || ncolB <= 0 ) {
   fprintf(stderr, "\n invalid matrix dimensions"
           "\n nrowA = %d, ncolA = %d, ncolB = %d\n",
           nrowA, ncolA, ncolB) ;
   exit(-1) ;
}
if ( majorA < 0 || majorA > 1
  || majorB < 0 || majorB > 1
  || majorC < 0 || majorC > 1 ) {
   fprintf(stderr, "\n invalid major values"
           "\n majorA = %d, majorB = %d, majorC = %d"
           "\n 0 --> row major, 1 --> column major\n",
           majorA, majorB, majorC) ;
   exit(-1) ;
}
seed = atoi(argv[9]) ;
fprintf(msgFile, "\n majorA = %d, majorB = %d, majorC = %d",
        majorA, majorB, majorC) ;
fprintf(msgFile, "\n nrowA = %d, ncolA = %d, ncolB = %d",
        nrowA, ncolA, ncolB) ;
fflush(msgFile) ;
/*
   -----------------------------
   initialize the matrix objects
   -----------------------------
*/
MARKTIME(t1) ;
a = DA2_new() ;
b = DA2_new() ;
c = DA2_new() ;
d = DA2_new() ;
if ( majorA == 0 ) {
   DA2_init(a, nrowA, ncolA, ncolA, 1, NULL) ;
   DA2_init(d, nrowA, ncolA, ncolA, 1, NULL) ;
} else {
   DA2_init(a, nrowA, ncolA, 1, nrowA, NULL) ;
   DA2_init(d, nrowA, ncolA, 1, nrowA, NULL) ;
}
if ( majorB == 0 ) {
   DA2_init(b, nrowA, ncolB, ncolB, 1, NULL) ;
} else {
   DA2_init(b, nrowA, ncolB, 1, nrowA, NULL) ;
}
if ( majorC == 0 ) {
   DA2_init(c, ncolB, ncolA, ncolA, 1, NULL) ;
} else {
   DA2_init(c, ncolB, ncolA, 1, ncolB, NULL) ;
}
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize matrix objects",
        t2 - t1) ;
MARKTIME(t1) ;
DA2_fillRandomUniform(a, 0, 1, seed) ;
DA2_fillRandomUniform(b, 0, 1, seed) ;
DA2_fillRandomUniform(c, 0, 1, seed) ;
DA2_copy(d, a) ;
MARKTIME(t2) ;
fprintf(msgFile, 
        "\n CPU : %.3f to fill matrix with random numbers", t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n matrix A") ;
   DA2_writeForHumanEye(a, msgFile) ;
   fprintf(msgFile, "\n matrix B") ;
   DA2_writeForHumanEye(b, msgFile) ;
   fprintf(msgFile, "\n matrix C") ;
   DA2_writeForHumanEye(c, msgFile) ;
}
/*
   ----------------------------------
   compute the matrix vector multiply
   using the simplest kernel
   ----------------------------------
*/
ops = 2*nrowA*ncolB*ncolA ;
MARKTIME(t1) ;
{
DV   *colDV, *rowDV ;
double   *col, *row ;

rowDV = DV_new() ;
DV_init(rowDV, ncolB, NULL) ;
row = DV_entries(rowDV) ;
colDV = DV_new() ;
DV_init(colDV, ncolB, NULL) ;
col = DV_entries(colDV) ;
for ( jj = 0 ; jj < ncolA ; jj++ ) {
   DA2_extractColumnDV(c, colDV, jj) ;
   for ( ii = 0 ; ii < nrowA ; ii++ ) {
      DA2_extractRowDV(b, rowDV, ii) ;
      value = - DVdot(ncolB, row, col) ;
      DA2_addEntry(d, ii, jj, value) ;
   }
}
}
MARKTIME(t2) ;
fprintf(msgFile, 
"\n CPU : %.3f compute simplest mvm, %.0f ops, %.3f megaflops", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n d") ;
   DA2_writeForHumanEye(d, msgFile) ;
}
/*
   ----------------------------------------
   compute the update with the other kernel
   ----------------------------------------
*/
MARKTIME(t1) ;
DA2_ndUpd(a, b, c) ;
MARKTIME(t2) ;
DA2_sub(d, a) ;
fprintf(msgFile, 
       "\n CPU : %.3f DA2_ndUpd() update, %.3f megaflops, %12.4e error",
        t2 - t1, 1.0e-6*ops/(t2 - t1), DA2_frobNorm(d)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n after the update") ;
   fprintf(msgFile, "\n matrix a") ;
   DA2_writeForHumanEye(a, msgFile) ;
}

fprintf(msgFile, "\n") ;
return ; }

/*--------------------------------------------------------------------*/
