/*  testMVM.c  */

#include "../DA2.h"
#include "../../Drand.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ----------------------------------------------
   test the dense matrix vector multiply routines

   Y = Y + alpha * X

   created -- 96sep13, cca
   ----------------------------------------------
*/
{
DA2      *A, *X, *Y, *Z ;
double   alpha, ops, t1, t2, value ;
double   *col, *row, *x0, *x1, *x2, *z0, *z1, *z2  ;
Drand    *drand ;
DV       *colDV, *rowDV ;
FILE     *msgFile ;
int      ii, inc1, inc2, irhs, jj, msglvl, ncol, nrow, nrhs, seed ;

if ( argc != 10 ) {
   fprintf(stdout, 
           "\n\n usage : %s msglvl msgFile nrow ncol inc1 inc2 nrhs "
           "\n         seed alpha"
           "\n    msglvl  -- message level"
           "\n    msgFile -- message file"
           "\n    nrow    -- # of rows in A"
           "\n    ncol    -- # of columns in A"
           "\n    inc1    -- row stride"
           "\n    inc2    -- column stride"
           "\n    nrhs    -- # of right hand sides"
           "\n    seed    -- random number seed"
           "\n    alpha   -- scalar"
           "\n", argv[0]) ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
nrow  = atoi(argv[3]) ;
ncol  = atoi(argv[4]) ;
inc1  = atoi(argv[5]) ;
inc2  = atoi(argv[6]) ;
nrhs  = atoi(argv[7]) ;
if (  nrow <= 0 || nrow <= 0 || nrhs <= 0 || inc1 < 1 || inc2 < 1 ) {
   fprintf(stderr, "\n invalid input"
      "\n nrow = %d, nrow = %d, nrhs = %d, inc1 = %d, inc2 = %d\n",
           nrow, nrow, nrhs, inc1, inc2) ;
   exit(-1) ;
}
seed   = atoi(argv[8])  ;
alpha  = atof(argv[9])  ;
/*
   --------------------------------------
   initialize the random number generator
   --------------------------------------
*/
drand = Drand_new() ;
Drand_init(drand) ;
Drand_setSeed(drand, seed) ;
Drand_setNormal(drand, 0.0, 1.0) ;
/*
   -----------------------------
   initialize the matrix objects
   -----------------------------
*/
MARKTIME(t1) ;
A = DA2_new() ;
DA2_init(A, nrow, ncol, inc1, inc2, NULL) ;
X = DA2_new() ;
DA2_init(X, ncol, nrhs, 1, ncol, NULL) ;
Y = DA2_new() ;
DA2_init(Y, nrow, nrhs, 1, nrow, NULL) ;
Z = DA2_new() ;
DA2_init(Z, nrow, nrhs, 1, nrow, NULL) ;
rowDV = DV_new() ;
DV_init(rowDV, ncol, NULL) ;
row = DV_entries(rowDV) ;
colDV = DV_new() ;
DV_init(colDV, ncol, NULL) ;
col = DV_entries(colDV) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize matrix objects",
        t2 - t1) ;
fflush(msgFile) ;
/*
   ---------------------------------------
   fill matrix and rhs with random numbers
   ---------------------------------------
*/
MARKTIME(t1) ;
Drand_fillDvector(drand, nrow*ncol, A->entries) ;
Drand_fillDvector(drand, ncol*nrhs, X->entries) ;
Drand_fillDvector(drand, nrow*nrhs, Y->entries) ;
DA2_copy(Z, Y) ;
MARKTIME(t2) ;
fprintf(msgFile, 
        "\n CPU : %.3f to fill matrix with random numbers", t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n matrix A") ;
   DA2_writeForHumanEye(A, msgFile) ;
   fprintf(msgFile, "\n matrix X") ;
   DA2_writeForHumanEye(X, msgFile) ;
   fprintf(msgFile, "\n matrix Y") ;
   DA2_writeForHumanEye(Y, msgFile) ;
   fprintf(msgFile, "\n matrix Z") ;
   DA2_writeForHumanEye(Z, msgFile) ;
}
/*
   ----------------------------------
   compute the matrix vector multiply
   using the simplest kernel
   ----------------------------------
*/
ops = 2*nrow*ncol*nrhs ;
MARKTIME(t1) ;
for ( jj = 0 ; jj < nrhs ; jj++ ) {
   DA2_extractColumnDV(X, colDV, jj) ;
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      DA2_extractRowDV(A, rowDV, ii) ;
      value = alpha * DVdot(ncol, row, col) ;
      DA2_addEntry(Y, ii, jj, value) ;
   }
}
MARKTIME(t2) ;
fprintf(msgFile, 
"\n CPU : %.3f compute simplest mvm, %.0f ops, %.3f megaflops", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Y") ;
   DA2_writeForHumanEye(Y, msgFile) ;
}
/*
   -------------------------
   now test the 1-vector MVM
   -------------------------
*/
MARKTIME(t1) ;
x0 = X->entries ;
z0 = Z->entries ;
for ( irhs = 0 ; irhs < nrhs ; irhs++ ) {
   DA2_mvm1vec(A, z0, alpha, x0) ;
   x0 = x0 + ncol ;
   z0 = z0 + nrow ;
}
MARKTIME(t2) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Z") ;
   DA2_writeForHumanEye(Z, msgFile) ;
}
DA2_sub(Y, Z) ;
fprintf(msgFile, 
"\n CPU : %.3f 1-vector mvm, %.0f ops, %.3f megaflops, error = %12.4e", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1), DA2_frobNorm(Y)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n error") ;
   DA2_writeForHumanEye(Y, msgFile) ;
}
/*
   ---------------------------------------
   fill matrix and rhs with random numbers
   ---------------------------------------
*/
MARKTIME(t1) ;
Drand_fillDvector(drand, nrow*ncol, A->entries) ;
Drand_fillDvector(drand, ncol*nrhs, X->entries) ;
Drand_fillDvector(drand, nrow*nrhs, Y->entries) ;
DA2_copy(Z, Y) ;
MARKTIME(t2) ;
fprintf(msgFile, 
        "\n CPU : %.3f to fill matrix with random numbers", t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n matrix A") ;
   DA2_writeForHumanEye(A, msgFile) ;
   fprintf(msgFile, "\n matrix X") ;
   DA2_writeForHumanEye(X, msgFile) ;
   fprintf(msgFile, "\n matrix Y") ;
   DA2_writeForHumanEye(Y, msgFile) ;
   fprintf(msgFile, "\n matrix Z") ;
   DA2_writeForHumanEye(Z, msgFile) ;
}
/*
   ----------------------------------
   compute the matrix vector multiply
   using the simplest kernel
   ----------------------------------
*/
ops = 2*nrow*ncol*nrhs ;
MARKTIME(t1) ;
for ( jj = 0 ; jj < nrhs ; jj++ ) {
   DA2_extractColumnDV(X, colDV, jj) ;
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      DA2_extractRowDV(A, rowDV, ii) ;
      value = alpha * DVdot(ncol, row, col) ;
      DA2_addEntry(Y, ii, jj, value) ;
   }
}
MARKTIME(t2) ;
fprintf(msgFile, 
"\n CPU : %.3f compute simplest mvm, %.0f ops, %.3f megaflops", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Y") ;
   DA2_writeForHumanEye(Y, msgFile) ;
}
/*
   -------------------------
   now test the 2-vector MVM
   -------------------------
*/
MARKTIME(t1) ;
x0 = X->entries ;
z0 = Z->entries ;
for ( irhs = 0 ; irhs < nrhs - 1 ; irhs += 2 ) {
   x1 = x0 + ncol ;
   z1 = z0 + nrow ;
   DA2_mvm2vec(A, z0, z1, alpha, x0, x1) ;
   x0 = x1 + ncol ;
   z0 = z1 + nrow ;
}
if ( irhs == nrhs - 1 ) {
   DA2_mvm1vec(A, z0, alpha, x0) ;
}
MARKTIME(t2) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Z") ;
   DA2_writeForHumanEye(Z, msgFile) ;
}
DA2_sub(Y, Z) ;
fprintf(msgFile, 
"\n CPU : %.3f 2-vector mvm, %.0f ops, %.3f megaflops, error = %12.4e", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1), DA2_frobNorm(Y)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n error") ;
   DA2_writeForHumanEye(Y, msgFile) ;
}
/*
   ---------------------------------------
   fill matrix and rhs with random numbers
   ---------------------------------------
*/
MARKTIME(t1) ;
Drand_fillDvector(drand, nrow*ncol, A->entries) ;
Drand_fillDvector(drand, ncol*nrhs, X->entries) ;
Drand_fillDvector(drand, nrow*nrhs, Y->entries) ;
DA2_copy(Z, Y) ;
MARKTIME(t2) ;
fprintf(msgFile, 
        "\n CPU : %.3f to fill matrix with random numbers", t2 - t1) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n matrix A") ;
   DA2_writeForHumanEye(A, msgFile) ;
   fprintf(msgFile, "\n matrix X") ;
   DA2_writeForHumanEye(X, msgFile) ;
   fprintf(msgFile, "\n matrix Y") ;
   DA2_writeForHumanEye(Y, msgFile) ;
   fprintf(msgFile, "\n matrix Z") ;
   DA2_writeForHumanEye(Z, msgFile) ;
}
/*
   ----------------------------------
   compute the matrix vector multiply
   using the simplest kernel
   ----------------------------------
*/
ops = 2*nrow*ncol*nrhs ;
MARKTIME(t1) ;
for ( jj = 0 ; jj < nrhs ; jj++ ) {
   DA2_extractColumnDV(X, colDV, jj) ;
   for ( ii = 0 ; ii < nrow ; ii++ ) {
      DA2_extractRowDV(A, rowDV, ii) ;
      value = alpha * DVdot(ncol, row, col) ;
      DA2_addEntry(Y, ii, jj, value) ;
   }
}
MARKTIME(t2) ;
fprintf(msgFile, 
"\n CPU : %.3f compute simplest mvm, %.0f ops, %.3f megaflops", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Y") ;
   DA2_writeForHumanEye(Y, msgFile) ;
}
/*
   -------------------------
   now test the 3-vector MVM
   -------------------------
*/
MARKTIME(t1) ;
x0 = X->entries ;
z0 = Z->entries ;
for ( irhs = 0 ; irhs < nrhs - 2 ; irhs += 3 ) {
   x1 = x0 + ncol ;
   x2 = x1 + ncol ;
   z1 = z0 + nrow ;
   z2 = z1 + nrow ;
   DA2_mvm3vec(A, z0, z1, z2, alpha, x0, x1, x2) ;
   x0 = x2 + ncol ;
   z0 = z2 + nrow ;
}
if ( irhs == nrhs - 2 ) {
   x1 = x0 + ncol ;
   z1 = z0 + nrow ;
   DA2_mvm2vec(A, z0, z1, alpha, x0, x1) ;
} else if ( irhs == nrhs - 1 ) {
   DA2_mvm1vec(A, z0, alpha, x0) ;
}
MARKTIME(t2) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n Z") ;
   DA2_writeForHumanEye(Z, msgFile) ;
}
DA2_sub(Y, Z) ;
fprintf(msgFile, 
"\n CPU : %.3f 3-vector mvm, %.0f ops, %.3f megaflops, error = %12.4e", 
    t2 - t1, ops, 1.0e-6*ops/(t2 - t1), DA2_frobNorm(Y)) ;
if ( msglvl > 3 ) {
   fprintf(msgFile, "\n error") ;
   DA2_writeForHumanEye(Y, msgFile) ;
}

fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
