/*  testLocalMaxes.c  */

#include "../DA2.h"
#include "../../timings.h"

/*--------------------------------------------------------------------*/

void
main ( int argc, char *argv[] )
/*
   ------------------------------------------------------------
   test the DA2_findLocalMax() and DA2_findLocalMaxes() methods

   created -- 97mar08, cca
   ------------------------------------------------------------
*/
{
double   maxInCol, maxInRow, t1, t2, value ;
double   *cpus, *vals ;
DA2      *a ;
FILE     *msgFile ;
int      ii, inc1, inc2, irow, itest, j, jcol, msglvl, n1, n2, 
         nsearch, ntest, s, seed ;
int      *cmark, *cols, *nsearches, *rmark, *rows ;

if ( argc != 9 ) {
   fprintf(stdout, 
"\n usage : testLocalMaxes msglvl msgFile n1 n2 inc1 inc2 "
"\n       seed ntest"
"\n    msglvl  -- message level"
"\n    msgFile -- message file"
"\n    n1      -- # of rows"
"\n    n2      -- # of columns"
"\n    inc1    -- row increment"
"\n    inc2    -- column increment"
"\n    seed    -- random number seed"
"\n    ntest   -- # of tests"
"\n") ;
   return ;
}
if ( (msglvl = atoi(argv[1])) < 0 ) {
   fprintf(stderr, "\n message level must be positive\n") ;
   exit(-1) ;
}
if ( strcmp(argv[2], "stdout") == 0 ) {
   msgFile = stdout ;
} else if ( (msgFile = fopen(argv[2], "a")) == NULL ) {
   fprintf(stderr, "\n unable to open file %s\n", argv[2]) ;
   return ;
}
if ( (n1 = atoi(argv[3])) < 0 || (n2 = atoi(argv[4])) < 0 ) {
   fprintf(stderr, "\n matrix dimension must be positive\n") ;
   exit(-1) ;
}
inc1  = atoi(argv[5]) ;
inc2  = atoi(argv[6]) ;
seed  = atoi(argv[7]) ;
ntest = atoi(argv[8]) ;
if ( ntest % 2 == 0 ) {
   ntest++ ;
}
/*
   ----------------------------
   initialize the matrix object
   ----------------------------
*/
a = DA2_new() ;
MARKTIME(t1) ;
DA2_init(a, n1, n2, 1, n1, NULL) ;
MARKTIME(t2) ;
fprintf(msgFile, "\n CPU : %.3f to initialize %d x %d object",
        t2 - t1, n1, n2) ;
/*
   ------------------------------------
   time the multiple local max searches
   ------------------------------------
*/
rmark     = IVinit(n1, -1) ;
cmark     = IVinit(n2, -1) ;
rows      = IVinit(8, -1) ;
cols      = IVinit(8, -1) ;
vals      = DVinit(8, -1) ;
nsearches = IVinit(8*ntest, 0) ;
cpus      = DVinit(8*ntest, 0) ;
for ( itest = 0 ; itest < ntest ; itest++ ) {
   IVfill(n1, rmark, -1) ;
   IVfill(n2, cmark, -1) ;
   for ( s = 1 ; s <= 8 ; s++ ) {
      DA2_fillRandomUniform(a, 0, 1, seed++) ;
      MARKTIME(t1) ;
      nsearch = DA2_findLocalMaxes(a, s, rows, cols, vals, 
                                   rmark, cmark, s) ;
      MARKTIME(t2) ;
      nsearches[itest + (s-1)*ntest] = nsearch ;
      cpus[itest + (s-1)*ntest] = t2 - t1 ;
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n s = %d, nsearch = %d, time = %.4f", 
                 s, nsearch, t2 - t1) ;
      }
      if ( msglvl > 1 ) {
         fprintf(msgFile, "\n elems = {") ;
         for ( j = 0 ; j < s ; j++ ) {
            fprintf(msgFile, " (%d,%d,%f)", rows[j], cols[j], vals[j]) ;
         }
         fprintf(msgFile, "}") ;
      }
/*
      ------------------------------------------
      verify that the entry is a local max entry
      ------------------------------------------
*/
      for ( ii = 0 ; ii < s ; ii++ ) {
         irow  = rows[ii] ;
         jcol  = cols[ii] ;
         value = vals[ii] ;
         maxInRow = DA2_infinityNormOfRow(a, irow) ;
         maxInCol = DA2_infinityNormOfRow(a, irow) ;
         if ( value != maxInRow || value != maxInCol ) {
            fprintf(stderr, 
    "\n error! value = %20.12e, maxInRow = %20.12e, maxInCol = %20.12e",
     value, maxInRow, maxInCol) ;
         }
      }
   }
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n stats") ;
   for ( itest = 0 ; itest < ntest ; itest++ ) {
      fprintf(msgFile, "\n      %4d %4d %4d %4d %4d %4d %4d %4d",
              nsearches[itest],
              nsearches[itest+ntest],
              nsearches[itest+2*ntest],
              nsearches[itest+3*ntest],
              nsearches[itest+4*ntest],
              nsearches[itest+5*ntest],
              nsearches[itest+6*ntest],
              nsearches[itest+7*ntest]) ;
   }
}
for ( s = 0 ; s < 8 ; s++ ) {
   IVqsortUp(ntest, &nsearches[s*ntest]) ;
   DVqsortUp(ntest, &cpus[s*ntest]) ;
}
if ( msglvl > 1 ) {
   fprintf(msgFile, "\n after sort, #nsearches") ;
   for ( itest = 0 ; itest < ntest ; itest++ ) {
      fprintf(msgFile, "\n      %4d %4d %4d %4d %4d %4d %4d %4d",
              nsearches[itest],
              nsearches[itest+ntest],
              nsearches[itest+2*ntest],
              nsearches[itest+3*ntest],
              nsearches[itest+4*ntest],
              nsearches[itest+5*ntest],
              nsearches[itest+6*ntest],
              nsearches[itest+7*ntest]) ;
   }
}
fprintf(msgFile, "\n\n # of searches to find a local max entry") ;
fprintf(msgFile, "\n  min %8d %8d %8d %8d %8d %8d %8d %8d",
        nsearches[0],
        nsearches[ntest],
        nsearches[2*ntest],
        nsearches[3*ntest],
        nsearches[4*ntest],
        nsearches[5*ntest],
        nsearches[6*ntest],
        nsearches[7*ntest]) ;
fprintf(msgFile, "\n  med %8d %8d %8d %8d %8d %8d %8d %8d",
        nsearches[ntest/2 + 0], 
        nsearches[ntest/2 + ntest], 
        nsearches[ntest/2 + 2*ntest], 
        nsearches[ntest/2 + 3*ntest],
        nsearches[ntest/2 + 4*ntest],
        nsearches[ntest/2 + 5*ntest],
        nsearches[ntest/2 + 6*ntest],
        nsearches[ntest/2 + 7*ntest]) ;
fprintf(msgFile, "\n  max %8d %8d %8d %8d %8d %8d %8d %8d",
        nsearches[ntest-1 + 0], 
        nsearches[ntest-1 + ntest], 
        nsearches[ntest-1 + 2*ntest], 
        nsearches[ntest-1 + 3*ntest], 
        nsearches[ntest-1 + 4*ntest], 
        nsearches[ntest-1 + 5*ntest], 
        nsearches[ntest-1 + 6*ntest],
        nsearches[ntest-1 + 7*ntest]) ;
fprintf(msgFile, 
        "\n  avg %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f %8.3f ",
        ((float) IVsum(ntest, &nsearches[0]))/ntest,
        ((float) IVsum(ntest, &nsearches[ntest]))/(2*ntest),
        ((float) IVsum(ntest, &nsearches[2*ntest]))/(3*ntest),
        ((float) IVsum(ntest, &nsearches[3*ntest]))/(4*ntest),
        ((float) IVsum(ntest, &nsearches[4*ntest]))/(5*ntest),
        ((float) IVsum(ntest, &nsearches[5*ntest]))/(6*ntest),
        ((float) IVsum(ntest, &nsearches[6*ntest]))/(7*ntest),
        ((float) IVsum(ntest, &nsearches[7*ntest]))/(8*ntest)) ;

fprintf(msgFile, "\n\n time to find a local max entry") ;
fprintf(msgFile, "\n  min %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f",
        cpus[0],
        cpus[ntest],
        cpus[2*ntest],
        cpus[3*ntest],
        cpus[4*ntest],
        cpus[5*ntest],
        cpus[6*ntest],
        cpus[7*ntest]) ;
fprintf(msgFile, "\n  med %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f",
        cpus[ntest/2 + 0], 
        cpus[ntest/2 + ntest], 
        cpus[ntest/2 + 2*ntest], 
        cpus[ntest/2 + 3*ntest],
        cpus[ntest/2 + 4*ntest],
        cpus[ntest/2 + 5*ntest],
        cpus[ntest/2 + 6*ntest],
        cpus[ntest/2 + 7*ntest]) ;
fprintf(msgFile, "\n  max %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f",
        cpus[ntest-1 + 0], 
        cpus[ntest-1 + ntest], 
        cpus[ntest-1 + 2*ntest], 
        cpus[ntest-1 + 3*ntest], 
        cpus[ntest-1 + 4*ntest], 
        cpus[ntest-1 + 5*ntest], 
        cpus[ntest-1 + 6*ntest],
        cpus[ntest-1 + 7*ntest]) ;
fprintf(msgFile, 
        "\n  avg %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f %8.5f ",
        DVsum(ntest, &cpus[0])/ntest,
        DVsum(ntest, &cpus[ntest])/(2*ntest),
        DVsum(ntest, &cpus[2*ntest])/(3*ntest),
        DVsum(ntest, &cpus[3*ntest])/(4*ntest),
        DVsum(ntest, &cpus[4*ntest])/(5*ntest),
        DVsum(ntest, &cpus[5*ntest])/(6*ntest),
        DVsum(ntest, &cpus[6*ntest])/(7*ntest),
        DVsum(ntest, &cpus[7*ntest])/(8*ntest)) ;

fprintf(msgFile, "\n") ;

return ; }

/*--------------------------------------------------------------------*/
