/*  DA2.h  */

#include "../cfiles.h"
#include "../CV.h"
#include "../IV.h"
#include "../DV.h"

/*====================================================================*/
/*
   ----------------------------------------------------
   the DA2 structure holds a 2-d dense array of doubles

   n1 -- number of rows (first dimension)
   n2 -- number of columns (second dimension)
   inc1 -- 
      increment in storage along first dimension,
      when inc1 == 1, the storage is column major
   inc2 -- 
      increment in storage along second dimension,
      when inc2 == 1, the storage is row major
   nowned -- number of owned entries starting at entries
      when > 0, storage pointed to by entries
      has been allocated here and can be free'd.
      when = 0, storage pointed to by entries
      has not been allocated here and cannot be free'd.
   entries -- base address for the entries

   entry(i,j) is found in entries[i*inc1 + j*inc2] 
   ----------------------------------------------------
*/
typedef
struct _DA2 {
   int      n1       ;
   int      n2       ;
   int      inc1     ;
   int      inc2     ;
   int      nowned   ;
   double   *entries ;
} DA2 ;

/*====================================================================*/
/*
------------------------------------------------------------------------
----- methods found in basics.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------
   simplest constructor
 
   created -- 96apr19, cca
   -----------------------
*/
DA2 *
DA2_new ( 
   void 
) ;
/*
   -----------------------
   set the default fields
 
   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setDefaultFields (
   DA2   *mtx
) ;
/*
   --------------------------------------------------
   clear the data fields, releasing allocated storage
 
   created -- 96apr19, cca
   --------------------------------------------------
*/
void
DA2_clearData (
   DA2   *mtx
) ;
/*
   ------------------------------------------
   destructor, free's the object and its data
 
   created -- 96apr19, cca
   ------------------------------------------
*/
void
DA2_free (
   DA2   *mtx
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in instance.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------
   return the number of rows in the array
 
   created -- 96apr27, cca
   --------------------------------------
*/
int
DA2_nrow (
   DA2   *mtx
) ;
/*
   -----------------------------------------
   return the number of columns in the array
 
   created -- 96apr27, cca
   -----------------------------------------
*/
int
DA2_ncol (
   DA2   *mtx
) ;
/*
   --------------------------
   return the first increment
 
   created -- 96nov18, cca
   --------------------------
*/
int
DA2_inc1 (
   DA2  *mtx
) ;
/*
   ---------------------------
   return the second increment
 
   created -- 96nov18, cca
   ---------------------------
*/
int
DA2_inc2 (
   DA2  *mtx
) ;
/*
   -------------------------------
   return a pointer to the entries
 
   created -- 96apr19, cca
   -------------------------------
*/
double *
DA2_entries (
   DA2      *mtx
) ;
/*
   --------------------------------------------
   return a pointer to the first entry in a row
 
   created -- 96nov27, cca
   --------------------------------------------
*/
double *
DA2_row (
   DA2   *mtx,
   int   irow
) ;
/*
   -----------------------------------------------
   return a pointer to the first entry in a column
 
   created -- 96nov27, cca
   -----------------------------------------------
*/
double *
DA2_column (
   DA2   *mtx,
   int   jcol
) ;
/*
   ------------------------------------------
   return a pointer to the matrix first entry
   in row irow and column jcol
 
   created -- 96nov27, cca
   ------------------------------------------
*/
double *
DA2_pointerToEntry (
   DA2   *mtx,
   int   irow,
   int   jcol
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in init.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------------------
   initializer. sets the n1, n2, inc1 and inc2 fields.
   must have
      mtx != NULL
      n1, n2, inc1, inc2 > 0
      (inc1 = 1 and inc2 = nrow) or (inc1 = ncol and inc2 = 1)
 
   if entries is NULL then
      entries are allocated and zeroed.
   else
      mtx->nowned is set to 0
      mtx->entries is set to entries.
   endif
   n1, n2, inc1, inc2 set
 
   created -- 96apr19, cca
   ------------------------------------------------------------------
*/
void
DA2_init ( 
   DA2      *mtx, 
   int      n1, 
   int      n2, 
   int      inc1, 
   int      inc2, 
   double   *entries 
) ;
/*
   --------------------------------------------------
   submatrix initializer
 
   A(0:lastrow-firstrow,0:lastcol-firstcol)
              = B(firstrow:lastrow, firstcol:lastcol)
 
   created -- 96aug18, cca
   --------------------------------------------------
*/
void
DA2_subDA2 (
   DA2   *mtxA,
   DA2   *mtxB,
   int   firstrow,
   int   lastrow,
   int   firstcol,
   int   lastcol
) ;
/*
   ------------------------------------------------
   resize the object
 
   if the object does not own its data then
      if entries are present then
         if not enough entries are present
            write error message and exit
         endif
      else
         allocate and own new entries
      endif
   else if not enough entries present then
      if old entries are present then
         free old entries
      endif
      allocate and own new entries
   endif
   set dimensions and increments
 
   created  -- 96nov16, cca
   ------------------------------------------------
*/
void
DA2_resize (
   DA2   *mtx,
   int   nrow,
   int   ncol,
   int   inc1,
   int   inc2
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in search.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------------------
   return first q such that |a(irow,q)| = max_{j} |a(irow,j)|
   ----------------------------------------------------------
*/
int
DA2_findMaxAbsInRow (
   DA2      *a,
   int      irow,
   double   *pval
) ;
/*
   ----------------------------------------------------------
   return first q such that |a(q,jcol)| = max_{i} |a(i,jcol)|
   ----------------------------------------------------------
*/
int
DA2_findMaxAbsInColumn (
   DA2      *a,
   int      jcol,
   double   *pval
) ;
/*
   --------------------------------------------------------
   find a local max entry
 
   if ( (istart,jstart) == (-1,-1) ) {
      search row 0 and column 0 and then continue
   } else if ( (istart,jstart) == (-1,jcol) ) {
      search column jcol and continue
   } else if ( (istart,jstart) == (irow,-1) ) {
      search row irow and continue
   } else {
      search row istart and column jstart and then continue
   }
 
   set *pirow to row
   set *picol to column
   set *pmaxval to local max value
 
   return value -- # of searches performed
   --------------------------------------------------------
*/
int
DA2_findLocalMax (
   DA2      *a,
   int      istart,
   int      jstart,
   int      *pirow,
   int      *pjcol,
   double   *pmaxval
) ;
/*
   -----------------------------------------------------------------
   find s local max elements
 
   rows[]  -- vector to be filled with row ids, -1 means no pivot
   cols[]  -- vector to be filled with column ids, -1 means no pivot
   vals[]  -- vector to be filled with pivot elements,
              if NULL, then ignored, otherwise filled
   rmark[] -- mark vector for rows, on entry rmark[*] != tag,
              on exit, if irow was searched, rmark[irow] = tag
   cmark[] -- mark vector for columns, on entry cmark[*] != tag
              on exit, if jcol was searched, cmark[jcol] = tag
   tag     -- tag for mark vector
 
   return value -- # of searches
   -----------------------------------------------------------------
*/
int
DA2_findLocalMaxes (
   DA2      *a,
   int      s,
   int      rows[],
   int      cols[],
   double   vals[],
   int      rmark[],
   int      cmark[],
   int      tag
) ;
/*
   ---------------------------------------------
   purpose -- to find s quasimax entries
 
   examine only those rows and columns such that
   rowmark[irow] == tag and colmark[irow] == tag
 
   created -- 96sep07, cca
   ---------------------------------------------
*/
int
DA2_findQuasimaxes (
   DA2      *mtx,
   int      s,
   int      rowids[],
   int      colids[],
   double   maxvals[],
   int      rowmark[],
   int      colmark[],
   int      tag
) ;
/*
   ------------------------------------------------------------
   purpose -- return the location of a quasi-max entry,
              an entry whose magnitude is greater than or equal
              to all entries in its row and column that belong
              to tagged rows and columns.
 
   created -- 96sep07, cca
   ------------------------------------------------------------
*/
void
DA2_quasimax (
   DA2      *mtx,
   int      rowmark[],
   int      colmark[],
   int      tag,
   int      *pirow,
   int      *pjcol,
   double   *pmaxval
) ;
/*
   ------------------------------------------------------------
   purpose -- return the location of a quasi-max entry,
              an entry whose magnitude is greater than or equal
              to all entries in its row and column that belong
              to tagged rows and columns.
 
   note: the matrix is square and symmetric, with entries
         stored only in the upper triangle.
 
   created -- 96sep20, cca
   ------------------------------------------------------------
*/
void
DA2_symQuasimax (
   DA2      *mtx,
   int      mark[],
   int      tag,
   int      *pirow,
   int      *pjcol,
   double   *pmaxval
) ;
/*
   ------------------------------------------------------------
   purpose -- return the location of the unmarked diagonal
      entry with the largest magnitude.
 
   created -- 97jan11, cca
   ------------------------------------------------------------
*/
int
DA2_maxDiagPivot (
   DA2      *mtx,
   int      mark[],
   int      tag
) ;
/*
   ------------------------------------------------------------
   purpose -- find a 1x1 or 2x2 pivot using
              the fast Bunch-Parlett algorithm
 
   note: the matrix is square and symmetric, with entries
         stored only in the upper triangle.
 
   created -- 96sep20, cca
   ------------------------------------------------------------
*/
void
DA2_fastBunchParlettPivot (
   DA2      *mtx,
   int      mark[],
   int      tag,
   int      *pirow,
   int      *pjcol
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in sort.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------
   permute the rows of the matrix
   A(*,*) = A(index(*),*)
   this method calls DA2_sortRowsUp
   but does not overwrite the index[] vector
 
   created -- 97nov17, cca
   -----------------------------------------
*/
void
DA2_permuteRows (
   DA2   *mtx,
   int   nrow,
   int   index[]
) ;
/*
   -----------------------------------------
   permute the columns of the matrix
   A(*,*) = A(*,index(*))
   this method calls DA2_sortColumnsUp
   but does not overwrite the index[] vector
 
   created -- 97nov17, cca
   -----------------------------------------
*/
void
DA2_permuteColumns (
   DA2   *mtx,
   int   ncol,
   int   index[]
) ;
/*
   ----------------------------------------------
   sort the rows of the matrix in ascending order
   of the rowids[] vector. on return, rowids is
   in asending order. return value is the number
   of row swaps made.
 
   created -- 97jan20, cca
   ----------------------------------------------
*/
int
DA2_sortRowsUp (
   DA2   *mtx,
   int   nrow,
   int   rowids[]
) ;
/*
   -------------------------------------------------
   sort the columns of the matrix in ascending order
   of the colids[] vector. on return, colids is
   in asending order. return value is the number
   of column swaps made.
 
   created -- 97jan20, cca
   -------------------------------------------------
*/
int
DA2_sortColumnsUp (
   DA2   *mtx,
   int   ncol,
   int   colids[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in norms.c -----------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------
   return the entry of maximum magnitude
 
   created -- 96sep15, cca
   -------------------------------------
*/
double
DA2_maxabs (
   DA2   *a
) ;
/*
   ---------------------------------------
   return the frobenius norm of the matrix
 
   created -- 96may03, cca
   ---------------------------------------
*/
double
DA2_frobNorm (
   DA2   *mtx
) ;
/*
   ---------------------------------
   return the one-norm of the matrix
 
   created -- 96may03, cca
   ---------------------------------
*/
double
DA2_oneNorm (
   DA2   *mtx
) ;
/*
   --------------------------------------
   return the infinity-norm of the matrix
 
   created -- 96may03, cca
   --------------------------------------
*/
double
DA2_infinityNorm (
   DA2   *mtx
) ;
/*
   ----------------------------------
   return the one-norm of column jcol
 
   created -- 96nov27, cca
   ----------------------------------
*/
double
DA2_oneNormOfColumn (
   DA2   *mtx,
   int   jcol
) ;
/*
   ----------------------------------
   return the two-norm of column jcol
 
   created -- 96nov27, cca
   ----------------------------------
*/
double
DA2_twoNormOfColumn (
   DA2   *mtx,
   int   jcol
) ;
/*
   ---------------------------------------
   return the infinity-norm of column jcol
 
   created -- 96nov27, cca
   ---------------------------------------
*/
double
DA2_infinityNormOfColumn (
   DA2   *mtx,
   int   jcol
) ;
/*
   -------------------------------
   return the one-norm of row irow
 
   created -- 96nov27, cca
   -------------------------------
*/
double
DA2_oneNormOfRow (
   DA2   *mtx,
   int   irow
) ;
/*
   -------------------------------
   return the two-norm of row irow
 
   created -- 96nov27, cca
   -------------------------------
*/
double
DA2_twoNormOfRow (
   DA2   *mtx,
   int   irow
) ;
/*
   ------------------------------------
   return the infinity-norm of row irow
 
   created -- 96nov27, cca
   ------------------------------------
*/
double
DA2_infinityNormOfRow (
   DA2   *mtx,
   int   irow
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in scale.c ----------------------------------------=
------------------------------------------------------------------------
*/
/*
   -----------------------------
   purpose -- replace A by B * A
 
   created -- 96sep14, cca
   -----------------------------
*/
void
DA2_scaleOnLeft (
   DA2   *A,
   DA2   *B,
   DV    *tmpDV
) ;
/*
   -----------------------------
   purpose -- replace A by A * B
 
   created -- 96sep14, cca
   -----------------------------
*/
void
DA2_scaleOnRight (
   DA2   *A,
   DA2   *B,
   DV    *tmpDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in scatter.c ---------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------
   scatter entries of A into B
      B(rowids, colids) = A
   rowids -- ids of rows to be scattered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be scattered,
      must be not NULL if B->n2 != A->n2
 
   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_scatter (
   DA2   *B,
   int   *rowids,
   int   *colids,
   DA2   *A
) ;
/*
   -------------------------------------------
   scatter/add entries of A into B
      B(rowids, colids) += A
   rowids -- ids of rows to be scattered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be scattered,
      must be not NULL if B->n2 != A->n2
 
   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_scatterAdd (
   DA2   *B,
   int   *rowids,
   int   *colids,
   DA2   *A
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in gather.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   -------------------------------------------
   gather entries
      B = A(rowids, colids)
   rowids -- ids of rows to be gathered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be gathered,
      must be not NULL if B->n2 != A->n2
 
   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_gather (
   DA2   *B,
   DA2   *A,
   int   *rowids,
   int   *colids
) ;
/*
   -------------------------------------------
   gather entries from A into B
      B = A(rowids, colids)
   and zero entries in A
   rowids -- ids of rows to be gathered,
      must be not NULL if B->n1 != A->n1
   colids -- ids of columns to be gathered,
      must be not NULL if B->n2 != A->n2
 
   created -- 96apr27, cca
   -------------------------------------------
*/
void
DA2_gatherZero (
   DA2   *B,
   DA2   *A,
   int   *rowids,
   int   *colids
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in invert.c ----------------------------------------
------------------------------------------------------------------------
*/
/*
   ---------------------------------------------------
   on input,
      A is the matrix to be inverted
      B contains the identity matrix
   on output,
      A contains the identity matrix
      B contains A^{-1}
 
   return value --
      1 if A is nonsingular
      0 if A is singular
 
   created -- 96jun29, cca
   ---------------------------------------------------
*/
int
DA2_invert (
   DA2   *A,
   DA2   *B
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in mvm.c -------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------
   matrix vector multiply
 
   y[] := y[] + alpha * a * x[]
 
   created -- 96apr19, cca
   ----------------------------
*/
void
DA2_mvm1vec (
   DA2      *a,
   double   y[],
   double   alpha,
   double   x[]
) ;
/*
   ----------------------------------------------
   matrix vector multiply
 
   [ y1 y2 ] := [ y1 y2 ] + alpha * a * [ x1 x2 ]
 
   created -- 96apr19, cca
   ----------------------------------------------
*/
void
DA2_mvm2vec (
   DA2      *a,
   double   y0[],
   double   y1[],
   double   alpha,
   double   x0[],
   double   x1[]
) ;
/*
   -------------------------------------------------------
   matrix vector multiply
 
   [ y1 y2 y3 ] := [ y1 y2 y3 ] + alpha * a * [ x1 x2 x3 ]
 
   created -- 96apr19, cca
   -------------------------------------------------------
*/
void
DA2_mvm3vec (
   DA2      *a,
   double   y0[],
   double   y1[],
   double   y2[],
   double   alpha,
   double   x0[],
   double   x1[],
   double   x2[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in sparseMVM.c -------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------
   purpose -- do a sparse matrix vector multiply
 
   if yind == NULL && xind == NULL
     y[*] = y[*] + alpha * A * x[*]
   else if yind == NULL && xind != NULL
     y[*] = y[*] + alpha * A * x[xind[*]]
   else if yind != NULL && xind == NULL
     y[yind[*]] = y[yind[*]] + alpha * A * x[*]
   else if yind != NULL && xind != NULL
     y[yind[*]] = y[yind[*]] + alpha * A * x[xind[*]]
   endif
 
   created -- 97apr04, cca
   --------------------------------------------------
*/
void
DA2_sparseMVM (
   DA2      *mtxA,
   double   y[],
   int      yind[],
   double   alpha,
   double   x[],
   int      xind[]
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in Upd.c -------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------
   compute a nonsymmetric dense update A -= B * C
 
   created -- 96oct18, cca
   ---------------------------------------------
*/
void
DA2_ndUpd (
   DA2   *A,
   DA2   *B,
   DA2   *C
) ;
/*
   -----------------------------------------------
   compute a nonsymmetric sparse update A -= B * C
 
   created -- 96oct18, cca
   ----------------------------------------------
*/
void
DA2_nsUpd (
   DA2   *A,
   int   rowloc[],
   int   colloc[],
   DA2   *B,
   DA2   *C
) ;
/*
   -------------------------------------------------
   nonsymmetric dense 3 term update A -= X * Y * Z^T
 
   created -- 96oct18, cca
   -------------------------------------------------
*/
void
DA2_nd3Upd (
   DA2   *A,
   DA2   *X,
   DA2   *Y,
   DA2   *Z,
   DV    *tmpDV
) ;
/*
   --------------------------------------------------
   nonsymmetric sparse 3 term update A -= X * Y * Z^T
 
   created -- 96oct18, cca
   --------------------------------------------------
*/
void
DA2_ns3Upd (
   DA2   *A,
   int   rowloc[],
   int   colloc[],
   DA2   *X,
   DA2   *Y,
   DA2   *Z,
   DV    *tmpDV
) ;
/*
   -------------------------------------------------
   symmetric dense 3 term update A -= X * Y * X^T
   only the upper triangle of A is updated
 
   created -- 96oct18, cca
   -------------------------------------------------
*/
void
DA2_sd3Upd (
   DA2   *A,
   DA2   *X,
   DA2   *Y,
   DV    *tmpDV
) ;
/*
   -------------------------------------------------
   symmetric sparse 3 term update A -= X * Y * X^T
   only the upper triangle of A is updated
 
   created -- 96oct18, cca
   -------------------------------------------------
*/
void
DA2_ss3Upd (
   DA2   *A,
   int   locind[],
   DA2   *X,
   DA2   *Y,
   DV    *tmpDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in util.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ----------------------------------------------
   return the number of bytes taken by the object
 
   created -- 96apr19, cca
   ----------------------------------------------
*/
int
DA2_sizeOf (
   DA2   *mtx
) ;
/*
   ---------------------------------------------------------------
   shift the base of the entries and adjust dimensions
 
   mtx(0:n1-rowoff-1,0:n2-coloff-1) = mtx(rowoff:n1-1,coloff:n2-1)
 
   created -- 96apr19, cca
   ---------------------------------------------------------------
*/
void
DA2_shiftBase (
   DA2   *mtx,
   int   rowoff,
   int   coloff
) ;
/*
   --------------------------------------------------------------
   returns 1 if the storage is row major, otherwise returns zero.
 
   created -- 96apr19, cca
   --------------------------------------------------------------
*/
int
DA2_rowMajor (
   DA2   *mtx
) ;
/*
   -----------------------------------------------------------------
   returns 1 if the storage is column major, otherwise returns zero.
 
   created -- 96apr19, cca
   -----------------------------------------------------------------
*/
int
DA2_columnMajor (
   DA2   *mtx
) ;
/*
   -----------------------
   transpose the matrix
 
   created -- 96jun29, cca
   -----------------------
*/
void
DA2_transpose (
   DA2   *mtx
) ;
/*
   ----------------------------
   extract row[*] = mtx(irow,*)
 
   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractRow (
   DA2      *mtx,
   double   row[],
   int      irow
) ;
/*
   ----------------------------
   extract col[*] = mtx(*,jcol)
 
   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractColumn (
   DA2      *mtx,
   double   col[],
   int      jcol
) ;
/*
   -----------------------
   set mtx(irow,*) = y[*]
 
   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setRow (
   DA2      *mtx,
   double   row[],
   int      irow
) ;
/*
   -----------------------
   set mtx(*,jcol) = y[*]
 
   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setColumn (
   DA2      *mtx,
   double   col[],
   int      jcol
) ;
/*
   ----------------------------
   extract row[*] = mtx(irow,*)
 
   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractRowDV (
   DA2   *mtx,
   DV    *rowDV,
   int   irow
) ;
/*
   ----------------------------
   extract col[*] = mtx(*,jcol)
 
   created -- 96nov18, cca
   ----------------------------
*/
void
DA2_extractColumnDV (
   DA2   *mtx,
   DV    *colDV,
   int   jcol
) ;
/*
   -----------------------
   set mtx(irow,*) = y[*]
 
   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setRowDV (
   DA2      *mtx,
   DV       *rowDV,
   int      irow
) ;
/*
   -----------------------
   set mtx(*,jcol) = y[*]
 
   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setColumnDV (
   DA2      *mtx,
   DV       *colDV,
   int      jcol
) ;
/*
   -----------------------
   return a(irow, jcol)
 
   created -- 96apr19, cca
   -----------------------
*/
double
DA2_entry (
   DA2      *mtx,
   int      irow,
   int      jcol
) ;
/*
   -----------------------
   set mtx(irow,jcol) = y
 
   created -- 96apr19, cca
   -----------------------
*/
void
DA2_setEntry (
   DA2      *mtx,
   int      irow,
   int      jcol,
   double   value
) ;
/*
   ---------------------------------------
   add and entry
 
   mtx(irow,jcol) = mtx(irow,jcol) + value
 
   created -- 96apr19, cca
   ---------------------------------------
*/
void
DA2_addEntry (
   DA2      *mtx,
   int      irow,
   int      jcol,
   double   value
) ;
/*
   ---------------------------------------------
   purpose -- test symmetry of the matrix
 
   if matrix is not square
      return zero
   else if |entry (i,j) - entry(j,i)| > tau then
      return zero
   else
      return 1
   endif
 
   created -- 96sep15, cca
   ---------------------------------------------
*/
int
DA2_isSymmetric (
   DA2      *mtx,
   double   tau
) ;
/*
   -------------------------------------------------------------
   fill the matrix with uniform random numbers in [lower, upper]
 
   created -- 96apr20, cca
   -------------------------------------------------------------
*/
void
DA2_fillRandomUniform (
   DA2      *a,
   double   lower,
   double   upper,
   int      seed
) ;
/*
   -----------------------------------------------
   fill the matrix with normal(0,1) random numbers
 
   created -- 96apr20, cca
   -----------------------------------------------
*/
void
DA2_fillRandomNormal (
   DA2      *a,
   double   mean,
   double   variance,
   int      seed
) ;
/*
   ----------------------------------------
   fill the matrix with the identity matrix
 
   created -- 96jun29, cca
   ----------------------------------------
*/
void
DA2_fillWithIdentity (
   DA2   *a
) ;
/*
   --------------------------
   fill the matrix with zeros
 
   created -- 96jun29, cca
   --------------------------
*/
void
DA2_zero (
   DA2   *a
) ;
/*
   ----------------------------
   copy one matrix into another
      A := B
 
   created  -- 96may03, cca
   modified -- 96aug24, cca
      nrow = min(A->n1, B->n1)
      ncol = min(A->n2, B->n2)
   ----------------------------
*/
void
DA2_copy (
   DA2   *A,
   DA2   *B
) ;
/*
   ---------------------------------------------------------------
   purpose --
      given a square array, copy the entries in the upper triangle
      to the lower triangle to make the matrix symmetric
 
   created -- 96sep15, cca
   ---------------------------------------------------------------
*/
void
DA2_copyUpperToLower (
   DA2   *mtx
) ;
/*
   ------------------------------------------------------------
   purpose -- copy entries to a vector. the portion copied
              can be a union of the strict lower portion,
              the diagonal portion, and the strict upper
              portion. there is one restriction, if the strict
              lower and strict upper are to be copied, the
              diagonal will also be copied.
 
   length    -- length of dvec[]
   dvec[]    -- vector to receive matrix entries
   copyflag  -- flag to denote what part of the entries to move
      1 --> move strict lower entries
      2 --> move lower entries (includes the diagonal)
      3 --> move diagonal entries
      4 --> move upper entries (includes the diagonal)
      5 --> move strict upper entries
      6 --> move all entries
   storeflag -- flag to denote how to store entries in dvec[]
      0 --> store by rows
      1 --> store by columns
 
   return value -- number of entries copied
 
   created -- 97jun03, cca, dkw
   ------------------------------------------------------------
*/
int
DA2_copyEntriesToVector (
   DA2      *mtx,
   int      length,
   double   *dvec,
   int      copyflag, 
   int      storeflag
) ;
/*
   --------------------------------
   subtract one matrix from another
 
   A := A - B
 
   created -- 96may03, cca
   ----------------------------
*/
void
DA2_sub (
   DA2   *A,
   DA2   *B
) ;
/*
   ---------------------------
   swap two rows of the matrix
 
   created -- 97mar08, cca
   ---------------------------
*/
void
DA2_swapRows (
   DA2   *a,
   int   irow1,
   int   irow2
) ;
/*
   ------------------------------
   swap two columns of the matrix
 
   created -- 97mar08, cca
   ------------------------------
*/
void
DA2_swapColumns (
   DA2   *a,
   int   jcol1,
   int   jcol2
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in QRfactor.c --------------------------------------
------------------------------------------------------------------------
*/
/*
   --------------------------------------------------------------
   purpose ---
      given a matrix A, convert to an upper triangular or
      upper trapezoidal R using householder transformations.
      the transformations are stored in the lower triangle
      or lower trapezoid of A, their coefficients are stored
      in the betaDV object
 
      this code follows closely page 210 from golub and van loan,
      "matrix computations", 3rd edition.
 
      note: presently the matrix must be column major
 
   created -- 97mar29, cca
   --------------------------------------------------------------
*/
void
DA2_QRfactor (
  DA2   *mtx,
  DV    *betaDV
) ;
/*
   -----------------------------------------------------------------
   apply a sequence of householder transformations to a matrix B
 
   B = \prod_{i=1}^{nstep} (I - beta*h*h^T) B
 
   the householder vectors are stored in in the lower triangle of H,
   the first is size nrow(H), the second has size nrow(H)-1, etc.
   the first nonzero component of each vector is 1.
   the beta coefficients are found in betaDV.
 
   created -- 97mar29, cca
   -----------------------------------------------------------------
*/
void
DA2_applyHouseholders (
   DA2   *H,
   DV    *betaDV,
   DA2   *B
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in drop.c ------------------------------------------
------------------------------------------------------------------------
*/
/*
   ------------------------------------------------------
   purpose -- to identify the rows to keep that preserves
      a fraction of the matrix's frobenius norm.
 
   frac    -- fraction of the frobenius norm to keep.
   flagsCV -- character vector object.
      flags[irow] = 'K' --> keep row irow
      flags[irow] = 'D' --> drop row irow
   tmpIV   -- work vector
   tmpDV   -- work vector
 
   created -- 97apr03, cca
   ------------------------------------------------------
*/
void
DA2_rowsToKeep (
   DA2      *mtx,
   double   frac,
   CV       *flagsCV,
   IV       *tmpIV,
   DV       *tmpDV
) ;
/*
   ---------------------------------------------------------
   purpose -- to identify the columns to keep that preserves
      a fraction of the matrix's frobenius norm.
 
   frac    -- fraction of the frobenius norm to keep.
   flagsCV -- character vector object.
      flags[jcol] = 'K' --> keep column jcol
      flags[jcol] = 'D' --> drop column jcol
   tmpIV   -- work vector
   tmpDV   -- work vector
 
   created -- 97apr03, cca
   ---------------------------------------------------------
*/
void
DA2_columnsToKeep (
   DA2      *mtx,
   double   frac,
   CV       *flagsCV,
   IV       *tmpIV,
   DV       *tmpDV
) ;
/*--------------------------------------------------------------------*/
/*
------------------------------------------------------------------------
----- methods found in IO.c --------------------------------------------
------------------------------------------------------------------------
*/
/*
   -----------------------------------------------------------
   purpose -- to read in the object from a file
 
   input --
 
      fn -- filename, must be *.da2b or *.da2f
 
   return value -- 1 if success, 0 if failure
   -----------------------------------------------------------
*/
int
DA2_readFromFile (
   DA2    *mtx,
   char   *fn
) ;
/*
   --------------------------------------------------
   purpose -- to read an object from a formatted file
 
   return value -- 1 if success, 0 if failure
   --------------------------------------------------
*/
int
DA2_readFromFormattedFile (
   DA2    *mtx,
   FILE   *fp
) ;
/*
   -----------------------------------------------
   purpose -- to read an object from a binary file
 
   return value -- 1 if success, 0  if failure
   -----------------------------------------------
*/
int
DA2_readFromBinaryFile (
   DA2    *mtx,
   FILE   *fp
) ;
/*
   ---------------------------------------------------------
   purpose -- to write an adjacency graph object to a file
 
   input --
 
      fn -- filename
        *.da2b -- binary
        *.da2f -- formatted
        anything else -- for human eye
   ---------------------------------------------------------
*/
void
DA2_writeToFile (
   DA2    *mtx,
   char   *fn
) ;
/*
   -------------------------------------------------
   purpose -- to write an object to a formatted file
   -------------------------------------------------
*/
void
DA2_writeToFormattedFile (
   DA2    *mtx,
   FILE   *fp
) ;
/*
   --------------------------------------------------------
   purpose -- to write an adjacency object to a binary file
   --------------------------------------------------------
*/
void
DA2_writeToBinaryFile (
   DA2    *mtx,
   FILE   *fp
) ;
/*
   ----------------------------------------------
   purpose -- to write the object for a human eye
   ----------------------------------------------
*/
void
DA2_writeForHumanEye (
   DA2    *mtx,
   FILE   *fp
) ;
/*
   --------------------------------------
   purpose -- to write out the statistics
   --------------------------------------
*/
void
DA2_writeStats (
   DA2    *mtx,
   FILE   *fp
) ;
/*====================================================================*/
