/*  IO.c  */

#include "../CV.h"

static const char *suffixb = ".cvb" ;
static const char *suffixf = ".cvf" ;

/*--------------------------------------------------------------------*/
/*
   ----------------------------------------------
   purpose -- to read in an CV object from a file

   input --

      fn -- filename, must be *.cvb or *.cvf

   return value -- 1 if success, 0 if failure

   created -- 97mar27, cca
   ----------------------------------------------
*/
int
CV_readFromFile ( 
   CV     *iv, 
   char   *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fn == NULL ) {
   fprintf(stderr, 
    "\n error in CV_readFromFile(%p,%s), file %s, line %d"
    "\n bad input\n", iv, fn, __FILE__, __LINE__) ;
   return(0) ;
}
/*
   -------------
   read the file
   -------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "rb")) == NULL ) {
         fprintf(stderr, "\n error in CV_readFromFile(%p,%s)"
                 "\n unable to open file %s", iv, fn, fn) ;
         rc = 0 ;
      } else {
         rc = CV_readFromBinaryFile(iv, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "r")) == NULL ) {
         fprintf(stderr, "\n error in CV_readFromFile(%p,%s)"
                 "\n unable to open file %s", iv, fn, fn) ;
         rc = 0 ;
      } else {
         rc = CV_readFromFormattedFile(iv, fp) ;
         fclose(fp) ;
      }
   } else {
      fprintf(stderr, "\n error in CV_readFromFile(%p,%s)"
              "\n bad CV file name %s,"
              "\n must end in %s (binary) or %s (formatted)\n",
              iv, fn, fn, suffixb, suffixf) ;
      rc = 0 ;
   }
} else {
   fprintf(stderr, "\n error in CV_readFromFile(%p,%s)"
       "\n bad CV file name %s,"
       "\n must end in %s (binary) or %s (formatted)\n",
       iv, fn, fn, suffixb, suffixf) ;
   rc = 0 ;
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   purpose -- to read an CV object from a formatted file

   return value -- 1 if success, 0 if failure

   created -- 97mar27, cca
   -----------------------------------------------------
*/
int
CV_readFromFormattedFile ( 
   CV     *iv, 
   FILE   *fp 
) {
int   rc, size ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in CV_readFromFormattedFile(%p,%p)"
           "\n bad input\n", iv, fp) ;
   return(0) ;
}
CV_clearData(iv) ;
/*
   ------------------------------
   read in the size of the vector
   ------------------------------
*/
if ( (rc = fscanf(fp, "%d", &size)) != 1 ) {
   fprintf(stderr, "\n error in CV_readFromFormattedFile(%p,%p)"
           "\n %d items of %d read\n", iv, fp, rc, 1) ;
   return(0) ;
}
/*
   ---------------------
   initialize the object
   ---------------------
*/
CV_init(iv, size, NULL) ;
iv->size = size ;
/*
   ------------------------
   read in the vec[] vector
   ------------------------
*/
if ( (rc = CVfscanf(fp, size, iv->vec)) != size ) {
   fprintf(stderr, "\n error in CV_readFromFormattedFile(%p,%p)"
           "\n %d items of %d read\n", iv, fp, rc, size) ;
   return(0) ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------
   purpose -- to read an CV object from a binary file

   return value -- 1 if success, 0  if failure

   created -- 97mar27, cca
   ---------------------------------------------------
*/
int
CV_readFromBinaryFile ( 
   CV     *iv, 
   FILE   *fp 
) {
int   rc, size ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in CV_readFromBinaryFile(%p,%p)"
           "\n bad input\n", iv, fp) ;
   return(0) ;
}
CV_clearData(iv) ;
/*
   ------------------------------
   read in the size of the vector
   ------------------------------
*/
if ( (rc = fread((void *) &size, sizeof(int), 1, fp)) != 1 ) {
   fprintf(stderr, "\n error in CV_readFromBinaryFile(%p,%p)"
           "\n itemp(3) : %d items of %d read\n", iv, fp, rc, 1) ;
   return(0) ;
}
/*
   ---------------------
   initialize the object
   ---------------------
*/
CV_init(iv, size, NULL) ;
iv->size = size ;
/*
   ------------------------
   read in the vec[] vector
   ------------------------
*/
if ( (rc = fread((void *) iv->vec, sizeof(char), size, fp)) != size ) {
   fprintf(stderr, "\n error in CV_readFromBinaryFile(%p,%p)"
           "\n sizes(%d) : %d items of %d read\n", 
           iv, fp, size, rc, size) ;
   return(0) ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------
   purpose -- to write an CV object to a file

   input --

      fn -- filename
        *.cvb -- binary
        *.cvf -- formatted
        anything else -- for human eye

   return value -- 1 if success, 0 otherwise

   created -- 97mar27, cca
   -------------------------------------------
*/
int
CV_writeToFile ( 
   CV     *iv, 
   char   *fn 
) {
FILE   *fp ;
int    fnlength, rc, sulength ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fn == NULL ) {
   fprintf(stderr, "\n fatal error in CV_writeToFile(%p,%s)"
    "\n bad input\n", iv, fn) ; 
}
/*
   ------------------
   write out the file
   ------------------
*/
fnlength = strlen(fn) ;
sulength = strlen(suffixb) ;
if ( fnlength > sulength ) {
   if ( strcmp(&fn[fnlength-sulength], suffixb) == 0 ) {
      if ( (fp = fopen(fn, "wb")) == NULL ) {
         fprintf(stderr, "\n error in CV_writeToFile(%p,%s)"
                 "\n unable to open file %s", iv, fn, fn) ;
         rc = 0 ;
      } else {
         rc = CV_writeToBinaryFile(iv, fp) ;
         fclose(fp) ;
      }
   } else if ( strcmp(&fn[fnlength-sulength], suffixf) == 0 ) {
      if ( (fp = fopen(fn, "w")) == NULL ) {
         fprintf(stderr, "\n error in CV_writeToFile(%p,%s)"
                 "\n unable to open file %s", iv, fn, fn) ;
         rc = 0 ;
      } else {
         rc = CV_writeToFormattedFile(iv, fp) ;
         fclose(fp) ;
      }
   } else {
      if ( (fp = fopen(fn, "a")) == NULL ) {
         fprintf(stderr, "\n error in CV_writeToFile(%p,%s)"
                 "\n unable to open file %s", iv, fn, fn) ;
         rc = 0 ;
      } else {
         rc = CV_writeForHumanEye(iv, fp) ;
         fclose(fp) ;
      }
   }
} else {
   if ( (fp = fopen(fn, "a")) == NULL ) {
      fprintf(stderr, "\n error in CV_writeToFile(%p,%s)"
              "\n unable to open file %s", iv, fn, fn) ;
      rc = 0 ;
   } else {
      rc = CV_writeForHumanEye(iv, fp) ;
      fclose(fp) ;
   }
}
return(rc) ; }

/*--------------------------------------------------------------------*/
/*
   -----------------------------------------------------
   purpose -- to write an CV object to a formatted file

   return value -- 1 if success, 0 otherwise

   created -- 97mar27, cca
   -----------------------------------------------------
*/
int
CV_writeToFormattedFile ( 
   CV     *iv, 
   FILE   *fp 
) {
int   ierr, rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fp == NULL || iv->size <= 0 ) {
   fprintf(stderr, "\n fatal error in CV_writeToFormattedFile(%p,%p)"
           "\n bad input\n", iv, fp) ;
   fprintf(stderr, "\n iv->size = %d", iv->size) ;
   exit(-1) ;
}
/*
   -------------------------------------
   write out the size of the vector
   -------------------------------------
*/
rc = fprintf(fp, "\n %d", iv->size) ;
if ( rc < 0 ) {
   fprintf(stderr, "\n fatal error in CV_writeToFormattedFile(%p,%p)"
           "\n rc = %d, return from first fprintf\n", iv, fp, rc) ;
   return(0) ;
}
if ( iv->size > 0 ) {
   CVfp80(fp, iv->size, iv->vec, 80, &ierr) ;
   if ( ierr < 0 ) {
      fprintf(stderr, 
              "\n fatal error in CV_writeToFormattedFile(%p,%p)"
              "\n ierr = %d, return from sizes[] CVfp80\n", 
              iv, fp, ierr) ;
      return(0) ;
   }
}

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   --------------------------------------------------
   purpose -- to write an CV object to a binary file

   return value -- 1 if success, 0 otherwise

   created -- 97mar27, cca
   --------------------------------------------------
*/
int
CV_writeToBinaryFile ( 
   CV     *iv, 
   FILE   *fp 
) {
int   rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fp == NULL || iv->size <= 0 ) {
   fprintf(stderr, "\n fatal error in CV_writeToBinaryFile(%p,%p)"
           "\n bad input\n", iv, fp) ;
   exit(-1) ;
}
rc = fwrite((void *) &iv->size, sizeof(int), 1, fp) ;
if ( rc != 1 ) {
   fprintf(stderr, "\n error in CV_writeToBinaryFile(%p,%p)"
           "\n %d of %d scalar items written\n", iv, fp, rc, 1) ;
   return(0) ;
}
rc = fwrite((void *) iv->vec, sizeof(char), iv->size, fp) ;
if ( rc != iv->size ) {
   fprintf(stderr, "\n error in CV_writeToBinaryFile(%p,%p)"
           "\n iv->sizes, %d of %d items written\n",
           iv, fp, rc, iv->size) ;
   return(0) ;
}
return(1) ; }

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------
   purpose -- to write an CV object for a human eye

   return value -- 1 if success, 0 otherwise

   created -- 97mar27, cca
   -------------------------------------------------
*/
int
CV_writeForHumanEye ( 
   CV     *iv, 
   FILE   *fp 
) {
int   ierr, rc ;

if ( iv == NULL || fp == NULL ) {
   fprintf(stderr, "\n fatal error in CV_writeForHumanEye(%p,%p)"
           "\n bad input\n", iv, fp) ;
   exit(-1) ;
}
if ( (rc = CV_writeStats(iv, fp)) == 0 ) {
   fprintf(stderr, "\n fatal error in CV_writeForHumanEye(%p,%p)"
           "\n rc = %d, return from CV_writeStats(%p,%p)\n",
           iv, fp, rc, iv, fp) ;
   return(0) ;
}
CVfp80(fp, iv->size, iv->vec, 80, &ierr) ;

return(1) ; }

/*--------------------------------------------------------------------*/
/*
   ---------------------------------------------------------
   purpose -- to write out the statistics for the CV object

   return value -- 1 if success, 0 otherwise

   created -- 97mar27, cca
   ---------------------------------------------------------
*/
int
CV_writeStats ( 
   CV     *iv, 
   FILE   *fp 
) {
int   rc ;
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fp == NULL ) {
   fprintf(stderr, "\n error in CV_writeStats(%p,%p)"
           "\n bad input\n", iv, fp) ;
   exit(-1) ;
}
rc = fprintf(fp, "\n CV : integer vector object : ") ;
if ( rc < 0 ) { goto IO_error ; }
rc = fprintf(fp, " size = %d, maxsize = %d, owned = %d", 
             iv->size, iv->maxsize, iv->owned) ;
if ( rc < 0 ) { goto IO_error ; }
return(1) ;

IO_error :
   fprintf(stderr, "\n fatal error in CV_writeStats(%p,%p)"
           "\n rc = %d, return from fprintf\n", iv, fp, rc) ;
   return(0) ;
}

/*--------------------------------------------------------------------*/
/*
   -------------------------------------------------------------------
  purpose -- to write out an integer vector with eighty column lines
 
   input --
 
      fp     -- file pointer, must be formatted and write access
      column -- present column
      pierr  -- pointer to int to hold return value, 
                should be 1 if any print was successful,
                if fprintf() failed, then ierr = -1
  
   return value -- present column
 
   created -- 97mar27, cca
   -------------------------------------------------------------------
*/
int
CV_fp80 (
   CV     *iv,
   FILE   *fp,
   int    column,
   int    *pierr
) {
/*
   ---------------
   check the input
   ---------------
*/
if ( iv == NULL || fp == NULL || pierr == NULL ) {
   fprintf(stderr, "\n fatal error in CV_fp80(%p,%p,%p)"
           "\n bad input\n", iv, fp, pierr) ;
   exit(-1) ;
}
if ( iv->size > 0 && iv->vec != NULL ) {
   column = CVfp80(fp, iv->size, iv->vec, column, pierr) ;
}

return(column) ; }

/*--------------------------------------------------------------------*/
