!
!  READ_MAT subroutine reads a matrix and its right hand sides,
!  all stored in Matrix Market format file. The B field is optional,.
!
!  Character                            :: filename*20
!     On Entry: name of file to be processed.
!     On Exit : unchanged.
!
!  Type(D_SPMAT)                        :: A
!     On Entry: fresh variable.
!     On Exit : will contain the global sparse matrix as follows:
!        A%AS for coefficient values
!        A%IA1  for column indices
!        A%IA2  for row pointers
!        A%M    for number of global matrix rows
!        A%K    for number of global matrix columns
!
!  Integer                              :: ICTXT
!     On Entry: BLACS context.
!     On Exit : unchanged.
!
!  Real(Kind(1.D0)), Pointer, Optional  :: B(:,:)
!     On Entry: fresh variable.
!     On Exit:  will contain right hand side(s).
!
!  Integer, Optional                    :: inroot
!     On Entry: Index of root processor (default: 0)
!     On Exit : unchanged.
!
!  Real(Kind(1.D0)), Pointer, Optional  :: indwork(:)
!     On Entry/Exit: Double Precision Work Area.
!
!  Integer, Pointer, Optional           :: iniwork()
!     On Entry/Exit: Integer Work Area.
!
MODULE READ_MAT
  PUBLIC READMAT
  PUBLIC READ_RHS
  PUBLIC ZREADMAT
  PUBLIC ZREAD_RHS


CONTAINS
  SUBROUTINE READMAT (FILENAME, A, ICTXT, INROOT,&
       & INDWORK, INIWORK)   
    USE TYPESP
    IMPLICIT NONE
    INTEGER                               :: ICTXT
    TYPE(D_SPMAT)                         :: A
    CHARACTER                             :: FILENAME*(*)
    INTEGER, OPTIONAL                     :: INROOT
    REAL(KIND(1.0D0)), POINTER, OPTIONAL  :: INDWORK(:)
    INTEGER, POINTER, OPTIONAL            :: INIWORK(:)   ! Local Variables
    INTEGER, PARAMETER          :: INFILE = 2
    CHARACTER                   :: MMHEADER*15, FMT*15, OBJECT*10, TYPE*10, SYM*15,&
         & LINE*1024
    INTEGER                     :: INDCRD,  PTRCRD, TOTCRD,&
         & VALCRD, RHSCRD, NROW, NCOL, NNZERO, NELTVL, NRHS, NRHSIX
    REAL(KIND(1.0D0)), POINTER  :: AS_LOC(:), DWORK(:)
    INTEGER, POINTER            :: IA1_LOC(:), IA2_LOC(:), IWORK(:), TMP(:), AUX(:)
    INTEGER                     :: D_ALLOC, I_ALLOC, IRCODE, I,IEL,PTR,NZR,&
         & J, LIWORK, LDWORK, ROOT, NPROW, NPCOL, MYPROW, MYPCOL
    IF (PRESENT(INROOT)) THEN
      ROOT = INROOT
    ELSE
      ROOT = 0
    END IF
    CALL BLACS_GRIDINFO(ICTXT, NPROW, NPCOL, MYPROW, MYPCOL)    
    IF (MYPROW == ROOT) THEN
      WRITE(*, *) 'Start read_matrix'      ! Open Input File
      OPEN(INFILE,FILE=FILENAME, STATUS='OLD', ERR=901, ACTION="READ")
      READ(INFILE,FMT=*, END=902) MMHEADER, OBJECT, FMT, TYPE, SYM
      CALL LOWERC(OBJECT,1,10)
      CALL LOWERC(FMT,1,15)
      IF ( (OBJECT .NE. 'matrix').OR.(FMT.NE.'coordinate')) THEN
        WRITE(0,*) 'READ_MATRIX: input file type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF

      READ(INFILE,FMT='(1a)') LINE
      DO WHILE(LINE(1:1).EQ.'%') 
         READ(INFILE,FMT='(1a)') LINE
      END DO
      BACKSPACE(INFILE)

      READ(INFILE,FMT=*) NROW,NCOL,NNZERO

      A%M    = NROW
      A%K    = NCOL
      A%FIDA = 'CSR'
      CALL LOWERC(TYPE,1,10)
      CALL LOWERC(SYM,1,15)
      IF ((TYPE == 'real').AND.(SYM == 'general')) THEN
        ALLOCATE(A%ASPK(NNZERO), A%IA1(NNZERO), A%IA2(NROW+1),&
             &A%PL(NROW),A%PR(NROW), TMP(NNZERO), AUX(NNZERO),STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
        DO I=1,NNZERO
           READ(INFILE,FMT=*,END=902) TMP(I),A%IA1(I),A%ASPK(I)
        END DO

        !     .... Order with key TMP (ROW INDEX) ...
        CALL MRGSRT(NNZERO,TMP,AUX,IRCODE)
        IF (IRCODE.EQ.0) CALL REORDVN(NNZERO,A%ASPK,TMP,A%IA1,AUX)
        !     .... Order with key A%IA1 (COLUMN INDEX) ...
        
        I    = 1
        J    = I
        DO WHILE (I.LE.(NNZERO))
           DO WHILE ((TMP(J).EQ.TMP(I)).AND.(J.LE.NNZERO))
              J = J+1
           ENDDO
           IEL = J - I
           CALL MRGSRT(IEL,A%IA1(I),AUX,IRCODE)
           IF (IRCODE.EQ.0) CALL REORDVN(IEL,A%ASPK(I),TMP(I),&
         &   A%IA1(I), AUX)
           I = J
        ENDDO

   ! Convert to CSR format     
        IEL = 1
        A%IA2(1) = 1
        DO I = A%IA2(1), NROW
           
           DO WHILE (TMP(IEL).EQ.I)
              IEL = IEL + 1
           ENDDO
           A%IA2(I+1) = IEL
        ENDDO
        DEALLOCATE(AUX,TMP)
      ELSE IF ((TYPE == 'real').AND.(SYM == 'symmetric')) THEN
        ! We are generally working with non-symmetric matrices, so
        ! we de-symmetrize what we are about to read
        ALLOCATE(A%ASPK(2*NNZERO),A%IA1(2*NNZERO),&
             & A%IA2(2*NNZERO),AS_LOC(2*NNZERO),&
             & IA1_LOC(2*NNZERO),IA2_LOC(2*NNZERO),&
             &A%PL(NROW),A%PR(NROW), STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
        DO I=1,NNZERO
           READ(INFILE,FMT=*,END=902) A%IA1(I),A%IA2(I),A%ASPK(I)
        END DO

        LDWORK = 2 * NNZERO
        IF (PRESENT(INDWORK)) THEN
          IF (SIZE(INDWORK) >= LDWORK) THEN
            DWORK => INDWORK
            D_ALLOC = 0
          ELSE
            ALLOCATE(DWORK(LDWORK), STAT = IRCODE)
            D_ALLOC = 1
          END IF
        ELSE
          ALLOCATE(DWORK(LDWORK), STAT = IRCODE)
          D_ALLOC = 1
        END IF
        IF (IRCODE /= 0)   GOTO 993     
        LIWORK = 2*NNZERO
        IF (PRESENT(INIWORK)) THEN
          IF (SIZE(INIWORK) >= LIWORK) THEN
            IWORK => INIWORK
            I_ALLOC = 0
          ELSE
            ALLOCATE(IWORK(LIWORK), STAT = IRCODE)
            I_ALLOC = 1
          END IF
        ELSE
          ALLOCATE(IWORK(LIWORK), STAT = IRCODE)
          I_ALLOC = 1
        END IF
        IF (IRCODE /= 0)   GOTO 993  
        ! After this call NNZERO contains the actual value for
        ! desymetrized matrix
        CALL DESYM(NROW, A%ASPK, A%IA2, A%IA1, AS_LOC, IA2_LOC,&
             & IA1_LOC, IWORK, DWORK, NNZERO, PTR, NZR, 1)     

        DEALLOCATE(A%ASPK,A%IA1,A%IA2)
        NNZERO=NZR
        ALLOCATE(A%ASPK(NNZERO),A%IA1(NNZERO),&
             & A%IA2(NROW+1), TMP(NNZERO), STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
       
        A%ASPK(1:nnzero) = AS_LOC(PTR:PTR+NNZERO-1)
        A%IA1(1:nnzero)  = IA2_LOC(PTR:PTR+NNZERO-1)
        TMP(1:nnzero)    = IA1_LOC(PTR:PTR+NNZERO-1)


        IEL = 1
        A%IA2(1) = 1
        DO I = A%IA2(1), NROW
           
           DO WHILE (TMP(IEL).EQ.I)
              IEL = IEL + 1
           ENDDO
           A%IA2(I+1) = IEL
        ENDDO

        DEALLOCATE(AS_LOC, IA1_LOC, IA2_LOC,TMP)
        IF (D_ALLOC == 1)   DEALLOCATE(DWORK)
        IF (I_ALLOC == 1)   DEALLOCATE(IWORK)
      ELSE
        WRITE(0,*) 'READ_MATRIX: matrix type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF 
      WRITE(*,*) 'End READ_MATRIX'
    END IF 
    RETURN 
    ! Open failed
901 WRITE(0,*) 'READ_MATRIX: Could not open file ',&
         & INFILE,' for input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Unexpected End of File
902 WRITE(0,*) 'READ_MATRIX: Unexpected end of file ',INFILE,&
         & ' during input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Allocation Failed
993 WRITE(0,*) 'READ_MATRIX: Memory allocation failure'
    CALL BLACS_ABORT(ICTXT, 1)
  END SUBROUTINE READMAT



  SUBROUTINE ZREADMAT (FILENAME, A, ICTXT, INROOT,&
       & INDWORK, INIWORK)   
    USE TYPESP
    IMPLICIT NONE
    INTEGER                               :: ICTXT
    TYPE(Z_SPMAT)                         :: A
    CHARACTER                             :: FILENAME*(*)
    INTEGER, OPTIONAL                     :: INROOT
    COMPLEX(KIND(1.0D0)), POINTER, OPTIONAL  :: INDWORK(:)
    INTEGER, POINTER, OPTIONAL            :: INIWORK(:)   ! Local Variables
    INTEGER, PARAMETER          :: INFILE = 2
    CHARACTER                   :: MMHEADER*15, FMT*15, OBJECT*10, TYPE*10, SYM*15,&
         & LINE*1024
    INTEGER                     :: INDCRD,  PTRCRD, TOTCRD,&
         & VALCRD, RHSCRD, NROW, NCOL, NNZERO, NELTVL, NRHS, NRHSIX
    COMPLEX(KIND(1.0D0)), POINTER  :: AS_LOC(:), DWORK(:)
    INTEGER, POINTER            :: IA1_LOC(:), IA2_LOC(:), IWORK(:), TMP(:), AUX(:)
    INTEGER                     :: D_ALLOC, I_ALLOC, IRCODE, I,IEL,&
         & J, LIWORK, LDWORK, ROOT, NPROW, NPCOL, MYPROW, MYPCOL,PTR,NZR
    REAL(KIND(1.D0))            :: RE,IM

    IF (PRESENT(INROOT)) THEN
      ROOT = INROOT
    ELSE
      ROOT = 0
    END IF
    CALL BLACS_GRIDINFO(ICTXT, NPROW, NPCOL, MYPROW, MYPCOL)    
    IF (MYPROW == ROOT) THEN
      WRITE(*, *) 'Start read_matrix'      ! Open Input File
      OPEN(INFILE,FILE=FILENAME, STATUS='OLD', ERR=901, ACTION="READ")
      READ(INFILE,FMT=*, END=902) MMHEADER, OBJECT, FMT, TYPE, SYM
      CALL LOWERC(OBJECT,1,10)
      CALL LOWERC(FMT,1,15)
      IF ( (OBJECT .NE. 'matrix').OR.(FMT.NE.'coordinate')) THEN
        WRITE(0,*) 'READ_MATRIX: input file type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF

      READ(INFILE,FMT='(1a)') LINE
      DO WHILE(LINE(1:1).EQ.'%') 
         READ(INFILE,FMT='(1a)') LINE
      END DO
      BACKSPACE(INFILE)

      READ(INFILE,FMT=*)NROW,NCOL,NNZERO

      A%M    = NROW
      A%K    = NCOL
      A%FIDA = 'CSR'
      CALL LOWERC(TYPE,1,10)
      CALL LOWERC(SYM,1,15)
      IF ((TYPE == 'complex').AND.(SYM == 'general')) THEN
        ALLOCATE(A%ASPK(NNZERO), A%IA1(NNZERO), A%IA2(NROW+1),&
             &A%PL(NROW),A%PR(NROW), TMP(NNZERO), AUX(NNZERO),STAT = IRCODE)
        a%aspk(:)=0
        a%ia1(:)=0
        a%ia2(:)=0
        IF (IRCODE /= 0)   GOTO 993
        DO I=1,NNZERO
           READ(INFILE,FMT=*,END=902) TMP(I),A%IA1(I),RE,IM
           A%ASPK(I)=CMPLX(RE,IM)
        END DO

        !     .... Order with key TMP (ROW INDEX) ...
        CALL MRGSRT(NNZERO,TMP,AUX,IRCODE)
        IF (IRCODE.EQ.0) CALL ZREORDVN(NNZERO,A%ASPK,TMP,A%IA1,AUX)
        !     .... Order with key A%IA1 (COLUMN INDEX) ...
        
        I    = 1
        J    = I
        DO WHILE (I.LE.(NNZERO))
           DO WHILE ((TMP(J).EQ.TMP(I)).AND.(J.LE.NNZERO))
              J = J+1
           ENDDO
           IEL = J - I
           CALL MRGSRT(IEL,A%IA1(I),AUX,IRCODE)
           IF (IRCODE.EQ.0) CALL ZREORDVN(IEL,A%ASPK(I),TMP(I),&
         &   A%IA1(I), AUX)
           I = J
        ENDDO

   ! Convert to CSR format     
        IEL = 1
        A%IA2(1) = 1
        DO I = A%IA2(1), NROW
           
           DO WHILE (TMP(IEL).EQ.I)
              IEL = IEL + 1
           ENDDO
           A%IA2(I+1) = IEL
        ENDDO
        DEALLOCATE(AUX,TMP)
      ELSE IF ((TYPE == 'complex').AND.(SYM == 'symmetric')) THEN
        ! We are generally working with non-symmetric matrices, so
        ! we de-symmetrize what we are about to read
        ALLOCATE(A%ASPK(2*NNZERO),A%IA1(2*NNZERO),&
             & A%IA2(2*NNZERO),AS_LOC(2*NNZERO),&
             & IA1_LOC(2*NNZERO),IA2_LOC(2*NNZERO),&
             &A%PL(NROW),A%PR(NROW), STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
        DO I=1,NNZERO
            READ(INFILE,FMT=*,END=902) A%IA1(I),A%IA2(I),RE,IM 
           A%ASPK(I) = CMPLX(RE,IM)
        END DO

        LDWORK = 2 * NNZERO
        IF (PRESENT(INDWORK)) THEN
          IF (SIZE(INDWORK) >= LDWORK) THEN
            DWORK => INDWORK
            D_ALLOC = 0
          ELSE
            ALLOCATE(DWORK(LDWORK), STAT = IRCODE)
            D_ALLOC = 1
          END IF
        ELSE
          ALLOCATE(DWORK(LDWORK), STAT = IRCODE)
          D_ALLOC = 1
        END IF
        IF (IRCODE /= 0)   GOTO 993     
        LIWORK = 2*NNZERO
        IF (PRESENT(INIWORK)) THEN
          IF (SIZE(INIWORK) >= LIWORK) THEN
            IWORK => INIWORK
            I_ALLOC = 0
          ELSE
            ALLOCATE(IWORK(LIWORK), STAT = IRCODE)
            I_ALLOC = 1
          END IF
        ELSE
          ALLOCATE(IWORK(LIWORK), STAT = IRCODE)
          I_ALLOC = 1
        END IF
        IF (IRCODE /= 0)   GOTO 993  
        ! After this call NNZERO contains the actual value for
        ! desymetrized matrix
        CALL ZDESYM(NROW, A%ASPK, A%IA2, A%IA1, AS_LOC, IA2_LOC,&
             & IA1_LOC, IWORK, DWORK, NNZERO, PTR, NZR,1)     
        DEALLOCATE(A%ASPK,A%IA1,A%IA2)
        NNZERO=NZR
        ALLOCATE(A%ASPK(NNZERO),A%IA1(NNZERO),&
             & A%IA2(nnzero), TMP(NNZERO), STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
        a%aspk(:)=0
        a%ia1(:)=0
        a%ia2(:)=0

        A%ASPK(1:nnzero) = AS_LOC(PTR:PTR+NNZERO-1)
        A%IA1(1:nnzero)  = IA2_LOC(PTR:PTR+NNZERO-1)
        TMP(1:nnzero)    = IA1_LOC(PTR:PTR+NNZERO-1)

        IEL = 1
        A%IA2(1) = 1
        DO I = A%IA2(1), NROW
           
           DO WHILE (TMP(IEL).EQ.I)
              IEL = IEL + 1
           ENDDO
           A%IA2(I+1) = IEL
        ENDDO

        DEALLOCATE(AS_LOC, IA1_LOC, IA2_LOC,TMP)
        IF (D_ALLOC == 1)   DEALLOCATE(DWORK)
        IF (I_ALLOC == 1)   DEALLOCATE(IWORK)
      ELSE
        WRITE(0,*) 'READ_MATRIX: matrix type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF 
      WRITE(*,*) 'End READ_MATRIX'
    END IF 
    RETURN 
    ! Open failed
901 WRITE(0,*) 'READ_MATRIX: Could not open file ',&
         & INFILE,' for input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Unexpected End of File
902 WRITE(0,*) 'READ_MATRIX: Unexpected end of file ',INFILE,&
         & ' during input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Allocation Failed
993 WRITE(0,*) 'READ_MATRIX: Memory allocation failure'
    CALL BLACS_ABORT(ICTXT, 1)
  END SUBROUTINE ZREADMAT



  SUBROUTINE READ_RHS (FILENAME, B, ICTXT, INROOT,&
       & INDWORK, INIWORK)   
    IMPLICIT NONE
    INTEGER                               :: ICTXT
    CHARACTER                             :: FILENAME*(*)
    INTEGER, OPTIONAL                     :: INROOT
    REAL(KIND(1.0D0)), POINTER, OPTIONAL  :: INDWORK(:)
    INTEGER, POINTER, OPTIONAL            :: INIWORK(:)   ! Local Variables
    INTEGER, PARAMETER          :: INFILE = 2
    INTEGER                     :: NROW, NCOL, I,ROOT, NPROW, NPCOL, MYPROW, MYPCOL, IRCODE, J
    CHARACTER                   :: MMHEADER*15, FMT*15, OBJECT*10, TYPE*10, SYM*15,&
         & LINE*1024
    REAL(KIND(1.0D0)), POINTER  :: B(:,:)
    IF (PRESENT(INROOT)) THEN
      ROOT = INROOT
    ELSE
      ROOT = 0
    END IF
    CALL BLACS_GRIDINFO(ICTXT, NPROW, NPCOL, MYPROW, MYPCOL)    
    IF (MYPROW == ROOT) THEN
      WRITE(*, *) 'Start read_rhs'      ! Open Input File
      OPEN(INFILE,FILE=FILENAME, STATUS='OLD', ERR=901, ACTION="READ")
      READ(INFILE,FMT=*, END=902) MMHEADER, OBJECT, FMT, TYPE, SYM
      write(0,*)'obj fmt',object, fmt
      IF ( (OBJECT .NE. 'matrix').OR.(FMT.NE.'array')) THEN
        WRITE(0,*) 'READ_RHS: input file type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF

      READ(INFILE,FMT='(1a)') LINE
      DO WHILE(LINE(1:1).EQ.'%') 
         READ(INFILE,FMT='(1a)') LINE
      END DO
      BACKSPACE(INFILE)

      READ(INFILE,FMT=*)NROW,NCOL

      CALL LOWERC(TYPE,1,10)
      CALL LOWERC(SYM,1,15)
      IF ((TYPE == 'real').AND.(SYM == 'general')) THEN
        ALLOCATE(B(NROW,NCOL),STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
        do i=1, nrow
          READ(INFILE,FMT=*,END=902) ,(B(I,J),J=1,NCOL)
        enddo
      ELSE
        WRITE(0,*) 'READ_RHS: RHS type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF      ! Read Right Hand Sides
      WRITE(*,*) 'End READ_RHS'
    END IF 
    RETURN 
    ! Open failed
901 WRITE(0,*) 'READ_RHS: Could not open file ',&
         & INFILE,' for input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Unexpected End of File
902 WRITE(0,*) 'READ_RHS: Unexpected end of file ',INFILE,&
         & ' during input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Allocation Failed
993 WRITE(0,*) 'READ_RHS: Memory allocation failure'
    CALL BLACS_ABORT(ICTXT, 1)
  END SUBROUTINE READ_RHS


  SUBROUTINE ZREAD_RHS(FILENAME, B, ICTXT, INROOT)
    IMPLICIT NONE
    INTEGER                               :: ICTXT
    CHARACTER                             :: FILENAME*(*)
    INTEGER, OPTIONAL                     :: INROOT
    INTEGER, PARAMETER          :: INFILE = 2
    INTEGER                     :: NROW, NCOL, I,ROOT, NPROW, NPCOL, MYPROW, MYPCOL, IRCODE, J
    CHARACTER                   :: MMHEADER*15, FMT*15, OBJECT*10, TYPE*10, SYM*15,&
         & LINE*1024
    COMPLEX(KIND(1.0D0)), POINTER  :: B(:,:)
    IF (PRESENT(INROOT)) THEN
      ROOT = INROOT
    ELSE
      ROOT = 0
    END IF
    CALL BLACS_GRIDINFO(ICTXT, NPROW, NPCOL, MYPROW, MYPCOL)    
    IF (MYPROW == ROOT) THEN
      WRITE(*, *) 'Start read_rhs'      ! Open Input File
      OPEN(INFILE,FILE=FILENAME, STATUS='OLD', ERR=901, ACTION="READ")
      READ(INFILE,FMT=*, END=902) MMHEADER, OBJECT, FMT, TYPE, SYM
      write(0,*)'obj fmt',object, fmt
      IF ( (OBJECT .NE. 'matrix').OR.(FMT.NE.'array')) THEN
        WRITE(0,*) 'READ_RHS: input file type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF

      READ(INFILE,FMT='(1a)') LINE
      DO WHILE(LINE(1:1).EQ.'%') 
         READ(INFILE,FMT='(1a)') LINE
      END DO
      BACKSPACE(INFILE)

      READ(INFILE,FMT=*)NROW,NCOL

      CALL LOWERC(TYPE,1,10)
      CALL LOWERC(SYM,1,15)
      IF ((TYPE == 'complex').AND.(SYM == 'general')) THEN
        ALLOCATE(B(NROW,NCOL),STAT = IRCODE)
        IF (IRCODE /= 0)   GOTO 993
        READ(INFILE,FMT=*,END=902) ((B(I,J), I=1,NROW),J=1,NCOL)
           
      ELSE
        WRITE(0,*) 'READ_RHS: RHS type not yet supported'
        CALL BLACS_ABORT(ICTXT, 1)
      END IF      ! Read Right Hand Sides
      WRITE(*,*) 'End READ_RHS'
    END IF 
    RETURN 
    ! Open failed
901 WRITE(0,*) 'READ_RHS: Could not open file ',&
         & INFILE,' for input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Unexpected End of File
902 WRITE(0,*) 'READ_RHS: Unexpected end of file ',INFILE,&
         & ' during input'
    CALL BLACS_ABORT(ICTXT, 1)   ! Allocation Failed
993 WRITE(0,*) 'READ_RHS: Memory allocation failure'
    CALL BLACS_ABORT(ICTXT, 1)
  END SUBROUTINE ZREAD_RHS



END MODULE READ_MAT
