/* ---------------------------------------------------------------------
*
*  -- PSBLAS routine (version 1.0) --
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "psblas.h"

void pszaxpby_( m, n, alpha, X, ix, jx, lldx, beta, Y, iy, jy, lldy,
	       desc_data, desc_halo, desc_ovrlap, desc_ovrlap_elem)
     /*
      *  .. Scalar Arguments ..
      */
     int      * ix, * iy, * jx, * jy, * n, * m, * lldx, * lldy;
     double   * alpha, * beta;
     /* ..
      *  .. Array Arguments ..
      */
     int      desc_data[], desc_halo[], desc_ovrlap[], desc_ovrlap_elem[] ;
     double   X[], Y[];
{
  /*
   *  Purpose
   *  =======
   *
   *  PSZAXPBY adds one distributed matrix to another,
   *
   *     sub( Y ) := beta * sub( Y ) + alpha * sub( X )
   *
   *  where sub( X ) denotes X(IX:IX+M-1,JX:JX+N-1)
   *
   *        sub( Y ) denotes Y(IY:IY+M-1,JY:JY+N-1).
   *
   *  Notes
   *  =====
   *
   *  Some description vectors are associated with each distributed sparse
   *  matrix. These vectors stores the information required to the
   *  communication needed to perform distributed operations.
   *
   *  They are:
   *
   *  NAME		 EXPLANATION
   *  ------------------ -------------------------------------------------------
   *  MATRIX_DATA	 Array of INTEGER that contains some local and global
   *			 information of matrix.
   *
   *  DESC_HALO          Array of INTEGER that contains informations for local 
   *                     halo points.
   *
   *  DESC_OVRLAP        Array of INTEGER that contains a list of local overlap 
   *                     elements.
   *                     
   *  OVRLAP_ELEM        Array of INTEGER that contains a list of local overlap
   *                     elements with their multiplicity.
   *
   *  Now we explain each of the above vectors.
   *
   *  Let A be a generic sparse matrix. We denote with MATDATA_A the MATRIX_DATA
   *  array for matrix A.
   *  Data stored in MATRIX_DATA array are:
   *
   *  NOTATION        STORED IN		     EXPLANATION
   *  --------------- ---------------------- -------------------------------------
   *  DEC_TYPE        MATDATA_A[DEC_TYPE_]   Decomposition type
   *  M 	      MATDATA_A[M_]          Total number of equations
   *  N 	      MATDATA_A[N_]          Total number of variables
   *  N_ROW           MATDATA_A[N_ROW_]      Number of local equations
   *  N_COL           MATDATA_A[N_COL_]      Number of local variables
   *  CTXT_A          MATDATA_A[CTXT_]       The BLACS context handle, indicating
   *					     the global context of the operation
   *					     on the matrix.
   *					     The context itself is global.
   *
   *  Let DESCHALO_P be the array DESC_HALO for local process.
   *  This is composed of variable dimension blocks for each process to 
   *  communicate to.
   *  Each block contain indexes of local halo elements to exchange with other 
   *  process.
   *  Let P be the pointer to the first element of a block in DESCHALO_P.
   *  This block is stored in DESCHALO_P as :
   *
   *  NOTATION        STORED IN		          EXPLANATION
   *  --------------- --------------------------- -----------------------------------
   *  PROCESS_ID      DESCHALO_P[P+PROC_ID_]      Identifier of process which exchange 
   *						  data with.
   *  N_ELEMENTS_RECV DESCHALO_P[P+N_ELEM_RECV_]  Number of elements to receive.
   *  ELEMENTS_RECV   DESCHALO_P[P+ELEM_RECV_+i]  Indexes of local elements to
   *					          receive. These are stored in the
   *					          array from location P+ELEM_RECV_ to
   *					          location P+ELEM_RECV_+
   *						  DESCHALO_P[P+N_ELEM_RECV_]-1.
   *  N_ELEMENTS_SEND DESCHALO_P[P+N_ELEM_SEND_]  Number of elements to send.
   *  ELEMENTS_SEND   DESCHALO_P[P+ELEM_SEND_+i]  Indexes of local elements to
   *					          send. These are stored in the
   *					          array from location P+ELEM_SEND_ to
   *					          location P+ELEM_SEND_+
   *						  DESCHALO_P[P+N_ELEM_SEND_]-1.
   *
   *
   *  Let DESCOVRLP_P be the array DESC_OVRLAP for local process.
   *  This is composed of variable dimension blocks for each process to 
   *  communicate to.
   *  Each block contain indexes of local overlap elements to exchange with
   *  other process.
   *  Let P be the pointer to the first element of a block in DESCOVRLP_P.
   *  This block is stored in DESCOVRLP_P as :
   *
   *  NOTATION        STORED IN		            EXPLANATION
   *  ------------- ------------------------------- -----------------------------------
   *  PROCESS_ID    DESCOVRLP_P[P+PROC_ID_]         Identifier of process which exchange
   *						    data with.
   *  N_OVRLAP_ELEM DESCOVRLP_P[P+N_OVRLP_ELEM_]    Number of elements to exchange.
   *  OVRLAP_ELEM   DESCOVRLP_P[P+OVRLP_ELEM_TO_+i] Indexes of local elements to
   *					            exchange. These are stored in the
   *					            array from location P+OVRLP_ELEM_ to
   *					            location P+OVRLP_ELEM_+
   *						    DESCOVRLP_P[P+N_OVRLP_ELEM_]-1.
   *
   *
   *  Let OVR_ELEM_P be the array OVERLAP_ELEM for local process.
   *  This is composed of blocks of two elements. The block
   *  corresponding to the i-th overlapped elements, begin at index 
   *  P = i*2 in array OVR_ELEM_P.
   *  This block is stored in OVR_ELEM_P as :
   *
   *  NOTATION      STORED IN		       EXPLANATION
   *  ------------- -------------------------- ----------------------------------
   *  OVRLAP_ELEM   OVR_ELEM_P[P+OVRLP_ELEM_]  The index of local overlapped 
   *					       element.
   *  N_POMAINS     OVR_ELEM_P[P+N_DOM_OVR_]   The number of copies of
   *					       local overlapped element.
   *
   *
   *  Parameters
   *  ==========
   *
   *  M       (global input) pointer to INTEGER.
   *          Number of rows of the distributed dense matrix to be added. M >= 0.
   *
   *  N       (global input) pointer to INTEGER.
   *          The number of columns of the distributed submatrix to
   *          be added.
   *
   *  ALPHA   (global input) pointer to REAL
   *          The scalar used to multiply each component of sub( X ).
   *
   *  X       (local input) REAL array containing the local pieces of 
   *          a distributed matrix.
   *          This array contains the entries of the distributed vector
   *          sub( X ).
   *
   *  IX      (global input) pointer to INTEGER
   *          The global row index of the submatrix of the distributed
   *          matrix X to operate on.
   *
   *  JX      (global input) pointer to INTEGER
   *          The global column index of the submatrix of the distributed
   *          matrix X to operate on.
   *
   *  LLDX    (local input) pointer to INTEGER
   *          The leading dimension of local dense matrix X.
   *
   *  BETA    (global input) pointer to REAL.
   *          The scalar used to multiply sub(Y).
   *
   *  Y       (local input/local output) REAL array
   *          containing the local pieces of a distributed matrix.
   *          This array contains the entries of the distributed vector
   *          sub( Y ).
   *          On exit sub( Y ) is overwritten by beta*sub( Y ) + alpha*sub( X ).
   *
   *  IY      (global input) pointer to INTEGER
   *          The global row index of the submatrix of the distributed
   *          matrix Y to operate on.
   *
   *  JY      (global input) pointer to INTEGER
   *          The global column index of the submatrix of the distributed
   *          matrix Y to operate on.
   *
   *  LLDY    (local input) pointer to INTEGER
   *          The leading dimension of local dense matrix Y.
   *
   *  DESCDATA (global and local input) INTEGER array. Is the MATRIX_DATA
   *           array.
   *
   *  DESCHALO (local input) INTEGER array. Is the DESC_HALO array.
   *
   *  DESCOVRLAP (local input) INTEGER array. Is the DESC_OVRLAP array.
   *
   *  DESCOVRLAPELEM (local input) INTEGER array. Is the OVRLAP_ELEM array.
   *
   *  =====================================================================
   *
   *  .. Local Scalars ..
   */
  int ictxt, iix, iiy, jjx, jjy, 
  mycol, myrow, nprow, npcol;
  double *tmpx, *tmpy;

  /* Error variables */
  int         info=0, err, ierror;
  int         int_err[5];
  double      real_err[5];
  /* ..
   *  .. External Functions ..
   */
  void        blacs_gridinfo();
  void        pbchkvect();
  void        pberror_();
  void        zaxpby_(int *m, int *n, double *alpha, double *X, int *lldx, 
		      double *beta, double *Y, int *lldy);

  /* ..
   *  .. Executable Statements ..
   *
   *  Get grid parameters
   */
  ictxt = desc_data[CTXT_];
  Cblacs_gridinfo( ictxt, &nprow, &npcol, &myrow, &mycol );
  /*
   *  Test the input parameters
   */
  info = 0;
  if( nprow == -1 ) {
    info = 2010;
  } else {
    if ((*lldx != desc_data[N_COL_])||(*lldy != desc_data[N_COL_]))
      info = 3010;
    
    pbchkvect( *m, *n, 1, *ix, *jx, *lldx, desc_data, 13, 4, &iix, &jjx,
	       &info, int_err);
    pbchkvect( *m, *n, 1, *iy, *jy, *lldy, desc_data, 11, 9, &iiy, &jjy,
	       &info, int_err);
    
    err = info;
    Cigamx2d(ictxt, ALL, TOPDEF, 1, 1, &err, 1, tmpx, tmpx, -1 ,-1,-1);
    
    if( err == 0 )
      if (*n != 0)
	if (*ix == *iy)	/* No communications */
	  if ((iix == 1)&&(iiy == 1)) /* All Columns vectors */
	    if (desc_data[N_ROW_] > 0) { /* I partecipate to the computation */
	      tmpx = &X[2 * ((jjx-1)*(*lldx)+iix-1)]; 
	      tmpy = &Y[2 * ((jjy-1)*(*lldy)+iiy-1)];	      
	      zaxpby_(&(desc_data[N_COL_]), n, alpha, 
		      tmpx, lldx, beta, tmpy, lldy);
	    }
	    else ;
	  else			/* Case: x or y are not Column vectors */
	    info = 3040 ;
	else			/* Case: ix != iy */
	  info = 3050 ;
  }
  
  if( (info != 0) || (err != 0)) {
    psderror_( &ictxt, &info, "PSZAXPBY\0", int_err, real_err );
    return;
  }
}

