/* ---------------------------------------------------------------------
*
*  -- PSSBLAS routine (version 1.0) --
*
*  ---------------------------------------------------------------------
*/
/*
*  Include files
*/
#include "psblas.h"

void psdmdot_( n, k, dot, 
	     X, ix, jx, lldx, 
	     Y, iy, jy, lldy,
	     desc_data, desc_halo, 
	     desc_ovrlap, desc_ovrlap_elem )
     /*
      *  .. Scalar Arguments ..
      */
     int         * ix, * iy, * jx, * jy, 
                 * lldx, *lldy, *n, *k;
     double      *dot;
     /* ..
      *  .. Array Arguments ..
      */
     int         desc_data[], desc_halo[], desc_ovrlap[], desc_ovrlap_elem[] ;
     double      X[], Y[];
{
  /*
   *  Purpose
   *  =======
   *
   *  PSDDOT forms the dot product of two distributed vectors,
   *
   *     dot := sub( X )**T * sub( Y )
   *
   *  where sub( X ) denotes X(IX:IX+N-1,JX+K-1)
   *
   *        sub( Y ) denotes Y(IY:IY+N-1,JY).
   *
   *
   *  Notes
   *  =====
   *
   *  Some description vectors are associated with each distributed sparse
   *  matrix. These vectors stores the information required to the
   *  communication needed to perform distributed operations.
   *
   *  They are:
   *
   *  NAME		 EXPLANATION
   *  ------------------ -------------------------------------------------------
   *  MATRIX_DATA	 Array of INTEGER that contains some local and global
   *			 information of matrix.
   *
   *  DESC_HALO          Array of INTEGER that contains informations for local 
   *                     halo points.
   *
   *  DESC_OVRLAP        Array of INTEGER that contains a list of local overlap 
   *                     elements.
   *                     
   *  OVRLAP_ELEM        Array of INTEGER that contains a list of local overlap
   *                     elements with their multiplicity.
   *
   *  Now we explain each of the above vectors.
   *
   *  Let A be a generic sparse matrix. We denote with MATDATA_A the MATRIX_DATA
   *  array for matrix A.
   *  Data stored in MATRIX_DATA array are:
   *
   *  NOTATION        STORED IN		     EXPLANATION
   *  --------------- ---------------------- -------------------------------------
   *  DEC_TYPE        MATDATA_A[DEC_TYPE_]   Decomposition type
   *  M 	      MATDATA_A[M_]          Total number of equations
   *  N 	      MATDATA_A[N_]          Total number of variables
   *  N_ROW           MATDATA_A[N_ROW_]      Number of local equations
   *  N_COL           MATDATA_A[N_COL_]      Number of local variables
   *  CTXT_A          MATDATA_A[CTXT_]       The BLACS context handle, indicating
   *					     the global context of the operation
   *					     on the matrix.
   *					     The context itself is global.
   *
   *  Let DESCHALO_P be the array DESC_HALO for local process.
   *  This is composed of variable dimension blocks for each process to 
   *  communicate to.
   *  Each block contain indexes of local halo elements to exchange with other 
   *  process.
   *  Let P be the pointer to the first element of a block in DESCHALO_P.
   *  This block is stored in DESCHALO_P as :
   *
   *  NOTATION        STORED IN		          EXPLANATION
   *  --------------- --------------------------- -----------------------------------
   *  PROCESS_ID      DESCHALO_P[P+PROC_ID_]      Identifier of process which exchange 
   *						  data with.
   *  N_ELEMENTS_RECV DESCHALO_P[P+N_ELEM_RECV_]  Number of elements to receive.
   *  ELEMENTS_RECV   DESCHALO_P[P+ELEM_RECV_+i]  Indexes of local elements to
   *					          receive. These are stored in the
   *					          array from location P+ELEM_RECV_ to
   *					          location P+ELEM_RECV_+
   *						  DESCHALO_P[P+N_ELEM_RECV_]-1.
   *  N_ELEMENTS_SEND DESCHALO_P[P+N_ELEM_SEND_]  Number of elements to send.
   *  ELEMENTS_SEND   DESCHALO_P[P+ELEM_SEND_+i]  Indexes of local elements to
   *					          send. These are stored in the
   *					          array from location P+ELEM_SEND_ to
   *					          location P+ELEM_SEND_+
   *						  DESCHALO_P[P+N_ELEM_SEND_]-1.
   *
   *
   *  Let DESCOVRLP_P be the array DESC_OVRLAP for local process.
   *  This is composed of variable dimension blocks for each process to 
   *  communicate to.
   *  Each block contain indexes of local overlap elements to exchange with
   *  other process.
   *  Let P be the pointer to the first element of a block in DESCOVRLP_P.
   *  This block is stored in DESCOVRLP_P as :
   *
   *  NOTATION        STORED IN		            EXPLANATION
   *  ------------- ------------------------------- -----------------------------------
   *  PROCESS_ID    DESCOVRLP_P[P+PROC_ID_]         Identifier of process which exchange
   *						    data with.
   *  N_OVRLAP_ELEM DESCOVRLP_P[P+N_OVRLP_ELEM_]    Number of elements to exchange.
   *  OVRLAP_ELEM   DESCOVRLP_P[P+OVRLP_ELEM_TO_+i] Indexes of local elements to
   *					            exchange. These are stored in the
   *					            array from location P+OVRLP_ELEM_ to
   *					            location P+OVRLP_ELEM_+
   *						    DESCOVRLP_P[P+N_OVRLP_ELEM_]-1.
   *
   *
   *  Let OVR_ELEM_P be the array OVERLAP_ELEM for local process.
   *  This is composed of blocks of two elements. The block
   *  corresponding to the i-th overlapped elements, begin at index 
   *  P = i*2 in array OVR_ELEM_P.
   *  This block is stored in OVR_ELEM_P as :
   *
   *  NOTATION      STORED IN		       EXPLANATION
   *  ------------- -------------------------- ----------------------------------
   *  OVRLAP_ELEM   OVR_ELEM_P[P+OVRLP_ELEM_]  The index of local overlapped 
   *					       element.
   *  N_DOMAINS     OVR_ELEM_P[P+N_DOM_OVR_]   The number of copies of
   *					       local overlapped element.
   *
   *
   *  Parameters
   *  ==========
   *
   *  N       (global input) pointer to INTEGER
   *          The number of rows to be operated on, that is the 
   *          number of rows of submatrix sub(X).  N >= 0.
   *
   *  K       (global input) pointer to INTEGER
   *          The number of columns to be operated on, that is the
   *          number of columns of submatrix sub(X).  K >= 0.
   *
   *  DOT     (local output) pointer to REAL
   *          The dot product of sub( X ) and sub( Y ).
   *
   *  X       (local input) REAL array containing the local
   *          pieces of a distributed dense matrix.
   *          This array contains the entries of the distributed vector
   *          sub( X ).
   *
   *  IX      (global input) pointer to INTEGER
   *          The global row index of the subvector of the distributed
   *          matrix X, to operate on.
   *
   *  JX      (global input) pointer to INTEGER
   *          The global column index of the subvector of the distributed
   *          matrix X to operate on.
   *
   *  LLDX    (local input) pointer to INTEGER
   *          The leading dimension of local dense matrix X.
   *
   *  Y       (local input) REAL array containing the local
   *          pieces of a distributed dense matrix.
   *          This array contains the entries of the distributed vector
   *          sub( Y ).
   *
   *  IY      (global input) pointer to INTEGER
   *          The global row index of the subvector of the distributed
   *          matrix Y, to operate on.
   *
   *  JY      (global input) pointer to INTEGER
   *          The global column index of the subvector of the distributed
   *          matrix Y to operate on.
   *
   *  LLDY    (local input) pointer to INTEGER
   *          The leading dimension of local dense matrix Y.
   *
   *  DESCDATA (global and local input) INTEGER array. Is the MATRIX_DATA
   *           array.
   *
   *  DESCHALO (local input) INTEGER array. Is the DESC_HALO array.
   *
   *  DESCOVRLAP (local input) INTEGER array. Is the DESC_OVRLAP array.
   *
   *  DESCOVRLAPELEM (local input) INTEGER array. Is the OVRLAP_ELEM array.
   *
   *  =====================================================================
   *
   *  .. Local Scalars ..
   */
  int         ictxt, iix, iiy, jjx,
              jjy, mone=-1, mycol, myrow,
              nprow, npcol, ione=1, i, jj;
  int nDim=0;
  double *tmpx, *tmpy;

  /* Error variables */
  int         info=0, err;
  int         int_err[5];
  double      real_err[5];

  /* ..
   *  .. External Functions ..
   */
  void        blacs_gridinfo();
  void        dgsum2d();
  void        pbchkvect();
  void        pberror_();
  double       ddot();
  /* ..
   *  .. Executable Statements ..
   *
   *  Get grid parameters
   */
  ictxt = desc_data[CTXT_];
  Cblacs_gridinfo( ictxt, &nprow, &npcol, &myrow, &mycol );
  /*
   *  Test the input parameters
   */

  info = 0;
  if( nprow == -1 ) {
    info = 2010;
  } else {
    pbchkvect( *n, 1, 1, *ix, *jx, *lldx, desc_data, 11, 3, &iix, &jjx,
	       &info, int_err);
    
    pbchkvect( *n, 1, 1, *iy, *jy, *lldy, desc_data, 11, 7, &iiy, &jjy,
	       &info, int_err);
    
    err = info;
    Cigamx2d(ictxt, ALL, TOPDEF, 1, 1, &err, 1, tmpx, tmpx, -1 ,-1,-1);
    
    if( err == 0 )
      if (*n != 0)
	if (*ix == *iy)	/* No communications */
	  if ((iix == 1)&&(iiy == 1)) /* All Columns vectors */
	    if (desc_data[N_ROW_] > 0) /* I partecipate to the computation */ {
	      /* Number of Local elements to compute */
	      nDim = desc_data[N_ROW_];
	      for (jj=0; jj< *k; jj++) {
		tmpx = &X[(jjx+jj-1)*(*lldx)]; /* Pointer to the local X vector */
		tmpy = &Y[(jjy+jj-1)*(*lldy)]; /* Pointer to the local Y vector */
		dot[jj] = ddot( &nDim, tmpx, &ione, tmpy, &ione );
		/* Adjust *dot because the overlapped elements are computed more
		   than one time */
		i=0;
		while (desc_ovrlap_elem[i] != -1) {
		  dot[jj] -= (double)(desc_ovrlap_elem[i+N_DOM_OVR_]-1)/desc_ovrlap_elem[i+N_DOM_OVR_]*
		    tmpx[desc_ovrlap_elem[i+OVRLP_ELEM_]-1]*
		    tmpy[desc_ovrlap_elem[i+OVRLP_ELEM_]-1];
		  i += 2;
		}
	      }
	      Cdgsum2d( ictxt, ALL, TOPDEF, *k, ione, dot,
			*k, mone, mycol );
	    } else   /* My process not partecipate to the computation */{
	      for (jj=0; jj<*k; jj++)
		dot[jj] = ZERO;
	      Cdgsum2d( ictxt, ALL, TOPDEF, *k, ione, dot,
			*k, mone, mycol );
	    }
	  else			/* Case: x or y are not Column vectors */
	    info = 3040 ;
	else			/* Case: ix != iy */
	  info = 3050 ;
      else
	*dot = 0.0;
  }
  
  if ( err ) {
    psderror_( &ictxt, &info, "PSDDOT", int_err, real_err );
    return;
  }
}

