/* QR Factorization Optimizer    $Revision: 1.17 $      */

/* See the LICENSE file for conditions of usage */

/* This program will read a matrix structure from disk
 * and find an optimal row/column ordering with respect
 * to the number of FLOPS required for solution of the
 * system with a sequence of Givens Transforms (found by
 * greedy search)
 */

/* Usage is:
 *            optimqr <system file> [accept factor] [output directory]
 */

/* The strategy is:
 *  1) Load system
 *  2) Make some initial row- and column-ordering
 *  3) Perform B&B search
 *  4) Print best sequence of Givens transformations, and
 *     best column/row ordering
 *
 */

#define INSTANTIATE
#include "optimqr.h"
#undef INSTANTIATE

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <assert.h>

char outputdir[1024];

void usage(void);

/* The main() routine is the main optimizer routine.
 * What happens is fairly trivial.  The routine parses
 * arguments given, loads the system structure from disk,
 * sets up memory subsystem, etc.
 */
int main(int argc, char** argv) {
  char TMP[1024];
  int rc;
  struct stat st;
  dmh_list* tmp;
  int nnz,i,j;
  
  if(argc < 2 || argc > 4) usage();

  if(argc >= 3) {
    int rc = sscanf(argv[2], "%lf", &accept_factor);
    if(rc != 1) {
      /* It's not accept_factor, but output file */
      strcpy(outputdir, argv[2]);
      accept_factor = ACCEPT_FACTOR;
    } else {
      if(argc == 4) {
	strcpy(outputdir, argv[3]);
      } else {
	strcpy(outputdir, "Output");
      }
    }
  } else {
    accept_factor = ACCEPT_FACTOR;
    strcpy(outputdir, "Output");
  }

  if(accept_factor > 1) {
    accept_factor /= 100;
  }

  /* Validate output directory */
  printf("Using %f as acceptance factor\n", accept_factor);
  printf("Using %s as output directory\n", outputdir);
  strcat(outputdir, "/");
  /* Does output directory exist ? */
  rc = stat(outputdir, &st);
  if(rc == -1) { /* Create output directory */
    rc = mkdir(outputdir, 0755);
    if(rc == -1) fatal("Couldn't create output directory!");  
  } else 
    if(!S_ISDIR(st.st_mode)) 
      fatal("Output directory is not a directory");

  /* Load structure and initialize dimension variables */
  printf("Loading matrix structure...");
  loadsys(argv[1]);
  strcpy(TMP, outputdir);
  strcat(TMP, "optimqr.initial");
  printsys(matrixA, TMP);
  printf(" %ux%u system loaded.\n", dim1,dim2);

  nnz = 0;
  for(j = 0; j < dim1; j++)
    for(i = 0; i < dim2; i++)
      if(matrixA->elements[i][j]) nnz++;
  printf("System density is %3.3f %%\n", 100*(double)nnz/(double)(dim1*dim2));

  /* Initialize memory subsystem */
  printf("Initializing memory subsystem...\n");
  dmh_initialize(DMHSIZE);

  /* Give the BB routine something to work with */
  tmp = dmh_alloc();
  orderedsolution = tmp->solution;
  initialize_solution(orderedsolution);
  bestsolution = dmh_alloc()->solution;
  bestsolution->f = OVERMUCH;
  bestsolution->g = OVERMUCH;
  printf("Initial solution has f() = %u\n",bestsolution->f);

  printf("Starting branch-and-bound search...\n"); 

  bb_search();

  if(bestsolution) {
    char bestout[512];
    unsigned lower,objective;

    costs(bestsolution,&objective,&lower);

    printf("Optimal solution has f(best) = %u\n", lower);
    sprintf(bestout,"%s/optimqr.best", outputdir);
    writesys(bestout, bestsolution);

    sprintf(bestout,"%s/optimqr.seq", outputdir);
    seqout(bestout, bestsolution);

  } else 
    printf("No solution was found...\n");

  stat_heuristic();
  dmh_stat();

  exit(0);
}


/* The fatal() routine can be called upon
 * fatal errors.  It exits with memory stats
 * and an error message (given as the only argument).
 */
void fatal(char* text) {
  if(text) printf("\n%s\n\n",text);
  dmh_stat();
  exit(1);
}

/* The costs() routine calculates the value of the objective function,
 * and the lower bound on the cost for a not-yet fully ordered system.
 *
 * This routine is used both in the main() routine, and in the
 * init_bounds() routine (from systools.c).
 *
 * The routine uses the find_sequence() routine to obtain the certain
 * cost of the ordered subspace of the system matrix, and in case the system
 * is not yet fully ordered (eg. there exists an unordered subspace too),
 * we produce a worst-case scenario cost of this undefined subspace.  We also
 * add a ``penalty'' for not having ordered the entire system yet (this is
 * done to ensure, that the optimizer sees that it pays off to order all of
 * the system).
 */
void costs(Tsolution* sys, unsigned* objective, unsigned* lower) {
  int n_undef_size;   /* Not-yet-ordered area size */
  int n_undef_lt;     /* the lower triangle size of non-ordered area */
  int n_undef_ut;
  int n_undef_cost;  
  int certain_cost;

  assert(sys);
  assert(matrixA);

  certain_cost = find_sequence(sys,0,NULL);

  /* If something is still undefined */
  if(dim1 > sys->ordered_pairs || dim2 > sys->ordered_pairs) {
    unsigned n,penalty;

    /* Worst case is, all upper-triangle fillins, and all elements in the
     * lower triangle.
     * Ofcourse, if there are more nonzeros than there are lower-
     * triangle spaces, we will have to leave some nonzeros in the
     * upper triangle.
     */

    /* Calc. lower-triangle size */
    /* The lower triangle in a matrix of n*n size starting at row s
     * has size:
     *   (n+s-1)*(n-s)/2
     */
    n = dim1 - sys->ordered_pairs;

    n_undef_lt = (n-1) * n / 2;
    n_undef_ut = (n+1) * n / 2;
    n_undef_size = n*n;

    /* \Gamma = G \cdot N_{TL} + N_{TU}   */
    n_undef_cost = GIVENS_COST * n_undef_lt + (n_undef_size - n_undef_lt);
    penalty = dim1 - sys->ordered_pairs;
    assert(penalty);

    *objective = certain_cost + n_undef_cost + penalty;

    /* Calc. g(s)  */

    *lower = certain_cost;

    if(sys->ordered_pairs) {
      sys->f_history[sys->ordered_pairs-1] = sys->f;
      sys->g_history[sys->ordered_pairs-1] = sys->g;
/*       sys->utility[sys->ordered_pairs-1] = certain_cost; */
    }

/*     writesys("new-branch-dump",sys);   */

  } else {

    assert(dim1 == sys->ordered_pairs);
    assert(dim2 == sys->ordered_pairs);

    /* All of system is ordered */
    *objective = *lower = certain_cost;

    sys->f_history[sys->ordered_pairs-1] = sys->f;
    sys->g_history[sys->ordered_pairs-1] = sys->g;
  }
}


/* The usage() function simply prints usage instructions for the
 * optimizer, then exits.  It's called when the optimizer is run with
 * an incorrect number of arguments.
 */
void usage(void)
{
  printf("Usage:  optimqr <system file> [accept factor] [output directory]\n\n");
  exit(1);  
}
