/* This is the dynamic memory handler $Revision: 1.8 $  */

/* This will handle a pool of ``solution buffers'' and ``small
 * buffers'' allowing O(1) time allocation and de-allocation of data
 * areas for new branches in the search, and for temporary data used
 * by the various row-ordering strategies.
 */

#include "optimqr.h"
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

Tsolution* dmh_pool = NULL;
dmh_list* free_buffers = NULL;
dmh_list* taken_buffers = NULL;

stype* dmh_spool = NULL;
dmh_slist* free_sbuffers = NULL;
dmh_slist* taken_sbuffers = NULL;

unsigned dmh_no_free = 0;
unsigned dmh_no_taken = 0;
unsigned dmh_no_allocs = 0;
unsigned dmh_no_frees = 0;

unsigned dmh_no_sfree = 0;
unsigned dmh_no_staken = 0;
unsigned dmh_no_sallocs = 0;
unsigned dmh_no_sfrees = 0;

/* The dmh_get_num_sfree() routine simply returns the
 * number of free ``small'' buffers. This routine is
 * used by the recursive row-ordering routine (in recorder.c)
 * to help it decide when it should throw away orderings
 * in order to not use to much memory.
 */
unsigned dmh_get_num_sfree(void)
{
  return dmh_no_sfree;
}

/* The dmh_initialize() routine initializes the memory
 * management subsystem, that later will allow us to
 * allocate and free buffers for holding (partial) solutions
 * and row ordering data.
 *
 * The routine takes one argument, which is the number of
 * solution (or ``large'') buffers it should allocate.
 * It allocates the requested number of buffers, and calculates
 * how many ``small'' buffers it can allocate, while staying
 * below the MAX_MEM (defined in optimqr.h) memory usage limit.
 */
void dmh_initialize(unsigned count) {
/* Allocate a base pool */
  int i;
  dmh_list* new_buf;
  dmh_slist* new_sbuf;
  unsigned scount;

  dmh_pool = (Tsolution*)malloc(sizeof(Tsolution)*count);
  if(!dmh_pool) fatal("Out of memory allocatin dmh pool");
  printf("dmh allocated %i solution buffers\n", count);

  scount = MAX_MEM - count*sizeof(Tsolution);
  scount /= sizeof(stype)*dim1 + sizeof(dmh_slist);
  dmh_spool = (stype*)malloc(sizeof(stype)*dim1*scount);
  if(!dmh_spool) fatal("Out of memory allocatin dmh spool");
  printf("dmh allocated %i small buffers\n", scount);

  dmh_pool->row_ordering = 
    (short*)malloc(count*dim1*sizeof(short));
  if(!dmh_pool->row_ordering) 
    fatal("Out of memory allocating row ordering buffer");

  dmh_pool->column_ordering = 
    (short*)malloc(count*dim2*sizeof(short));
  if(!dmh_pool->column_ordering) 
    fatal("Out of memory allocating column ordering buffer");

  dmh_pool->f_history = 
    (unsigned short*)malloc(count*dim2*sizeof(unsigned short));
  if(!dmh_pool->f_history) 
    fatal("Out of memory allocating f history buffer");
  
  dmh_pool->g_history = 
    (unsigned short*)malloc(count*dim2*sizeof(unsigned short));
  if(!dmh_pool->g_history) 
    fatal("Out of memory allocating g history buffer");

  dmh_pool->h_history = 
    (unsigned short*)malloc(count*dim2*sizeof(unsigned short));
  if(!dmh_pool->h_history) 
    fatal("Out of memory allocating h history buffer");

/*   dmh_pool->utility =  */
/*     (unsigned*)malloc(count*dim2*sizeof(unsigned)); */
/*   if(!dmh_pool->utility)  */
/*     fatal("Out of memory allocating utility history buffer"); */

  new_buf = (dmh_list*)malloc(count*sizeof(dmh_list));
  if(!new_buf) fatal("Out of memory allocating pool buffer");

  new_sbuf = (dmh_slist*)malloc(scount*sizeof(dmh_slist));
  if(!new_sbuf) fatal("Out of memory allocating spool buffer");

  for(i = 0; i < count; i++) {

    dmh_pool[i].row_ordering = dmh_pool->row_ordering + dim1*i;
    dmh_pool[i].column_ordering = dmh_pool->column_ordering + dim2*i;
    dmh_pool[i].f_history = dmh_pool->f_history + dim2*i;
    dmh_pool[i].g_history = dmh_pool->g_history + dim2*i;
    dmh_pool[i].h_history = dmh_pool->h_history + dim2*i;
/*     dmh_pool[i].utility = dmh_pool->utility + dim2*i; */

    new_buf[i].solution = &(dmh_pool[i]);

    /* insert buffer in free_buffers */
    new_buf[i].prev = NULL;
    new_buf[i].next = free_buffers;
    if(free_buffers != NULL) free_buffers->prev = new_buf+i;
    free_buffers = new_buf+i;
    
    dmh_no_free++;
  }

  for(i = 0; i < scount; i++) {
#ifdef ROWORDER_RECORDER
    new_sbuf[i].row_ordering = dmh_spool + dim1*i;
#endif
#ifdef ROWORDER_BIPORDER
    new_sbuf[i].vertex = NULL;
#endif
    /* insert buffer in free_buffers */
    new_sbuf[i].prev = NULL;
    new_sbuf[i].next = free_sbuffers;
    if(free_sbuffers != NULL) free_sbuffers->prev = new_sbuf+i;
    free_sbuffers = new_sbuf+i;   
    dmh_no_sfree++;
  }
}

/* The dmh_alloc() routine allocates a ``large'' buffer, and
 * returns a pointer to this buffer.
 *
 * If we run out of memory, we terminate the program with
 * an error message.
 */
dmh_list* dmh_alloc(void) {
  /* Grab a buffer from free_buffser 
   * and move it to taken_buffers 
   */
  
  dmh_list* grab;

  if(!free_buffers) fatal("DMH is out of free buffers!");

  /* Grab buffer and remove from free buffers */
  grab = free_buffers;
  free_buffers = free_buffers->next;
  if(free_buffers) free_buffers->prev = NULL;
  assert(dmh_no_free > 0);
  dmh_no_free--;
  
  /* insert in taken buffers */
  grab->prev = NULL;
  grab->next = taken_buffers;
  if(taken_buffers) taken_buffers->prev = grab;
  taken_buffers = grab;
  dmh_no_taken++;

  /* Return as a hero in O(1) time */
  dmh_no_allocs++;
  return grab;
}


/* The dmh_salloc() routine allocates a ``small'' buffer
 * and returns a pointer to this buffer.
 *
 * If we run out of small buffers, we terminate the program
 * with an error message.
 */
dmh_slist* dmh_salloc(void) {
  /* Grab an sbuffer from free_sbuffers 
   * and move it to taken_sbuffers 
   */
  
  dmh_slist* grab;

  if(!free_sbuffers) fatal("DMH is out of free sbuffers!");

  /* Grab buffer and remove from free sbuffers */
  grab = free_sbuffers;
  free_sbuffers = free_sbuffers->next;
  if(free_sbuffers) free_sbuffers->prev = NULL;
  assert(dmh_no_sfree > 0);
  dmh_no_sfree--;
  
  /* insert in taken buffers */
  grab->prev = NULL;
  grab->next = taken_sbuffers;
  if(taken_sbuffers) taken_sbuffers->prev = grab;
  taken_sbuffers = grab;
  dmh_no_staken++;

  /* Return as a hero in O(1) time */
  dmh_no_sallocs++;
  return grab;
}


/* The dmh_free() routine frees a large buffer.  The routine
 * takes one argument, which is a pointer to the buffer we wish
 * to de-allocate.
 */
void dmh_free(dmh_list* buffer) {
  /* Move the buffer from taken_buffers 
   * to free_buffers 
   */

  /* Remove the buffer from taken_buffers */
  if(buffer == taken_buffers) taken_buffers = buffer->next;
  if(buffer->prev) buffer->prev->next = buffer->next;
  if(buffer->next) buffer->next->prev = buffer->prev;
  assert(dmh_no_taken > 0);
  dmh_no_taken--;

  /* Insert the buffer in free_buffers */
  buffer->prev = NULL;
  buffer->next = free_buffers;
  if(free_buffers != NULL) free_buffers->prev = buffer;
  free_buffers = buffer;  
  dmh_no_free++;

  dmh_no_frees++;
  /* Return in O(1) time */
}

/* The dmh_sfree() routine frees a small buffer. The routine
 * takes one argument, which is a pointer to the small buffer we
 * wish to de-allocate.
 */
void dmh_sfree(dmh_slist* buffer) {
  /* Move the buffer from taken_sbuffers 
   * to free_sbuffers 
   */

  /* Remove the sbuffer from taken_sbuffers */
  if(buffer == taken_sbuffers) taken_sbuffers = buffer->next;
  if(buffer->prev) buffer->prev->next = buffer->next;
  if(buffer->next) buffer->next->prev = buffer->prev;
  assert(dmh_no_staken > 0);
  dmh_no_staken--;

  /* Insert the buffer in free_sbuffers */
  buffer->prev = NULL;
  buffer->next = free_sbuffers;
  if(free_sbuffers != NULL) free_sbuffers->prev = buffer;
  free_sbuffers = buffer;  
  dmh_no_sfree++;
  dmh_no_sfrees++;
}

/* The dmh_stat() routine prints statistics for the memory
 * subsystem. This routine is called on exit from the optimizer,
 * in order to give the user hints about the memory usage statistics
 * etc.
 */
void dmh_stat(void) {
  printf("DMH statistics:\n");
  printf(" Free buffers:   %u\n", dmh_no_free);
  printf(" Taken buffers:  %u\n", dmh_no_taken);
  printf(" Allocs done:    %u\n", dmh_no_allocs);
  printf(" Frees done:     %u\n", dmh_no_frees);
  printf(" Free sbuffers:  %u\n", dmh_no_sfree);
  printf(" Taken sbuffers: %u\n", dmh_no_staken);
  printf(" sAllocs done:   %u\n", dmh_no_sallocs);
  printf(" sFrees done:    %u\n", dmh_no_sfrees);
}
