/*
 * Bipartite Matching Row Ordering Strategy Routine
 *
 * $Revision: 1.2 $
 *
 * This is much more efficient than recorder.c.  Unfortunately it does
 * not give as good estimates, but we can live with that if this is
 * fast enough.
 */

#include "optimqr.h"
#include <stdlib.h>
#include <assert.h>

Vertex  *  vertices = NULL;
Vertex *** adjlists = NULL;
Vertex  ** adjarea  = NULL;  
Vertex *** dfs = NULL;

#define LEFT      -1
#define RIGHT     -2
#define UNMATCHED NULL

dmh_slist * queue = NULL;
dmh_slist * qtail = NULL;

int nsolution = 0;

void init_queue(void);
int  empty_queue(void);
void enqueue(Vertex* v);
void dequeue(Vertex** v);

void DFS(Vertex* v, int l);
void rDFS(Vertex* v, int l, int* success);

/* The roworder() routine takes one argument: The system
 * for which to calculate the ordering of the non-ordered
 * rows.
 *
 * Only one ordering is found, although a very large number
 * of orderings are usually valid.
 * We may return a lucky ordering, eg. one with a cost
 * close to the best possible, or we may shoot far from that.
 *
 * The good thing is, that the bipartite ordering can be used
 * efficiently on larger systems than the recursive ordering
 * algorithm.
 */
int roworder(Tsolution* sys)
{
  int i,j;
  int maxfound;

  /* eventually allocate storage */
  if(!vertices) {
    assert(!adjlists);   
    assert(!adjarea);
    assert(!dfs);

    vertices = (Vertex*)malloc(sizeof(Vertex) * dim1 * 2);
    dfs = (Vertex***)malloc(sizeof(Vertex**) * dim1 * 2);

    assert(dfs);
    assert(vertices);

    for(i = 0; i < dim1*2; i++) {
      dfs[i] = NULL;
    }

    adjarea = (Vertex**)malloc(sizeof(Vertex*) * dim1 * dim2 * 2);
    assert(adjarea);

    adjlists = (Vertex***)malloc(sizeof(Vertex**) * dim1 * 2);
    assert(adjlists);

    for(i = 0; i < dim1 * 2; i++)
      adjlists[i] = adjarea + i*dim1;
  }  

  init_queue();

  /* Build a representation of the bipartite graph */
  /* We represent allready fixed rows as rows with only one
   * row element, namely the one that will fix their location
   */
  for(i = 0; i < dim1; i++) {
    vertices[i].id = i;
    vertices[i].level = LEFT;
    vertices[i].mate = UNMATCHED;
  }
  for(i = 0; i < dim2; i++) {
    vertices[dim1+i].id = dim1+i;
    vertices[dim1+i].level = RIGHT;
    vertices[dim1+i].mate = UNMATCHED;
  }
  /* Set diagonal row-element for fixed rows */
  for(i = 0; i < sys->ordered_pairs; i++) {
    adjlists[sys->row_ordering[i]][0] = 
      &vertices[i+dim1];
    adjlists[sys->row_ordering[i]][1] = NULL;
  }
  /* Treat remaining rows */
  for(i = sys->ordered_pairs; i < dim1; i++) {
    int adjcnt = 0;
    for(j = sys->ordered_pairs; j < dim2; j++)
      if(MATRIX(sys,i,j))
	adjlists[sys->row_ordering[i]][adjcnt++] = 
	  &vertices[j+dim1];
    if(adjcnt < dim1)
      adjlists[sys->row_ordering[i]][adjcnt] = NULL;
  }
  /* Set diagonal column-element for columns belonging to
   * a fixed row */
  for(j = 0; j < sys->ordered_pairs; j++) {
    adjlists[dim1+j][0] = 
      &vertices[sys->row_ordering[j]];
    adjlists[dim1+j][1] = NULL;
  }
  /* Treat remaining columns */
  for(j = sys->ordered_pairs; j < dim1; j++) {
    int adjcnt = 0;
    for(i = sys->ordered_pairs; i < dim2; i++)
      if(MATRIX(sys,i,j))
	adjlists[j+dim1][adjcnt++] = 
	  &vertices[sys->row_ordering[i]];
    if(adjcnt < dim1) 
      adjlists[j+dim1][adjcnt] = NULL;
  }

  /* Attempt matching */
  
  do {
    int CurrentLevel, RightSideEnqueued;

    maxfound = 1;

    for(i = 0; i < 2*dim1; i++) {
      if(vertices[i].level == LEFT) {
	if(vertices[i].mate == UNMATCHED) {
	  vertices[i].level = 0;
	  enqueue(&vertices[i]);
	}
      }	else if(vertices[i].level == RIGHT) {
	dfs[i] = adjlists[i];
      }
    } 

    CurrentLevel = 1;
    RightSideEnqueued = 0;

    /* Breadth first search... */
    while(!empty_queue()) {

      Vertex* v;
      Vertex* p;
      int pindex;

      dequeue(&v);

      if(v->level == CurrentLevel) {

	if(!RightSideEnqueued) {
	  CurrentLevel++;
	} else {
	  while(v->mate != UNMATCHED)
	    dequeue(&v);

	  DFS(v, CurrentLevel);
	  while(!empty_queue()) {
	    dequeue(&v);
	    DFS(v, CurrentLevel);
	  }
	  
	  for(i = 0; i < dim1*2; i++)
	    if(vertices[i].level >= 0) {
	      vertices[i].level = LEFT;
	    }

	  /* now leapfrog back to the main while() loop */
	  maxfound = 0;
	  continue;	  
	} /* If !RightSideEnqueued */
      } /* if v->level == CurrentLevel */
      
      pindex = 0;
      while((p = adjlists[v->id][pindex++]) && pindex < dim1) {
	if(p->mate == UNMATCHED) {
	  if(p->level == RIGHT) { /* not in queue */
	    p->level = CurrentLevel;
	    RightSideEnqueued = 1;
	    enqueue(p);
	  } 
	} else {
	  Vertex* w = p->mate;
	  assert(w);
	  /* if unseen and still relevant */
	  if(w->level == LEFT  && !RightSideEnqueued) {
	    w->level = CurrentLevel;
	    enqueue(w);
	  }
	}
      } /* while (p = p->next) != NULL ...  */
    } /* while ( ! is_empty_queue() ) */

    if(maxfound) {

/*       char name[1024]; */
      /* maximum matching found. */

      for(i = 0; i < dim1; i++) {
	if(!vertices[i+dim1].mate) {
	  return -2;
	}
 	sys->row_ordering[i] = vertices[i+dim1].mate->id;

 	if(!MATRIX(sys,i,i)) { 
 	  printf("Fucked up system at i = %i!\n", i); 
 	  writesys("assertion", sys); 
 	  exit(-1); 
 	} 

      }

/*       sprintf(name, "Output/biporder.%i", nsolution++); */
/*       writesys(name, sys); */
      
      return 0;
    }

  } while ( 86 != 42 );
  assert(0);
}


/* The init_queue() function deletes all
 * entries in the "queue" global list
 * variable.
 */
void init_queue(void)
{
  /* initialize queue */
  dmh_slist * trvrs = queue;
  while(trvrs) {
    dmh_slist * del = trvrs;
    trvrs = trvrs->aux_next;
    dmh_sfree(del);
  }
  queue = NULL;
  qtail = NULL;
}

/* tell us whether the queue is empty */
int  empty_queue(void)
{
  return queue == NULL;
}

/* The enqueue() routine puts a vertex
 * in the bipartite graph onto the queue.
 */
void enqueue(Vertex* v)
{
  /* enqueue */
  if(!qtail) {
    assert(!queue);
    queue = qtail = dmh_salloc();
    queue->aux_next = NULL;
  } else {
    dmh_slist * new;
    assert(queue);
    new = dmh_salloc();
    qtail->aux_next = new;
    qtail = new;
  }
  qtail->vertex = v;
  qtail->aux_next = NULL;
}

/* The dequeue() routine removes a vertex
 * from the queue structure. The removed
 * vertex is also returned in the argument
 * variable.
 */
void dequeue(Vertex** v)
{
  /* dequeue */
  dmh_slist* del;
  assert(queue);
  *v = queue->vertex;
  del = queue;
  queue = queue->aux_next;
  if(!queue) qtail = NULL;
  dmh_sfree(del);
}


/* Depth first search */
void DFS(Vertex* v, int l)
{
  int dummy = 0;
  rDFS(v, l, &dummy);
  v->level = RIGHT;
}

/* The rDFS() routine performs a recursive depth-first
 * search in the bipartite graph.
 */
void rDFS(Vertex* v, int l, int* success)
{
  /* recursive part of DFS */
  int dindex = 0;
  assert(v);
  assert(v->id >= dim1);  /* dfs is only defined for rhs */
  assert(v->id < dim1*2);
  while(dfs[v->id][dindex] && dindex < dim1) {

    if(dfs[v->id][dindex]->level == l-1) {

      if( l == 1 ) *success = 1;
      else rDFS(dfs[v->id][dindex]->mate, l-1, success);

      if(*success) {
	/* Augment path */
	dfs[v->id][dindex]->mate = v;
	v->mate = dfs[v->id][dindex];
	dfs[v->id][dindex]->level = LEFT;
	return;
      }
    }
    dindex++;
  }
}



