#include "EliminationForest.h"

#undef __FUNC__
#define __FUNC__ "allocate"
void EliminationForest::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    frontPointer_ = new Array<int>(numberOfNodes_ + 1);
    frontMap_ = new Array<int>(numberOfNodes_);
    borderSize_ = new Array<int>(numberOfNodes_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
void EliminationForest::free(void)
{
  BEGIN_FUNCTION();

  delete frontPointer_;
  frontPointer_ = 0;
  delete frontMap_;
  frontMap_ = 0;
  delete borderSize_;
  borderSize_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "EliminationForest"
EliminationForest::EliminationForest():
  order_(0),
  frontPointer_(0),
  frontMap_(0),
  borderSize_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "EliminationForest"
EliminationForest::EliminationForest(int numberOfNodes):
  Forest(numberOfNodes),
  order_(numberOfNodes),
  frontPointer_(0),
  frontMap_(0),
  borderSize_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~EliminationForest"
EliminationForest::~EliminationForest()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
void EliminationForest::resize(int numberOfNodes)
{
  BEGIN_FUNCTION();

  Forest::resize(numberOfNodes);
  free();
  order_ = numberOfNodes;
  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
int EliminationForest::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = Forest::getHeapSize() +
                 frontPointer_->getHeapSize() + 
                 frontMap_->getHeapSize() + 
                 borderSize_->getHeapSize();

  END_FUNCTION();

  return heapSize;
}

#undef __FUNC__
#define __FUNC__ "check"
bool EliminationForest::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (!Forest::check())
    return false;
  if (order_ < 0)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "print"
void EliminationForest::print(void) const
{
  BEGIN_FUNCTION();

  const int* parentItem = parent_->getItem();
  const int* firstChildItem = firstChild_->getItem();
  const int* nextSiblingItem = nextSibling_->getItem();
  const int* frontPointerItem = frontPointer_->getItem();
  const int* frontMapItem = frontMap_->getItem();
  const int* borderSizeItem = borderSize_->getItem();

  cout << "ELIMINATION_FOREST object" << endl
       << "Number of nodes: " << numberOfNodes_ << endl
       << "Root: " << root_ + 1 << endl
       << "Order: " << order_ << endl;

  cout << "Parents:";
  {for (int i = 0; i < numberOfNodes_; i++)
    cout << " " << parentItem[i] + 1;}
  cout << endl;

  cout << "First children:";
  {for (int i = 0; i < numberOfNodes_; i++)
    cout << " " << firstChildItem[i] + 1;}
  cout << endl;

  cout << "Next siblings:";
  {for (int i = 0; i < numberOfNodes_; i++)
    cout << " " << nextSiblingItem[i] + 1;}
  cout << endl;

  cout << "Front pointers:";
  {for (int i = 0; i <= numberOfNodes_; i++)
    cout << " " << frontPointerItem[i] + 1;}
  cout << endl;

  cout << "Front map:";
  {for (int i = 0; i < order_; i++)
    cout << " " << frontMapItem[i] + 1;}
  cout << endl;

  cout << "Border sizes:";
  for (int i = 0; i < numberOfNodes_; i++)
    cout << " " << borderSizeItem[i];
  cout << endl;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "save"
void EliminationForest::save(const char* fileName) const
{
  BEGIN_FUNCTION();

  ofstream fout(fileName);
  if (!fout)
    SET_ERROR1(FileOpen);

  int* parentItem = parent_->getItem();
  int* firstChildItem = firstChild_->getItem();
  int* nextSiblingItem = nextSibling_->getItem();
  int* frontPointerItem = frontPointer_->getItem();
  int* frontMapItem = frontMap_->getItem();

  {for (int i = 0; i < numberOfNodes_; i++)
    fout << " " << parentItem[i] + 1;}
  fout << endl;

  {for (int i = 0; i < numberOfNodes_; i++)
    fout << " " << firstChildItem[i] + 1;}
  fout << endl;

  {for (int i = 0; i < numberOfNodes_; i++)
    fout << " " << nextSiblingItem[i] + 1;}
  fout << endl;

  {for (int i = 0; i <= numberOfNodes_; i++)
    fout << " " << frontPointerItem[i] + 1;}
  fout << endl;

  for (int i = 0; i < order_; i++)
    fout << " " << frontMapItem[i] + 1;
  fout << endl;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
void EliminationForest::initialize(const SparseMatrix<double>& a,
                                   Permutation& p)
{
  BEGIN_FUNCTION();

  if (numberOfNodes_ != a.getOrder())
    SET_ERROR1(InvalidInput);

  if (numberOfNodes_ != p.getOrder())
    SET_ERROR1(InvalidInput);

  initialize(a.getColumnPointer(), a.getRowIndex(), p);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
void EliminationForest::initialize(const SparseMatrix<Complex>& a,
                                   Permutation& p)
{
  BEGIN_FUNCTION();

  if (numberOfNodes_ != a.getOrder())
    SET_ERROR1(InvalidInput);

  if (numberOfNodes_ != p.getOrder())
    SET_ERROR1(InvalidInput);

  initialize(a.getColumnPointer(), a.getRowIndex(), p);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
void EliminationForest::initialize(const SparseFactors<double>& l)
{
  BEGIN_FUNCTION();

  if (numberOfNodes_ != l.getOrder())
    SET_ERROR1(InvalidInput);

  initialize(l.getColumnPointer(), l.getRowIndex());

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
void EliminationForest::initialize(const SparseFactors<Complex>& l)
{
  BEGIN_FUNCTION();

  if (numberOfNodes_ != l.getOrder())
    SET_ERROR1(InvalidInput);

  initialize(l.getColumnPointer(), l.getRowIndex());

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "compress"
void EliminationForest::compress(void)
{
  BEGIN_FUNCTION();

  int newNumberOfNodes;
  Array<int> newParent(numberOfNodes_);
  Array<int> newFirstChild(numberOfNodes_);
  Array<int> newNextSibling(numberOfNodes_);
  Array<int> newFrontPointer(numberOfNodes_ + 1);
  Array<int> newBorderSize(numberOfNodes_);
  CHECK_ERROR1();

  int* parentItem = parent_->getItem();
  int* firstChildItem = firstChild_->getItem();
  int* nextSiblingItem = nextSibling_->getItem();
  int* frontPointerItem = frontPointer_->getItem();
  int* frontMapItem = frontMap_->getItem();
  int* borderSizeItem = borderSize_->getItem();

  int* newParentItem = newParent.getItem();
  int* newFirstChildItem = newFirstChild.getItem();
  int* newNextSiblingItem = newNextSibling.getItem();
  int* newFrontPointerItem = newFrontPointer.getItem();
  int* newBorderSizeItem = newBorderSize.getItem();

  /* Compute the new number of nodes, front pointers, front map
     and border sizes (fundamental fronts). */

  newNumberOfNodes = 0;
  /* For every old node: */
  {for (int u = 0; u < numberOfNodes_; u++)
  {
    if (u == 0)
    /* If <u> is the first old node then it belongs to the first front. */
    {
      newFrontPointerItem[newNumberOfNodes] = u;
      frontMapItem[u] = newNumberOfNodes;
      newBorderSizeItem[newNumberOfNodes] = borderSizeItem[u];
      newNumberOfNodes++;
    }
    else if ((u == parentItem[u - 1]) &&
             (firstChildItem[u] == u - 1) &&
             (borderSizeItem[u] == borderSizeItem[u - 1] - 1))
    /* Else if <u> is the parent of <u - 1>, <u - 1> is its only child
       and they have identical off-diagonal sparsity patterns then <u>
       belongs to the current front. */
    {
      frontMapItem[u] = newNumberOfNodes - 1;
      newBorderSizeItem[newNumberOfNodes - 1]--;
    }
    else
    /* Else <u> belongs to a new front. */
    {
      newFrontPointerItem[newNumberOfNodes] = u;
      frontMapItem[u] = newNumberOfNodes;
      newBorderSizeItem[newNumberOfNodes] = borderSizeItem[u];
      newNumberOfNodes++;
    }
  }}
  newFrontPointerItem[newNumberOfNodes] = order_;

  /* Compute the new parent, first child and next sibling links,
     and the new root. */

  /* Initialize the new root. */
  int newRoot = -1;

  /* Initialize the new first child and next sibling links. */
  {for (int x = 0; x < newNumberOfNodes; x++)
    newFirstChildItem[x] = newNextSiblingItem[x] = -1;}

  /* For every new node (backwards): */
  {for (int x = newNumberOfNodes - 1; x >= 0; x--)
  {
    /* Get the last old node in <x> and its parent. */
    int u = newFrontPointerItem[x + 1] - 1,
        v = parentItem[u];
    if (v == -1)
    /* If <u> is a root then <x> is also a root. */
    {
      newParentItem[x] = -1;
      newNextSiblingItem[x] = newRoot;
      newRoot = x;
    }
    else
    /* Else the parent of <x> is the new node that contains <v>,
       and <x> is its first child. */
    {
      int y = frontMapItem[v];
      newParentItem[x] = y;
      newNextSiblingItem[x] = newFirstChildItem[y];
      newFirstChildItem[y] = x;
    }
  }}

  numberOfNodes_ = newNumberOfNodes;
  root_ = newRoot;
  parent_->resize(numberOfNodes_);
  firstChild_->resize(numberOfNodes_);
  nextSibling_->resize(numberOfNodes_);
  frontPointer_->resize(numberOfNodes_ + 1);
  borderSize_->resize(numberOfNodes_);
  CHECK_ERROR1();

  parentItem = parent_->getItem();
  firstChildItem = firstChild_->getItem();
  nextSiblingItem = nextSibling_->getItem();
  frontPointerItem = frontPointer_->getItem();
  frontMapItem = frontMap_->getItem();
  borderSizeItem = borderSize_->getItem();

  /* Transfer data. */
  for (int x = 0; x < numberOfNodes_; x++)
  {
    parentItem[x] = newParentItem[x];
    firstChildItem[x] = newFirstChildItem[x];
    nextSiblingItem[x] = newNextSiblingItem[x];
    frontPointerItem[x] = newFrontPointerItem[x];
    borderSizeItem[x] = newBorderSizeItem[x];
  }
  frontPointerItem[numberOfNodes_] = newFrontPointerItem[numberOfNodes_];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getNumberOfRowIndices"
int EliminationForest::getNumberOfRowIndices(void) const
{
  BEGIN_FUNCTION();

  const int* frontPointerItem = frontPointer_->getItem();
  const int* borderSizeItem = borderSize_->getItem();

  int numberOfRowIndices = 0;
  for (int u = 0; u < numberOfNodes_; u++)
    numberOfRowIndices += frontPointerItem[u + 1] -
                          frontPointerItem[u] +
                          borderSizeItem[u];

  END_FUNCTION();

  return numberOfRowIndices;
}

#undef __FUNC__
#define __FUNC__ "getNumberOfOffDiagonals"
int EliminationForest::getNumberOfOffDiagonals(void) const
{
  BEGIN_FUNCTION();

  const int* frontPointerItem = frontPointer_->getItem();
  const int* borderSizeItem = borderSize_->getItem();

  int numberOfOffDiagonals = 0;
  for (int u = 0; u < numberOfNodes_; u++)
    for (int numberOfColumns = frontPointerItem[u + 1] -
                               frontPointerItem[u],
             frontNumberOfOffDiagonals = frontPointerItem[u + 1] -
                                         frontPointerItem[u] +
                                         borderSizeItem[u] - 1;
         numberOfColumns > 0;
         numberOfColumns--, frontNumberOfOffDiagonals--)
      numberOfOffDiagonals += frontNumberOfOffDiagonals;

  END_FUNCTION();

  return numberOfOffDiagonals;
}

#undef __FUNC__
#define __FUNC__ "getWork"
void EliminationForest::getWork(float& eliminationWork,
                                float& assemblyWork,
                                float& solutionWork,
                                double) const
{
  BEGIN_FUNCTION();

  const int* frontPointerItem = frontPointer_->getItem();
  const int* borderSizeItem = borderSize_->getItem();

  eliminationWork = assemblyWork = solutionWork = 0;

  /* For every node: */
  for (int u = 0; u < numberOfNodes_; u++)
  {
    float frontSize = (float)(frontPointerItem[u + 1] -
                        frontPointerItem[u]),
          borderSize = (float)borderSizeItem[u],
          order = frontSize + borderSize;

    /* For every column: */
    for (int j = frontPointerItem[u]; j < frontPointerItem[u + 1]; j++)
    {
      eliminationWork += order * order + order - 1;
      solutionWork += 4 * order - 3;
      order -= 1;
    }

    assemblyWork += borderSize * (borderSize + 1) / 2;
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getWork"
void EliminationForest::getWork(float& eliminationWork,
                                float& assemblyWork,
                                float& solutionWork,
                                Complex) const
{
  BEGIN_FUNCTION();

  const int* frontPointerItem = frontPointer_->getItem();
  const int* borderSizeItem = borderSize_->getItem();

  eliminationWork = assemblyWork = solutionWork = 0;

  /* For every node: */
  for (int u = 0; u < numberOfNodes_; u++)
  {
    float frontSize = (float)(frontPointerItem[u + 1] -
                        frontPointerItem[u]),
          borderSize = (float)borderSizeItem[u],
          order = frontSize + borderSize;

    /* For every column: */
    for (int j = frontPointerItem[u]; j < frontPointerItem[u + 1]; j++)
    {
      eliminationWork += 4 * order * order + 8 * order - 1;
      solutionWork += 16 * order - 5;
      order -= 1;
    }

    assemblyWork += borderSize * (borderSize + 1);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
void EliminationForest::initialize(const Array<int>* aColumnPointer,
                                   const Array<int>* aRowIndex,
                                   Permutation& p)
{
  BEGIN_FUNCTION();

  Permutation q(numberOfNodes_); /* Postorder permutation. */
  Array<int> ancestor(numberOfNodes_); /* Path compression array. */
  Array<int> map(numberOfNodes_); /* Postorder map. */
  Array<int> color(numberOfNodes_); /* Marker. */
  CHECK_ERROR1();

  int* parentItem = parent_->getItem();
  int* firstChildItem = firstChild_->getItem();
  int* nextSiblingItem = nextSibling_->getItem();
  int* frontPointerItem = frontPointer_->getItem();
  int* frontMapItem = frontMap_->getItem();
  int* borderSizeItem = borderSize_->getItem();
  int* ancestorItem = ancestor.getItem();
  int* mapItem = map.getItem();
  int* colorItem = color.getItem();

  const int* aColumnPointerItem = aColumnPointer->getItem();
  const int* aRowIndexItem = aRowIndex->getItem();
  int* pOldToNewItem = p.getOldToNew()->getItem();
  int* pNewToOldItem = p.getNewToOld()->getItem();
  int* qOldToNewItem = q.getOldToNew()->getItem();
  int* qNewToOldItem = q.getNewToOld()->getItem();

  /* Compute the parent links. */

  /* For every node: */
  {for (int u = 0; u < numberOfNodes_; u++)
  {
    /* Initialize the links. */
    parentItem[u] = -1;
    ancestorItem[u] = -1;

    /* For each column in the corresponding matrix row: */
    int i = pNewToOldItem[u];
    for (int p = aColumnPointerItem[i]; p < aColumnPointerItem[i + 1]; p++)
    {
      /* Get the corresponding node. */
      int v = pOldToNewItem[aRowIndexItem[p]];
      if (v < u)
      /* Node <u> is an ancestor of node <v>. */
      {
        /* First, find the root of the subtree containing node <v>. */
	while ((ancestorItem[v] != -1) && (ancestorItem[v] != u))
	{
	  int tmp = ancestorItem[v];
	  ancestorItem[v] = u;
	  v = tmp;
	}
	/* Then, make node <u> the root of this subtree. */
	if (ancestorItem[v] == -1)
	{
	  ancestorItem[v] = u;
	  parentItem[v] = u;
	}
      }
    }
  }}

  /* Compute the first child and next sibling links,
     and the root and the number of trees. */

  /* Initialize the root and the number of trees. */
  root_ = -1;
  numberOfTrees_ = 0;

  /* Initialize the links. */
  {for (int u = 0; u < numberOfNodes_; u++)
    firstChildItem[u] = nextSiblingItem[u] = -1;}

  /* For every node (backwards): */
  {for (int u = numberOfNodes_ - 1; u >= 0; u--)
  {
    int v = parentItem[u];
    if (v == -1)
    /* If the curent node has no parent it becomes the root
       of a new tree and the root of the forest as well. */
    {
       nextSiblingItem[u] = root_;
       root_ = u;
       numberOfTrees_++;
    }
    else
    /* Otherwise, the current node becomes the
       first child of its parent. */
    {
      nextSiblingItem[u] = firstChildItem[v];
      firstChildItem[v] = u;
    }
  }}

  /* Postorder the forest. */

  /* For every node (in postorder): */
  {for (int u = getFirstPostorderNode(), v = 0;
       u != -1;
       u = getNextPostorderNode(u), v++)
  {
    qOldToNewItem[u] = v;
    qNewToOldItem[v] = u;
  }}

  root_ = qOldToNewItem[root_];

  {for (int u = 0; u < numberOfNodes_; u++)
    if (parentItem[u] != -1)
      mapItem[qOldToNewItem[u]] = qOldToNewItem[parentItem[u]];
    else
      mapItem[qOldToNewItem[u]] = -1;}
  {for (int u = 0; u < numberOfNodes_; u++)
    parentItem[u] = mapItem[u];}

  {for (int u = 0; u < numberOfNodes_; u++)
    if (firstChildItem[u] != -1)
      mapItem[qOldToNewItem[u]] = qOldToNewItem[firstChildItem[u]];
    else
      mapItem[qOldToNewItem[u]] = -1;}
  {for (int u = 0; u < numberOfNodes_; u++)
    firstChildItem[u] = mapItem[u];}

  {for (int u = 0; u < numberOfNodes_; u++)
    if (nextSiblingItem[u] != -1)
      mapItem[qOldToNewItem[u]] = qOldToNewItem[nextSiblingItem[u]];
    else
      mapItem[qOldToNewItem[u]] = -1;}
  {for (int u = 0; u < numberOfNodes_; u++)
    nextSiblingItem[u] = mapItem[u];}

  p.compose(q);

  /* Compute the front pointers and the front map and initialize
     the border sizes and the marker. */

  /* For every node: */
  {for (int u = 0; u < numberOfNodes_; u++)
  {
    frontPointerItem[u] = u;
    frontMapItem[u] = u;
    borderSizeItem[u] = 0;
    colorItem[u] = -1;
  }}
  frontPointerItem[numberOfNodes_] = numberOfNodes_;

  /* Compute the border sizes. */

  /* For every node: */
  {for (int u = 0; u < numberOfNodes_; u++)
  {
    /* Initialize the marker. */
    colorItem[u] = u;

    /* For each column in the corresponding matrix row: */
    int i = pNewToOldItem[u];
    for (int p = aColumnPointerItem[i]; p < aColumnPointerItem[i + 1]; p++)
    {
      /* Get the corresponding node. */
      int v = pOldToNewItem[aRowIndexItem[p]];
      if (v < u)
      {
	/* Node <u> is an ancestor of node <v> and there is a factor
           nonzero entry in the row and column corresponding to
           to <u> and <v> respectively. On the path from <v> to <u>
           (<u> is the root of the current pruned subtree), mark
           all nodes as belonging to the current pruned subtree
           and update the counter. */
	while (colorItem[v] != u)
	{
	  borderSizeItem[v]++;
	  colorItem[v] = u;
	  v = parentItem[v];
	}
      }
    }
  }}

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
void EliminationForest::initialize(const Array<int>* lColumnPointer,
                                   const Array<int>* lRowIndex)
{
  BEGIN_FUNCTION();

  Permutation q(numberOfNodes_); /* Postorder permutation. */
  Array<int> map(numberOfNodes_); /* Postorder map. */
  CHECK_ERROR1();

  int* parentItem = parent_->getItem();
  int* firstChildItem = firstChild_->getItem();
  int* nextSiblingItem = nextSibling_->getItem();
  int* frontPointerItem = frontPointer_->getItem();
  int* frontMapItem = frontMap_->getItem();
  int* borderSizeItem = borderSize_->getItem();
  int* mapItem = map.getItem();

  const int* lColumnPointerItem = lColumnPointer->getItem();
  const int* lRowIndexItem = lRowIndex->getItem();
  int* qOldToNewItem = q.getOldToNew()->getItem();
  int* qNewToOldItem = q.getNewToOld()->getItem();

  /* Compute the parent links. */

  /* For every node: */
  {for (int u = 0; u < numberOfNodes_; u++)
    if (lColumnPointerItem[u] + 1 == lColumnPointerItem[u + 1])
      parentItem[u] = -1;
    else
    {
      /* Find the minimum row index. The row indices may not be
         in increasing order if pivoting was performed and the factors
         were not explicitly sorted. */
      int minimumRowIndex = order_;
      for (int p = lColumnPointerItem[u] + 1;
           p < lColumnPointerItem[u + 1];
           p++)
        if (minimumRowIndex > lRowIndexItem[p])
          minimumRowIndex = lRowIndexItem[p];
      parentItem[u] = minimumRowIndex;
    }}

  /* Compute the first child and next sibling links,
     and the root and the number of trees. */

  /* Initialize the root and the number of trees. */
  root_ = -1;
  numberOfTrees_ = 0;

  /* Initialize the links. */
  {for (int u = 0; u < numberOfNodes_; u++)
    firstChildItem[u] = nextSiblingItem[u] = -1;}

  /* For every node (backwards): */
  {for (int u = numberOfNodes_ - 1; u >= 0; u--)
  {
    int v = parentItem[u];
    if (v == -1)
    /* If the curent node has no parent it becomes the root
       of a new tree and the root of the forest as well. */
    {
       nextSiblingItem[u] = root_;
       root_ = u;
       numberOfTrees_++;
    }
    else
    /* Otherwise, the current node becomes the
       first child of its parent. */
    {
      nextSiblingItem[u] = firstChildItem[v];
      firstChildItem[v] = u;
    }
  }}

  /* Postorder the forest. */

  /* For every node (in postorder): */
  {for (int u = getFirstPostorderNode(), v = 0;
       u != -1;
       u = getNextPostorderNode(u), v++)
  {
    qOldToNewItem[u] = v;
    qNewToOldItem[v] = u;
  }}

  root_ = qOldToNewItem[root_];

  {for (int u = 0; u < numberOfNodes_; u++)
    if (parentItem[u] != -1)
      mapItem[qOldToNewItem[u]] = qOldToNewItem[parentItem[u]];
    else
      mapItem[qOldToNewItem[u]] = -1;}
  {for (int u = 0; u < numberOfNodes_; u++)
    parentItem[u] = mapItem[u];}

  {for (int u = 0; u < numberOfNodes_; u++)
    if (firstChildItem[u] != -1)
      mapItem[qOldToNewItem[u]] = qOldToNewItem[firstChildItem[u]];
    else
      mapItem[qOldToNewItem[u]] = -1;}
  {for (int u = 0; u < numberOfNodes_; u++)
    firstChildItem[u] = mapItem[u];}

  {for (int u = 0; u < numberOfNodes_; u++)
    if (nextSiblingItem[u] != -1)
      mapItem[qOldToNewItem[u]] = qOldToNewItem[nextSiblingItem[u]];
    else
      mapItem[qOldToNewItem[u]] = -1;}
  {for (int u = 0; u < numberOfNodes_; u++)
    nextSiblingItem[u] = mapItem[u];}

  /* Compute the front pointers, the front map and the border sizes. */

  /* For every node: */
  {for (int u = 0; u < numberOfNodes_; u++)
  {
    frontPointerItem[u] = u;
    frontMapItem[u] = u;
    borderSizeItem[u] = lColumnPointerItem[u + 1] - lColumnPointerItem[u] - 1;
  }}
  frontPointerItem[numberOfNodes_] = numberOfNodes_;

  END_FUNCTION();
}
