#ifndef VECTOR_H
#define VECTOR_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "Error.h"
#include "Utilities.h"

#undef __CLASS__
#define __CLASS__ "Vector"
template<class T>
class Vector
{
  private:
    int order_;
    Array<T>* entry_;

    Vector(const Vector<T>&);
    Vector<T>& operator=(const Vector<T>&);

    void allocate(void);
    void free(void);

  public:
    Vector();
    Vector(int order);
    virtual ~Vector();

    int getOrder(void) const
      {return order_;}
    Array<T>* getEntry(void)
      {return entry_;}
    const Array<T>* getEntry(void) const
      {return entry_;}

    void resize(int order);
    int getHeapSize(void) const;
    bool check(void) const;

    void print(void) const;
    void load(const char* fileName);
    void save(const char* fileName) const;

    void initialize(void);
    void add(const Vector<T>& x);
    void add(const Vector<T>& x, const Vector<T>& y);
    void subtract(const Vector<T>& x);
    void subtract(const Vector<T>& x, const Vector<T>& y);

    double getNorm1(void) const;

    void fill(T value)
      {entry_->fill(value);}
};

#undef __FUNC__
#define __FUNC__ "allocate"
template<class T>
void Vector<T>::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    entry_ = new Array<T>(order_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
template<class T>
void Vector<T>::free(void)
{
  BEGIN_FUNCTION();

  delete entry_;
  entry_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "Vector"
template<class T>
Vector<T>::Vector():
  order_(0),
  entry_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "Vector"
template<class T>
Vector<T>::Vector(int order):
  order_(order),
  entry_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~Vector"
template<class T>
Vector<T>::~Vector()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
template<class T>
void Vector<T>::resize(int order)
{
  BEGIN_FUNCTION();

  free();
  order_ = order;
  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
template<class T>
int Vector<T>::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = entry_->getHeapSize();

  END_FUNCTION();

  return heapSize;
}

#undef __FUNC__
#define __FUNC__ "check"
template<class T>
bool Vector<T>::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (order_ < 0)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "print"
template<class T>
void Vector<T>::print(void) const
{
  BEGIN_FUNCTION();

  const T* entryItem = entry_->getItem();

  cout << "VECTOR object" << endl
       << "Order: " << order_ << endl;

  cout << "Entries:";
  for (int i = 0; i < order_; i++)
    cout << " " << entryItem[i];
  cout << endl;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "load"
template<class T>
void Vector<T>::load(const char* fileName)
{
  BEGIN_FUNCTION();

  ifstream fin(fileName);
  if (!fin)
    SET_ERROR1(FileOpen);

  int order;
  fin >> order;
  resize(order);
  CHECK_ERROR1();

  T* entryItem = entry_->getItem();

  for (int i = 0; i < order_; i++)
    fin >> entryItem[i];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "save"
template<class T>
void Vector<T>::save(const char* fileName) const
{
  BEGIN_FUNCTION();

  ofstream fout(fileName);
  if (!fout)
    SET_ERROR1(FileOpen);

  T* entryItem = entry_->getItem();

  fout << order_ << endl;

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(1);

  for (int i = 0; i < order_; i++)
    fout << " " << entryItem[i];
  fout << endl;

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initialize"
template<class T>
void Vector<T>::initialize(void)
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();

  for (int i = 0; i < order_; i++)
    entryItem[i] = i + 1;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "add"
template<class T>
void Vector<T>::add(const Vector<T>& x)
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();
  const T* xEntryItem = x.getEntry()->getItem();

  for (int i = 0; i < order_; i++)
    entryItem[i] += xEntryItem[i];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "add"
template<class T>
void Vector<T>::add(const Vector<T>& x, const Vector<T>& y)
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();
  const T* xEntryItem = x.getEntry()->getItem();
  const T* yEntryItem = y.getEntry()->getItem();

  for (int i = 0; i < order_; i++)
    entryItem[i] = xEntryItem[i] + yEntryItem[i];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "subtract"
template<class T>
void Vector<T>::subtract(const Vector<T>& x)
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();
  const T* xEntryItem = x.getEntry()->getItem();

  for (int i = 0; i < order_; i++)
    entryItem[i] -= xEntryItem[i];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "subtract"
template<class T>
void Vector<T>::subtract(const Vector<T>& x, const Vector<T>& y)
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();
  const T* xEntryItem = x.getEntry()->getItem();
  const T* yEntryItem = y.getEntry()->getItem();

  for (int i = 0; i < order_; i++)
    entryItem[i] = xEntryItem[i] - yEntryItem[i];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getNorm1"
template<class T>
double Vector<T>::getNorm1(void) const
{
  BEGIN_FUNCTION();

  const T* entryItem = entry_->getItem();

  double norm = 0.0;
  for (int i = 0; i < order_; i++)
    norm += Abs(entryItem[i]);

  END_FUNCTION();

  return norm;
}

#endif
