#ifndef UPDATE_MATRIX_H
#define UPDATE_MATRIX_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "DenseMatrix.h"
#include "Error.h"

#undef __CLASS__
#define __CLASS__ "UpdateMatrix"
template<class T>
class UpdateMatrix: public DenseMatrix<T>
{
  private:
    int delayedFrontSize_;
    Array<int>* localToGlobal_;

    UpdateMatrix(const UpdateMatrix<T>&);
    UpdateMatrix<T>& operator=(const UpdateMatrix<T>&);

    void allocate(void);
    void free(void);

  public:
    UpdateMatrix();
    UpdateMatrix(int order, int delayedFrontSize);
    virtual ~UpdateMatrix();

    int getDelayedFrontSize(void) const
      {return delayedFrontSize_;}
    Array<int>* getLocalToGlobal(void)
      {return localToGlobal_;}
    const Array<int>* getLocalToGlobal(void) const
      {return localToGlobal_;}

    void resize(int order, int delayedFrontSize);
    int getHeapSize(void) const;
    bool check(void) const;

    void print(void) const;
};

#undef __FUNC__
#define __FUNC__ "allocate"
template<class T>
void UpdateMatrix<T>::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    localToGlobal_ = new Array<int>(order_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
template<class T>
void UpdateMatrix<T>::free(void)
{
  BEGIN_FUNCTION();

  delete localToGlobal_;
  localToGlobal_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "UpdateMatrix"
template<class T>
UpdateMatrix<T>::UpdateMatrix():
  delayedFrontSize_(0),
  localToGlobal_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "UpdateMatrix"
template<class T>
UpdateMatrix<T>::UpdateMatrix(int order, int delayedFrontSize):
  DenseMatrix<T>(order),
  delayedFrontSize_(delayedFrontSize),
  localToGlobal_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~UpdateMatrix"
template<class T>
UpdateMatrix<T>::~UpdateMatrix()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
template<class T>
void UpdateMatrix<T>::resize(int order, int delayedFrontSize)
{
  BEGIN_FUNCTION();

  DenseMatrix<T>::resize(order);
  free();
  delayedFrontSize_ = 0;
  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
template<class T>
int UpdateMatrix<T>::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = DenseMatrix<T>::getHeapSize() +
                 localToGlobal_->getHeapSize();

  END_FUNCTION();

  return heapSize;
}

#undef __FUNC__
#define __FUNC__ "check"
template<class T>
bool UpdateMatrix<T>::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (!DenseMatrix<T>::check())
    return false;
  if (delayedFrontSize_ < 0)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "print"
template<class T>
void UpdateMatrix<T>::print(void) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const int* localToGlobalItem = localToGlobal_->getItem();
  const T* entryItem = entry_->getItem();

  cout << "UPDATE_MATRIX object" << endl
       << "Order: " << order_ << endl
       << "Delayed front size: " << delayedFrontSize_ << endl;

  cout << "Column pointers:";
  {for (int j = 0; j < order_; j++)
    cout << " " << columnPointerItem[j] + 1;}
  cout << endl;

  cout << "Local to global map:";
  {for (int j = 0; j < order_; j++)
    cout << " " << localToGlobalItem[j] + 1;}
  cout << endl;

  for (int j = 0; j < order_; j++)
  {
    cout << "Entries(" << j << "):";
    for (int i = j, p = columnPointerItem[j]; i < order_; i++, p++)
      cout << " " << entryItem[p];
    cout << endl;
  }

  END_FUNCTION();
}

#endif
