#ifndef SPARSE_MATRIX_H
#define SPARSE_MATRIX_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "Error.h"
#include "Utilities.h"

enum Storage {Lower, Upper, Square};
enum Format {Compressed, Uncompressed};

#undef __CLASS__
#define __CLASS__ "SparseMatrix"
template<class T>
class SparseMatrix
{
  private:
    int order_;
    int numberOfOffDiagonals_;
    Array<int>* columnPointer_;
    Array<int>* diagonalPointer_;
    Array<int>* rowIndex_;
    Array<T>* entry_;
    Storage storage_; /* <Lower>, <Upper> or <Square>. */
    Format format_; /* <Compressed> or <Uncompressed>. */
    bool valued_;
    int precision_;

    SparseMatrix(const SparseMatrix<T>&);
    SparseMatrix<T>& operator=(const SparseMatrix<T>&);

    void allocate(void);
    void free(void);

    void loadTriangleCompressed(ifstream& fin);
    void loadSquareCompressed(ifstream& fin);
    void loadTriangleUncompressed(ifstream& fin);
    void loadSquareUncompressed(ifstream& fin);
    void saveLowerCompressed(ofstream& fout) const;
    void saveUpperCompressed(ofstream& fout) const;
    void saveSquareCompressed(ofstream& fout) const;
    void saveLowerUncompressed(ofstream& fout) const;
    void saveUpperUncompressed(ofstream& fout) const;
    void saveSquareUncompressed(ofstream& fout) const;
    void lowerToSquare(void);
    void upperToSquare(void);
    void generateEntries(void);

  public:
    SparseMatrix();
    SparseMatrix(int order, int numberOfOffDiagonals);
    virtual ~SparseMatrix();

    int getOrder(void) const
      {return order_;}
    int getNumberOfOffDiagonals(void) const
      {return numberOfOffDiagonals_;}
    Array<int>* getColumnPointer(void)
      {return columnPointer_;}
    const Array<int>* getColumnPointer(void) const
      {return columnPointer_;}
    Array<int>* getDiagonalPointer(void)
      {return diagonalPointer_;}
    const Array<int>* getDiagonalPointer(void) const
      {return diagonalPointer_;}
    Array<int>* getRowIndex(void)
      {return rowIndex_;}
    const Array<int>* getRowIndex(void) const
      {return rowIndex_;}
    Array<T>* getEntry(void)
      {return entry_;}
    const Array<T>* getEntry(void) const
      {return entry_;}
    Storage getStorage(void) const
      {return storage_;}
    Format getFormat(void) const
      {return format_;}
    bool getValued(void) const
      {return valued_;}
    int getPrecision(void) const
      {return valued_;}

    void setStorage(Storage storage)
      {storage_ = storage;}
    void setFormat(Format format)
      {format_ = format;}
    void setValued(bool valued)
      {valued_ = valued;}
    void setPrecision(int precision)
      {precision_ = precision;}

    void resize(int order, int numberOfOffDiagonals);
    int getHeapSize(void) const;
    bool check(void) const;

    void print(void) const;
    void load(const char* fileName);
    void save(const char* fileName) const;

    void saveNg (const char* fileName) const;

    double getNorm1(void) const;

    void sort(void);
};

#undef __FUNC__
#define __FUNC__ "allocate"
template<class T>
void SparseMatrix<T>::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    columnPointer_ = new Array<int>(order_ + 1);
    diagonalPointer_ = new Array<int>(order_);
    rowIndex_ = new Array<int>(order_ + 2 * numberOfOffDiagonals_);
    entry_ = new Array<T>(order_ + 2 * numberOfOffDiagonals_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
template<class T>
void SparseMatrix<T>::free(void)
{
  BEGIN_FUNCTION();

  delete columnPointer_;
  columnPointer_ = 0;
  delete diagonalPointer_;
  diagonalPointer_ = 0;
  delete rowIndex_;
  rowIndex_ = 0;
  delete entry_;
  entry_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "SparseMatrix"
template<class T>
SparseMatrix<T>::SparseMatrix():
  order_(0),
  numberOfOffDiagonals_(0),
  columnPointer_(0),
  diagonalPointer_(0),
  rowIndex_(0),
  entry_(0),
  storage_(Square),
  format_(Compressed),
  valued_(false),
  precision_(1)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "SparseMatrix"
template<class T>
SparseMatrix<T>::SparseMatrix(int order, int numberOfOffDiagonals):
  order_(order),
  numberOfOffDiagonals_(numberOfOffDiagonals),
  columnPointer_(0),
  diagonalPointer_(0),
  rowIndex_(0),
  entry_(0),
  storage_(Square),
  format_(Compressed),
  valued_(false),
  precision_(1)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~SparseMatrix"
template<class T>
SparseMatrix<T>::~SparseMatrix()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
template<class T>
void SparseMatrix<T>::resize(int order, int numberOfOffDiagonals)
{
  BEGIN_FUNCTION();

  free();
  order_ = order;
  numberOfOffDiagonals_ = numberOfOffDiagonals;
  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
template<class T>
int SparseMatrix<T>::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = columnPointer_->getHeapSize() +
                 diagonalPointer_->getHeapSize() +
                 rowIndex_->getHeapSize() +
                 entry_->getHeapSize();

  END_FUNCTION();

  return heapSize; 
}

#undef __FUNC__
#define __FUNC__ "check"
template<class T>
bool SparseMatrix<T>::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (order_ < 0)
    return false;
  if (numberOfOffDiagonals_ < 0)
    return false;
  if (order_ * order_ < order_ + 2 * numberOfOffDiagonals_)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "print"
template<class T>
void SparseMatrix<T>::print(void) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const int* diagonalPointerItem = diagonalPointer_->getItem();
  const int* rowIndexItem = rowIndex_->getItem();
  const T* entryItem = entry_->getItem();

  cout << "SPARSE_MATRIX object" << endl
       << "Order: " << order_ << endl
       << "Number of off-diagonals: " << numberOfOffDiagonals_ << endl;

  cout << "Column pointers:";
  {for (int j = 0; j <= order_; j++)
    cout << " " << columnPointerItem[j] + 1;}
  cout << endl;

  cout << "Diagonal pointers:";
  for (int j = 0; j < order_; j++)
    cout << " " << diagonalPointerItem[j] + 1;
  cout << endl;

  cout << "Row indices:";
  {for (int p = 0; p < order_ + 2 * numberOfOffDiagonals_; p++)
    cout << " " << rowIndexItem[p] + 1;}
  cout << endl;

  cout.setf(ios::left | ios::scientific | ios::showpoint);
  cout.precision(precision);

  cout << "Entries:";
  for (int p = 0; p < order_ + 2 * numberOfOffDiagonals_; p++)
    cout << " " << entryItem[p];
  cout << endl;

  cout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "load"
template<class T>
void SparseMatrix<T>::load(const char* fileName)
{
  BEGIN_FUNCTION();

  ifstream fin(fileName);
  if (!fin)
    SET_ERROR1(FileOpen);

  /* Read the header. */
  int order, numberOfOffDiagonals;
  fin >> order >> numberOfOffDiagonals;
  resize(order, numberOfOffDiagonals);
  CHECK_ERROR1();
  
  int tmp;
  fin >> tmp;
  if (tmp == 0)
    storage_ = Lower;
  else if (tmp == 1)
    storage_ = Upper;
  else
    storage_ = Square;

  fin >> tmp;
  if (tmp == 0)
    format_ = Compressed;
  else
    format_ = Uncompressed;

  fin >> tmp;
  if (valued_ && (tmp == 0))
    valued_ = false;

  /* Select the loading procedure. */
  switch (storage_)
  {
    case Lower:
      if (format_ == Compressed)
        loadTriangleCompressed(fin);
      else
        loadTriangleUncompressed(fin);
      CHECK_ERROR1();
      lowerToSquare();
      CHECK_ERROR1();
      break;

    case Upper:
      if (format_ == Compressed)
        loadTriangleCompressed(fin);
      else
        loadTriangleUncompressed(fin);
      CHECK_ERROR1();
      upperToSquare();
      CHECK_ERROR1();
      break;

    case Square:
      if (format_ == Compressed)
        loadSquareCompressed(fin);
      else
        loadSquareUncompressed(fin);
      CHECK_ERROR1();
      break;
  }

  /* Generate entries if they are missing. */
  if (!valued_)
    generateEntries();

  sort();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "save"
template<class T>
void SparseMatrix<T>::save(const char* fileName) const
{
  BEGIN_FUNCTION();

  ofstream fout(fileName);
  if (!fout)
    SET_ERROR1(FileOpen);

  /* Write the header. */
  fout << order_
       << " " << numberOfOffDiagonals_
       << " " << (int)storage_
       << " " << (int)format_
       << " " << (valued_ ? 1 : 0) << endl;

  /* Select the saving procedure. */
  switch (storage_)
  {
    case Lower:
      if (format_ == Compressed)
        saveLowerCompressed(fout);
      else
        saveLowerUncompressed(fout);
      break;

    case Upper:
      if (format_ == Compressed)
        saveUpperCompressed(fout);
      else
        saveUpperUncompressed(fout);
      break;

    case Square:
      if (format_ == Compressed)
        saveSquareCompressed(fout);
      else
        saveSquareUncompressed(fout);
      break;
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveNg"
template<class T>
void SparseMatrix<T>::saveNg(const char* fileName) const
{
  BEGIN_FUNCTION();

  ofstream fout(fileName);
  if (!fout)
    SET_ERROR1(FileOpen);

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();

  fout << "DATA" << endl;
  fout << order_ << " " << (2 * numberOfOffDiagonals_) << endl;

  {for (int j = 0; j <= order_; j++)
    fout << " " << columnPointerItem[j] - j + 1;}
  fout << endl;

  {for (int j = 0; j < order_; j++)
  {
    for (int p = columnPointerItem[j]; p < diagonalPointerItem[j]; p++)
      fout << " " << rowIndexItem[p] + 1;
    for (int p = diagonalPointerItem[j] + 1; p < columnPointerItem[j + 1]; p++)
      fout << " " << rowIndexItem[p] + 1;
  }}
  fout << endl;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getNorm1"
template<class T>
double SparseMatrix<T>::getNorm1(void) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const T* entryItem = entry_->getItem();

  double norm = 0.0;
  for (int j = 0; j < order_; j++)
  {
    double columnSum = 0.0;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      columnSum += Abs(entryItem[p]);
    if (columnSum > norm)
      norm = columnSum;
  }

  END_FUNCTION();

  return norm;
}

#undef __FUNC__
#define __FUNC__ "sort"
template<class T>
void SparseMatrix<T>::sort(void)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  /* Sort the rows. */
  {for (int j = 0; j < order_; j++)
    Sort(rowIndexItem, entryItem,
         columnPointerItem[j], columnPointerItem[j + 1]);}

  /* Compute the diagonal pointers. */
  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadTriangleCompressed"
template<class T>
void SparseMatrix<T>::loadTriangleCompressed(ifstream& fin)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  {for (int j = 0; j <= order_; j++)
  {
    fin >> columnPointerItem[j];
    columnPointerItem[j]--;
  }}

  {for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
  {
    fin >> rowIndexItem[p];
    rowIndexItem[p]--;
  }}

  if (valued_)
    for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
      fin >> entryItem[p];

  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadSquareCompressed"
template<class T>
void SparseMatrix<T>::loadSquareCompressed(ifstream& fin)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  {for (int j = 0; j <= order_; j++)
  {
    fin >> columnPointerItem[j];
    columnPointerItem[j]--;
  }}

  {for (int p = 0; p < order_ + 2 * numberOfOffDiagonals_; p++)
  {
    fin >> rowIndexItem[p];
    rowIndexItem[p]--;
  }}

  if (valued_)
    for (int p = 0; p < order_ + 2 * numberOfOffDiagonals_; p++)
      fin >> entryItem[p];

  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadTriangleUncompressed"
template<class T>
void SparseMatrix<T>::loadTriangleUncompressed(ifstream& fin)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  {for (int p = 0, j = -1; p < order_ + numberOfOffDiagonals_; p++)
  {
    int columnIndex;

    fin >> rowIndexItem[p] >> columnIndex;
    if (valued_)
      fin >> entryItem[p];

    rowIndexItem[p]--;
    columnIndex--;

    if (columnIndex != j)
    {
      j++;
      columnPointerItem[j] = p;
    }
  }}
  columnPointerItem[order_] = order_ + numberOfOffDiagonals_;

  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadSquareUncompressed"
template<class T>
void SparseMatrix<T>::loadSquareUncompressed(ifstream& fin)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  {for (int p = 0, j = -1; p < order_ + 2 * numberOfOffDiagonals_; p++)
  {
    int columnIndex;

    fin >> rowIndexItem[p] >> columnIndex;
    if (valued_)
      fin >> entryItem[p];

    rowIndexItem[p]--;
    columnIndex--;

    if (columnIndex != j)
    {
      j++;
      columnPointerItem[j] = p;
    }
  }}
  columnPointerItem[order_] = order_ + 2 * numberOfOffDiagonals_;

  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveLowerCompressed"
template<class T>
void SparseMatrix<T>::saveLowerCompressed(ofstream& fout) const
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  fout << " " << 1;
  {for (int j = 1, p = 1; j <= order_; j++)
    fout << " " << (p += columnPointerItem[j] - diagonalPointerItem[j - 1]);}
  fout << endl;

  {for (int j = 0; j < order_; j++)
    for (int p = diagonalPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      fout << " " << rowIndexItem[p] + 1;}
  fout << endl;

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(precision_);

  if (valued_)
  {
    for (int j = 0; j < order_; j++)
      for (int p = diagonalPointerItem[j]; p < columnPointerItem[j + 1]; p++)
        fout << " " << entryItem[p];
    fout << endl;
  }

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveUpperCompressed"
template<class T>
void SparseMatrix<T>::saveUpperCompressed(ofstream& fout) const
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  fout << " " << 1;
  {for (int j = 1, p = 1; j <= order_; j++)
    fout << " "
         << (p += diagonalPointerItem[j - 1] - columnPointerItem[j - 1]);}
  fout << endl;

  {for (int j = 0; j < order_; j++)
    for (int p = columnPointerItem[j]; p < diagonalPointerItem[j]; p++)
      fout << " " << rowIndexItem[p] + 1;}
  fout << endl;

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(precision_);

  if (valued_)
  {
    for (int j = 0; j < order_; j++)
      for (int p = columnPointerItem[j]; p <= diagonalPointerItem[j]; p++)
        fout << " " << entryItem[p];
    fout << endl;
  }

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveSquareCompressed"
template<class T>
void SparseMatrix<T>::saveSquareCompressed(ofstream& fout) const
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  {for (int j = 0; j <= order_; j++)
    fout << " " << columnPointerItem[j] + 1;}
  fout << endl;

  {for (int j = 0; j < order_; j++)
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      fout << " " << rowIndexItem[p] + 1;}
  fout << endl;

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(precision_);

  if (valued_)
  {
    for (int j = 0; j < order_; j++)
      for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
        fout << " " << entryItem[p];
    fout << endl;
  }

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveLowerUncompressed"
template<class T>
void SparseMatrix<T>::saveLowerUncompressed(ofstream& fout) const
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(precision_);

  for (int j = 0; j < order_; j++)
    for (int p = diagonalPointerItem[j]; p < columnPointerItem[j + 1]; p++)
    {
      fout << rowIndexItem[p] + 1 << " " << j + 1;
      if (valued_)
        fout << " " << entryItem[p];
      fout << endl;
    }

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveUpperUncompressed"
template<class T>
void SparseMatrix<T>::saveUpperUncompressed(ofstream& fout) const
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(precision_);

  for (int j = 0; j < order_; j++)
    for (int p = columnPointerItem[j]; p < diagonalPointerItem[j]; p++)
    {
      fout << rowIndexItem[p] + 1 << " " << j + 1;
      if (valued_)
        fout << " " << entryItem[p];
      fout << endl;
    }

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveSquareUncompressed"
template<class T>
void SparseMatrix<T>::saveSquareUncompressed(ofstream& fout) const
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(precision_);

  for (int j = 0; j < order_; j++)
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
    {
      fout << rowIndexItem[p] + 1 << " " << j + 1;
      if (valued_)
        fout << " " << entryItem[p];
      fout << endl;
    }

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "lowerToSquare"
template<class T>
void SparseMatrix<T>::lowerToSquare(void)
{
  BEGIN_FUNCTION();

  /* Convention: the diagonal is included in the lower triangle. */

  Array<int> lowerColumnPointer(order_ + 1);
  Array<int> upperColumnPointer(order_ + 1);
  Array<int> lowerRowIndex(order_ + numberOfOffDiagonals_);
  Array<int> upperRowIndex(numberOfOffDiagonals_);
  Array<T> lowerEntry(order_ + numberOfOffDiagonals_);
  Array<T> upperEntry(numberOfOffDiagonals_);
  Array<int> upperNumberOfEntries(order_);
  CHECK_ERROR1();

  int* lowerColumnPointerItem = lowerColumnPointer.getItem();
  int* upperColumnPointerItem = upperColumnPointer.getItem();
  int* lowerRowIndexItem = lowerRowIndex.getItem();
  int* upperRowIndexItem = upperRowIndex.getItem();
  T* lowerEntryItem = lowerEntry.getItem();
  T* upperEntryItem = upperEntry.getItem();
  int* upperNumberOfEntriesItem = upperNumberOfEntries.getItem();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  /* Transfer data. */
  {for (int j = 0; j <= order_; j++)
    lowerColumnPointerItem[j] = columnPointerItem[j];}
  {for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
    lowerRowIndexItem[p] = rowIndexItem[p];}
  {for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
    lowerEntryItem[p] = entryItem[p];}

  /* Compute the structure of the upper triangle. First, count the
     entries in each column of the upper triangle. */
  {for (int j = 0; j < order_; j++)
    upperNumberOfEntriesItem[j] = 0;}
  {for (int j = 0; j < order_; j++)
    for (int p = lowerColumnPointerItem[j];
         p < lowerColumnPointerItem[j + 1];
         p++)
    {
      int i = lowerRowIndexItem[p];
      if (i == j)
        continue;
      upperNumberOfEntriesItem[i]++;
    }}

  /* Then, compute the upper triangle column pointers. */
  upperColumnPointerItem[0] = 0;
  {for (int j = 0; j < order_; j++)
    upperColumnPointerItem[j + 1] = upperColumnPointerItem[j] +
                                    upperNumberOfEntriesItem[j];}

  /* Then, compute the upper triangle row indices and entries. */
  {for (int j = 0; j < order_; j++)
    for (int p = lowerColumnPointerItem[j];
         p < lowerColumnPointerItem[j + 1];
         p++)
    {
      int i = lowerRowIndexItem[p];
      if (i == j)
        continue;
      upperRowIndexItem[upperColumnPointerItem[i]] = j;
      if (valued_)
#ifndef HERMITIAN
        upperEntryItem[upperColumnPointerItem[i]] = lowerEntryItem[p];
#else
        upperEntryItem[upperColumnPointerItem[i]] = Conj(lowerEntryItem[p]);
#endif
      upperColumnPointerItem[i]++;
    }}

  /* Finally, recompute the upper triangle column pointers. */
  upperColumnPointerItem[0] = 0;
  {for (int j = 0; j < order_; j++)
    upperColumnPointerItem[j + 1] = upperColumnPointerItem[j] +
                                    upperNumberOfEntriesItem[j];}

  /* Now, compute the square from the two triangles.
     First, compute the column pointers. */
  columnPointerItem[0] = 0;
  {for (int j = 0; j < order_; j++)
    columnPointerItem[j + 1] =
      columnPointerItem[j] +
      lowerColumnPointerItem[j + 1] - lowerColumnPointerItem[j] +
      upperColumnPointerItem[j + 1] - upperColumnPointerItem[j];}

  /* Then, compute the row indices and entries. */
  {for (int j = 0; j < order_; j++)
  {
    int p = columnPointerItem[j];
    for (int up = upperColumnPointerItem[j];
         up < upperColumnPointerItem[j + 1];
         up++, p++)
    {
      rowIndexItem[p] = upperRowIndexItem[up];
      if (valued_)
        entryItem[p] = upperEntryItem[up];
    }
    for (int lp = lowerColumnPointerItem[j];
         lp < lowerColumnPointerItem[j + 1];
         lp++, p++)
    {
      rowIndexItem[p] = lowerRowIndexItem[lp];
      if (valued_)
        entryItem[p] = lowerEntryItem[lp];
    }
  }}

  /* Finally, compute the diagonal pointers. */
  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "upperToSquare"
template<class T>
void SparseMatrix<T>::upperToSquare(void)
{
  BEGIN_FUNCTION();

  /* Convention: the diagonal is included in the upper triangle. */

  Array<int> lowerColumnPointer(order_ + 1);
  Array<int> upperColumnPointer(order_ + 1);
  Array<int> lowerRowIndex(numberOfOffDiagonals_);
  Array<int> upperRowIndex(order_ + numberOfOffDiagonals_);
  Array<T> lowerEntry(numberOfOffDiagonals_);
  Array<T> upperEntry(order_ + numberOfOffDiagonals_);
  Array<int> lowerNumberOfEntries(order_);
  CHECK_ERROR1();

  int* lowerColumnPointerItem = lowerColumnPointer.getItem();
  int* upperColumnPointerItem = upperColumnPointer.getItem();
  int* lowerRowIndexItem = lowerRowIndex.getItem();
  int* upperRowIndexItem = upperRowIndex.getItem();
  T* lowerEntryItem = lowerEntry.getItem();
  T* upperEntryItem = upperEntry.getItem();
  int* lowerNumberOfEntriesItem = lowerNumberOfEntries.getItem();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  /* Transfer data. */
  {for (int j = 0; j <= order_; j++)
    upperColumnPointerItem[j] = columnPointerItem[j];}
  {for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
    upperRowIndexItem[p] = rowIndexItem[p];}
  {for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
    upperEntryItem[p] = entryItem[p];}

  /* Compute the structure of the lower triangle. First, count the
     entries in each column of the lower triangle. */
  {for (int j = 0; j < order_; j++)
    lowerNumberOfEntriesItem[j] = 0;}
  {for (int j = 0; j < order_; j++)
    for (int p = upperColumnPointerItem[j];
         p < upperColumnPointerItem[j + 1];
         p++)
    {
      int i = upperRowIndexItem[p];
      if (i == j)
        continue;
      lowerNumberOfEntriesItem[i]++;
    }}

  /* Then, compute the lower triangle column pointers. */
  lowerColumnPointerItem[0] = 0;
  {for (int j = 0; j < order_; j++)
    lowerColumnPointerItem[j + 1] = lowerColumnPointerItem[j] +
                                    lowerNumberOfEntriesItem[j];}

  /* Then, compute the lower triangle row indices and the entries. */
  {for (int j = 0; j < order_; j++)
    for (int p = upperColumnPointerItem[j];
         p < upperColumnPointerItem[j + 1];
         p++)
    {
      int i = upperRowIndexItem[p];
      if (i == j)
        continue;
      lowerRowIndexItem[lowerColumnPointerItem[i]] = j;
      if (valued_)
#ifndef HERMITIAN
        lowerEntryItem[lowerColumnPointerItem[i]] = upperEntryItem[p];
#else
        lowerEntryItem[lowerColumnPointerItem[i]] = Conj(upperEntryItem[p]);
#endif
      lowerColumnPointerItem[i]++;
    }}

  /* Finally, recompute the lower triangle column pointers. */
  lowerColumnPointerItem[0] = 0;
  {for (int j = 0; j < order_; j++)
    lowerColumnPointerItem[j + 1] = lowerColumnPointerItem[j] +
                                    lowerNumberOfEntriesItem[j];}

  /* Now, compute the square from the two triangles.
     First, compute the column pointers. */
  columnPointerItem[0] = 0;
  {for (int j = 0; j < order_; j++)
    columnPointerItem[j + 1] =
      columnPointerItem[j] +
      lowerColumnPointerItem[j + 1] - lowerColumnPointerItem[j] +
      upperColumnPointerItem[j + 1] - upperColumnPointerItem[j];}

  /* Then, compute the row indices and entries. */
  {for (int j = 0; j < order_; j++)
  {
    int p = columnPointerItem[j];
    for (int up = upperColumnPointerItem[j];
         up < upperColumnPointerItem[j + 1];
         up++, p++)
    {
      rowIndexItem[p] = upperRowIndexItem[up];
      if (valued_)
        entryItem[p] = upperEntryItem[up];
    }
    for (int lp = lowerColumnPointerItem[j];
         lp < lowerColumnPointerItem[j + 1];
         lp++, p++)
    {
      rowIndexItem[p] = lowerRowIndexItem[lp];
      if (valued_)
        entryItem[p] = lowerEntryItem[lp];
    }
  }}

  /* Finally, compute the diagonal pointers. */
  for (int j = 0; j < order_; j++)
  {
    diagonalPointerItem[j] = -1;
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
      {
        diagonalPointerItem[j] = p;
        break;
      }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "generateEntries"
template<class T>
void SparseMatrix<T>::generateEntries(void)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* diagonalPointerItem = diagonalPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  for (int j = 0; j < order_; j++)
  {
    for (int p = columnPointerItem[j]; p < columnPointerItem[j + 1]; p++)
      if (rowIndexItem[p] == j)
        entryItem[diagonalPointerItem[j]] =
          columnPointerItem[j + 1] - columnPointerItem[j];
      else
        entryItem[p] = 1;
  }

  END_FUNCTION();
}

#endif
