#ifndef SPARSE_FACTORS_H
#define SPARSE_FACTORS_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "Error.h"
#include "Utilities.h"

#undef __CLASS__
#define __CLASS__ "SparseFactors"
template<class T>
class SparseFactors
{
  private:
    int order_;
    int numberOfOffDiagonals_;
    int numberOfSwappedColumns_;
    int numberOfDelayedColumns_;
    int numberOf1x1Pivots_;
    int numberOf2x2Pivots_;
    int numberOfEliminatedColumns_;
    Array<int>* columnPointer_;
    Array<int>* rowIndex_;
    Array<T>* entry_;
    Array<int>* pivotType_;

    SparseFactors(const SparseFactors<T>&);
    SparseFactors<T>& operator=(const SparseFactors<T>&);

    void allocate(void);
    void free(void);

  public:
    SparseFactors();
    SparseFactors(int order, int numberOfOffDiagonals);
    virtual ~SparseFactors();

    int getOrder(void) const
      {return order_;}
    int getNumberOfOffDiagonals(void) const
      {return numberOfOffDiagonals_;}
    int getNumberOfSwappedColumns(void) const
      {return numberOfSwappedColumns_;}
    int getNumberOfDelayedColumns(void) const
      {return numberOfDelayedColumns_;}
    int getNumberOf1x1Pivots(void) const
      {return numberOf1x1Pivots_;}
    int getNumberOf2x2Pivots(void) const
      {return numberOf2x2Pivots_;}
    int getNumberOfEliminatedColumns(void) const
      {return numberOfEliminatedColumns_;}
    Array<int>* getColumnPointer(void)
      {return columnPointer_;}
    const Array<int>* getColumnPointer(void) const
      {return columnPointer_;}
    Array<int>* getRowIndex(void)
      {return rowIndex_;}
    const Array<int>* getRowIndex(void) const
      {return rowIndex_;}
    Array<T>* getEntry(void)
      {return entry_;}
    const Array<T>* getEntry(void) const
      {return entry_;}
    Array<int>* getPivotType(void)
      {return pivotType_;}
    const Array<int>* getPivotType(void) const
      {return pivotType_;}

    void setNumberOfOffDiagonals(int numberOfOffDiagonals)
      {numberOfOffDiagonals_ = numberOfOffDiagonals;}
    void setNumberOfSwappedColumns(int numberOfSwappedColumns)
      {numberOfSwappedColumns_ = numberOfSwappedColumns;}
    void setNumberOfDelayedColumns(int numberOfDelayedColumns)
      {numberOfDelayedColumns_ = numberOfDelayedColumns;}
    void setNumberOf1x1Pivots(int numberOf1x1Pivots)
      {numberOf1x1Pivots_ = numberOf1x1Pivots;}
    void setNumberOf2x2Pivots(int numberOf2x2Pivots)
      {numberOf2x2Pivots_ = numberOf2x2Pivots;}
    void setNumberOfEliminatedColumns(int numberOfEliminatedColumns)
      {numberOfEliminatedColumns_ = numberOfEliminatedColumns;}

    void resize(int order, int numberOfOffDiagonals);
    int getHeapSize(void) const;
    bool check(void) const;

    void print(void) const;
    void save(const char* fileName) const;

    void sort(void);
};

#undef __FUNC__
#define __FUNC__ "allocate"
template<class T>
void SparseFactors<T>::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    columnPointer_ = new Array<int>(order_ + 1);
    rowIndex_ = new Array<int>(order_ + numberOfOffDiagonals_);
    entry_ = new Array<T>(order_ + numberOfOffDiagonals_);
    pivotType_ = new Array<int>(order_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
template<class T>
void SparseFactors<T>::free(void)
{
  BEGIN_FUNCTION();

  delete columnPointer_;
  columnPointer_ = 0;
  delete rowIndex_;
  rowIndex_ = 0;
  delete entry_;
  entry_ = 0;
  delete pivotType_;
  pivotType_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "SparseFactors"
template<class T>
SparseFactors<T>::SparseFactors():
  order_(0),
  numberOfOffDiagonals_(0),
  numberOfSwappedColumns_(0),
  numberOfDelayedColumns_(0),
  numberOf1x1Pivots_(0),
  numberOf2x2Pivots_(0),
  numberOfEliminatedColumns_(0),
  columnPointer_(0),
  rowIndex_(0),
  entry_(0),
  pivotType_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "SparseFactors"
template<class T>
SparseFactors<T>::SparseFactors(int order, int numberOfOffDiagonals):
  order_(order),
  numberOfOffDiagonals_(numberOfOffDiagonals),
  numberOfSwappedColumns_(0),
  numberOfDelayedColumns_(0),
  numberOf1x1Pivots_(0),
  numberOf2x2Pivots_(0),
  numberOfEliminatedColumns_(0),
  columnPointer_(0),
  rowIndex_(0),
  entry_(0),
  pivotType_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~SparseFactors"
template<class T>
SparseFactors<T>::~SparseFactors()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
template<class T>
void SparseFactors<T>::resize(int order, int numberOfOffDiagonals)
{
  BEGIN_FUNCTION();

  free();
  order_ = order;
  numberOfOffDiagonals_ = numberOfOffDiagonals;
  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
template<class T>
int SparseFactors<T>::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = columnPointer_->getHeapSize() +
                 rowIndex_->getHeapSize() +
                 entry_->getHeapSize() +
                 pivotType_->getHeapSize();

  END_FUNCTION();

  return heapSize;
}

#undef __FUNC__
#define __FUNC__ "check"
template<class T>
bool SparseFactors<T>::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (order_ < 0)
    return false;
  if (numberOfOffDiagonals_ < 0)
    return false;
  if (order_ * order_ < order_ + 2 * numberOfOffDiagonals_)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "print"
template<class T>
void SparseFactors<T>::print(void) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const int* rowIndexItem = rowIndex_->getItem();
  const T* entryItem = entry_->getItem();
  const int* pivotTypeItem = pivotType_->getItem();

  cout << "SPARSE_FACTORS object" << endl
       << "Order: " << order_ << endl
       << "Number of off-diagonals: " << numberOfOffDiagonals_ << endl;

  cout << "Column pointers:";
  {for (int j = 0; j <= order_; j++)
    cout << " " << columnPointerItem[j] + 1;}
  cout << endl;

  cout << "Row indices:";
  {for (int j = 0; j < order_ + numberOfOffDiagonals_; j++)
    cout << " " << rowIndexItem[j] + 1;}
  cout << endl;

  cout << "Entries:";
  {for (int p = 0; p < order_ + numberOfOffDiagonals_; p++)
    cout << " " << entryItem[p];}
  cout << endl;

  cout << "Pivot types:";
  {for (int j = 0; j < order_; j++)
    cout << " " << pivotTypeItem[j];}
  cout << endl;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "save"
template<class T>
void SparseFactors<T>::save(const char* fileName) const
{
  BEGIN_FUNCTION();

  ofstream fout(fileName);
  if (!fout)
    SET_ERROR1(FileOpen);

  const int* columnPointerItem = columnPointer_->getItem();
  const int* rowIndexItem = rowIndex_->getItem();
  const T* entryItem = entry_->getItem();

  {for (int j = 0; j <= order_; j++)
    fout << " " << columnPointerItem[j] + 1;}
  fout << endl;

  {for (int j = 0; j < order_; j++)
  {
    for (int p = columnPointerItem[j];
         p < columnPointerItem[j + 1];
         p++)
      fout << " " << rowIndexItem[p] + 1;
    fout << endl;
  }}

  fout.setf(ios::left | ios::scientific | ios::showpoint);
  fout.precision(1);

  {for (int j = 0; j < order_; j++)
  {
    for (int p = columnPointerItem[j];
         p < columnPointerItem[j + 1];
         p++)
      fout << " " << entryItem[p];
    fout << endl;
  }}

  fout.unsetf(ios::left | ios::scientific | ios::showpoint);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "sort"
template<class T>
void SparseFactors<T>::sort(void)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  int* rowIndexItem = rowIndex_->getItem();
  T* entryItem = entry_->getItem();

  for (int j = 0; j < order_; j++)
    Sort(rowIndexItem, entryItem,
         columnPointerItem[j], columnPointerItem[j + 1]);

  END_FUNCTION();
}

#endif
