#ifndef MULTIFRONTAL_FACTOR_H
#define MULTIFRONTAL_FACTOR_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "SparseMatrix.h"
#include "Permutation.h"
#include "EliminationForest.h"
#include "Symbolic.h"
#include "SparseFactors.h"
#include "UpdateMatrix.h"
#include "FrontalMatrix.h"
#include "UpdateStack.h"
#include "Error.h"

#undef __CLASS__
#define __CLASS__ "MultifrontalFactor"
template<class T>
class MultifrontalFactor
{
  private:
    const SparseMatrix<T>* a_;
    Permutation* p_;
    const EliminationForest* f_;
    const Symbolic* s_;
    UpdateStack<T>* u_;
    SparseFactors<T>* l_;
    FrontalMatrix<T>* fm_;
    UpdateMatrix<T>* um_;
    double threshold_;

    MultifrontalFactor(const MultifrontalFactor<T>&);
    MultifrontalFactor<T>& operator=(const MultifrontalFactor<T>&);

  public:
    MultifrontalFactor();
    virtual ~MultifrontalFactor();

    void setA(const SparseMatrix<T>& a)
      {a_ = &a;}
    void setP(Permutation& p)
      {p_ = &p;}
    void setF(const EliminationForest& f)
      {f_ = &f;}
    void setS(const Symbolic& s)
      {s_ = &s;}
    void setU(UpdateStack<T>& u)
      {u_ = &u;}
    void setL(SparseFactors<T>& l)
      {l_ = &l;}
    void setThreshold(double threshold)
      {threshold_ = threshold;}

    void run(void);
};

#undef __FUNC__
#define __FUNC__ "MultifrontalFactor"
template<class T>
MultifrontalFactor<T>::MultifrontalFactor():
  a_(0),
  p_(0),
  f_(0),
  s_(0),
  u_(0),
  l_(0),
  fm_(0),
  um_(0),
  threshold_(0)
{
  BEGIN_FUNCTION();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~MultifrontalFactor"
template<class T>
MultifrontalFactor<T>::~MultifrontalFactor()
{
  BEGIN_FUNCTION();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "run"
template<class T>
void MultifrontalFactor<T>::run(void)
{
  BEGIN_FUNCTION();

  if (a_ == 0)
    SET_ERROR1(InvalidInput);

  if (p_ == 0)
    SET_ERROR1(InvalidInput);

  if (f_ == 0)
    SET_ERROR1(InvalidInput);

  if (s_ == 0)
    SET_ERROR1(InvalidInput);

  if (u_ == 0)
    SET_ERROR1(InvalidInput);

  if (l_ == 0)
    SET_ERROR1(InvalidInput);

  if (a_->getOrder() != p_->getOrder())
    SET_ERROR1(InvalidInput);

  if (a_->getOrder() != f_->getOrder())
    SET_ERROR1(InvalidInput);

  if (a_->getOrder() != l_->getOrder())
    SET_ERROR1(InvalidInput);

  if (f_->getNumberOfNodes() != s_->getNumberOfFronts())
    SET_ERROR1(InvalidInput);

  if (f_->getNumberOfNodes() != u_->getMaximumSize())
    SET_ERROR1(InvalidInput);

  Permutation q(a_->getOrder());
  Array<int> globalToLocal(a_->getOrder());
  CHECK_ERROR1();

  const int* fParentItem = f_->getParent()->getItem();
  const int* fFirstChildItem = f_->getFirstChild()->getItem();
  const int* fNextSiblingItem = f_->getNextSibling()->getItem();
  const int* fFrontPointerItem = f_->getFrontPointer()->getItem();
  const int* fBorderSizeItem = f_->getBorderSize()->getItem();
  UpdateMatrix<T>** uEntryItem = u_->getEntry()->getItem();
  int *lColumnPointerItem = l_->getColumnPointer()->getItem();
  int *qOldToNewItem = q.getOldToNew()->getItem();

  /* Initialize <l_>. */
  lColumnPointerItem[0] = 0;
  l_->setNumberOfOffDiagonals(0);
  l_->setNumberOfSwappedColumns(0);
  l_->setNumberOfDelayedColumns(0);
  l_->setNumberOf1x1Pivots(0);
  l_->setNumberOf2x2Pivots(0);
  l_->setNumberOfEliminatedColumns(0);

  q.initialize();

  bool pivot = false;
  if ((threshold_ > 0) && (threshold_ < 1))
    pivot = true;

  /* For every node (in postorder): */
  for (int x = f_->getFirstPostorderNode();
       x != -1;
       x = f_->getNextPostorderNode(x))
  {
    /* Compute the number of delayed columns. */
    int delayedFrontSize = 0;
    if (pivot)
      /* For every child of the current node: */
      for (int y = fFirstChildItem[x], up = u_->getCurrentSize() - 1;
            y != -1;
            y = fNextSiblingItem[y], up--)
        delayedFrontSize += uEntryItem[up]->getDelayedFrontSize();

    /* Create the frontal matrix. */
    try
    {
      fm_ = new FrontalMatrix<T>(fFrontPointerItem[x + 1] -
                                   fFrontPointerItem[x] +
                                   fBorderSizeItem[x] +
                                   delayedFrontSize, /* Order. */
                                 fFrontPointerItem[x + 1] -
                                   fFrontPointerItem[x], /* Original. */
                                 fFrontPointerItem[x + 1] -
                                   fFrontPointerItem[x] +
                                   delayedFrontSize); /* Modified. */
    }
    catch (...)
    {
      SET_ERROR1(MemoryAllocation);
    }
    if (ErrorCode != None)
      delete fm_;
    CHECK_ERROR1();

    /* Load indices (original, update and delayed). */
    fm_->loadIndices(*s_, x);
    if (pivot)
    {
      for (int y = fFirstChildItem[x], up = u_->getCurrentSize() - 1;
            y != -1;
            y = fNextSiblingItem[y], up--)
        fm_->loadIndices(*uEntryItem[up]);
      fm_->orderIndices(*l_, q);
    }

    /* Compute global to local map. */
    fm_->computeGlobalToLocal(globalToLocal);

    /* Zero out the frontal matrix. */
    fm_->fill(0);

    /* Load entries (original, update and delayed). */
    fm_->loadEntries(*a_, *p_, globalToLocal);
    for (int y = fFirstChildItem[x]; y != -1; y = fNextSiblingItem[y])
    {
      /* Pop the update matrix. */
      um_ = u_->pop();
      fm_->loadEntries(*um_, globalToLocal);
      /* Destroy the update matrix. */
      delete um_;
    }

    /* Perform partial factorization. */
    if (pivot)
      fm_->factor(*l_, globalToLocal, q, threshold_);
    else
      fm_->factor(*l_);
    if (ErrorCode != None)
      delete fm_;
    CHECK_ERROR1();

    if (pivot && (fParentItem[x] == -1) &&
        (fm_->getFinalFrontSize() < fm_->getModifiedFrontSize()))
    {
      delete fm_;
      SET_ERROR1(RootDelay);
    }

    /* Create the update matrix. */
    try
    {
      um_ = new UpdateMatrix<T>(fm_->getOrder() -
                                  fm_->getFinalFrontSize(),
                                fm_->getModifiedFrontSize() -
                                  fm_->getFinalFrontSize());
    }
    catch (...)
    {
      delete fm_;
      SET_ERROR1(MemoryAllocation);
    }
    if (ErrorCode != None)
    {
      delete fm_;
      delete um_;
    }
    CHECK_ERROR1();

    int numberOfEntries = fm_->getOrder() * (fm_->getOrder() + 1) / 2 -
                          um_->getOrder() * (um_->getOrder() + 1) / 2;

    if (pivot)
    {
      while (l_->getRowIndex()->getSize() -
             l_->getNumberOfEliminatedColumns() -
             l_->getNumberOfOffDiagonals() < numberOfEntries)
      {
        l_->getRowIndex()->extend();
        l_->getEntry()->extend();
        if (ErrorCode != None)
        {
          delete fm_;
          delete um_;
        }
        CHECK_ERROR1();
      }
    }

    /* Save indices (original, update and delayed). */
    fm_->saveIndices(*l_);
    fm_->saveIndices(*um_);

    /* Save entries (original, update and delayed). */
    fm_->saveEntries(*l_);
    fm_->saveEntries(*um_);

    l_->setNumberOfOffDiagonals(l_->getNumberOfOffDiagonals() +
                                numberOfEntries - fm_->getFinalFrontSize());
    l_->setNumberOfEliminatedColumns(l_->getNumberOfEliminatedColumns() +
                                     fm_->getFinalFrontSize());

    /* Destroy the frontal matrix. */
    delete fm_;

    /* Push the update matrix. */
    u_->push(um_);
  }

  if (pivot)
  {
    int *lRowIndexItem = l_->getRowIndex()->getItem();

    for (int lp = 0;
         lp < l_->getOrder() + l_->getNumberOfOffDiagonals();
         lp++)
      lRowIndexItem[lp] = qOldToNewItem[lRowIndexItem[lp]];

    p_->compose(q);
  }

  END_FUNCTION();
}

#endif
