#ifndef FRONTAL_MATRIX_H
#define FRONTAL_MATRIX_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "DenseMatrix.h"
#include "SparseMatrix.h"
#include "Permutation.h"
#include "Symbolic.h"
#include "SparseFactors.h"
#include "UpdateMatrix.h"
#include "Queue.h"
#include "Error.h"
#include "Utilities.h"
#include "Kernels.h"

#undef __CLASS__
#define __CLASS__ "FrontalMatrix"
template<class T>
class FrontalMatrix: public DenseMatrix<T>
{
  private:
    int originalFrontSize_;
    int modifiedFrontSize_;
    int finalFrontSize_;
    int delayedFrontSize_;
    Array<int>* localToGlobal_;

    FrontalMatrix(const FrontalMatrix<T>&);
    FrontalMatrix<T>& operator=(const FrontalMatrix<T>&);

    void allocate(void);
    void free(void);

  public:
    FrontalMatrix();
    FrontalMatrix(int order, int originalFrontSize, int modifiedFrontSize);
    virtual ~FrontalMatrix();

    int getOriginalFrontSize(void) const
      {return originalFrontSize_;}
    int getModifiedFrontSize(void) const
      {return modifiedFrontSize_;}
    int getFinalFrontSize(void) const
      {return finalFrontSize_;}
    Array<int>* getLocalToGlobal(void)
      {return localToGlobal_;}
    const Array<int>* getLocalToGlobal(void) const
      {return localToGlobal_;}

    void resize(int order, int originalFrontSize, int modifiedFrontSize);
    int getHeapSize(void) const;
    bool check(void) const;

    void print(void) const;

    void computeGlobalToLocal(Array<int>& globalToLocal);
    void loadIndices(const Symbolic& s, int front);
    void loadIndices(const UpdateMatrix<T>& um);
    void orderIndices(const SparseFactors<T>& l,
                      Permutation& p) const;
    void loadEntries(const SparseMatrix<T>& a,
                     const Permutation& p,
                     const Array<int>& globalToLocal);
    void loadEntries(const UpdateMatrix<T>& um,
                     const Array<int>& globalToLocal);
    void saveIndices(SparseFactors<T>& l) const;
    void saveIndices(UpdateMatrix<T>& um) const;
    void saveEntries(SparseFactors<T>& l) const;
    void saveEntries(UpdateMatrix<T>& um) const;
    void factor(SparseFactors<T>& l);
    void factor(SparseFactors<T>& l,
                Array<int>& globalToLocal,
                Permutation& p,
                double threshold);
    void swapColumns(int column1, int column2,
                     Permutation &p,
                     Array<int>& globalToLocal);
    double getMaximumOffDiagonal(int column,
                                 int numberOfEliminatedColumns) const;
    double getMaximumOffDiagonal(int column,
                                 int numberOfEliminatedColumns,
                                 int &row) const;
};

#undef __FUNC__
#define __FUNC__ "allocate"
template<class T>
void FrontalMatrix<T>::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    localToGlobal_ = new Array<int>(order_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
template<class T>
void FrontalMatrix<T>::free(void)
{
  BEGIN_FUNCTION();

  delete localToGlobal_;
  localToGlobal_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "FrontalMatrix"
template<class T>
FrontalMatrix<T>::FrontalMatrix():
  originalFrontSize_(0),
  modifiedFrontSize_(0),
  finalFrontSize_(0),
  delayedFrontSize_(0),
  localToGlobal_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "FrontalMatrix"
template<class T>
FrontalMatrix<T>::FrontalMatrix(int order, int originalFrontSize,
                                int modifiedFrontSize):
  DenseMatrix<T>(order),
  originalFrontSize_(originalFrontSize),
  modifiedFrontSize_(modifiedFrontSize),
  finalFrontSize_(0),
  delayedFrontSize_(0),
  localToGlobal_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~FrontalMatrix"
template<class T>
FrontalMatrix<T>::~FrontalMatrix()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
template<class T>
void FrontalMatrix<T>::resize(int order, int originalFrontSize,
                              int modifiedFrontSize)
{
  BEGIN_FUNCTION();

  DenseMatrix<T>::resize(order);
  free();
  originalFrontSize_ = originalFrontSize;
  modifiedFrontSize_ = modifiedFrontSize;
  delayedFrontSize_ = 0;
  finalFrontSize_ = 0;
  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
template<class T>
int FrontalMatrix<T>::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = DenseMatrix<T>::getHeapSize() +
                 localToGlobal_->getHeapSize();

  END_FUNCTION();

  return heapSize;
}

#undef __FUNC__
#define __FUNC__ "check"
template<class T>
bool FrontalMatrix<T>::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (!DenseMatrix<T>::check())
    return false;
  if (originalFrontSize_ < 0)
    return false;
  if (modifiedFrontSize_ < 0)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "print"
template<class T>
void FrontalMatrix<T>::print(void) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const int* localToGlobalItem = localToGlobal_->getItem();
  const T* entryItem = entry_->getItem();

  cout << "FRONTAL_MATRIX object" << endl
       << "Order: " << order_ << endl
       << "Original front size: " << originalFrontSize_ << endl
       << "Modified front size: " << modifiedFrontSize_ << endl;

  cout << "Column pointers:";
  {for (int j = 0; j < order_; j++)
    cout << " " << columnPointerItem[j] + 1;}
  cout << endl;

  cout << "Local to global map:";
  {for (int j = 0; j < order_; j++)
    cout << " " << localToGlobalItem[j] + 1;}
  cout << endl;

  for (int j = 0; j < order_; j++)
  {
    cout << "Entries(" << j << "):";
    for (int i = j, p = columnPointerItem[j]; i < order_; i++, p++)
      cout << " " << entryItem[p];
    cout << endl;
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "computeGlobalToLocal"
template<class T>
void FrontalMatrix<T>::computeGlobalToLocal(Array<int>& globalToLocal)
{
  BEGIN_FUNCTION();

  int* globalToLocalItem = globalToLocal.getItem();
  int* localToGlobalItem = localToGlobal_->getItem();

  int order = order_;

  for (int fj = 0; fj < order; fj++)
    globalToLocalItem[localToGlobalItem[fj]] = fj;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadIndices"
template<class T>
void FrontalMatrix<T>::loadIndices(const Symbolic& s, int front)
{
  BEGIN_FUNCTION();

  int* localToGlobalItem = localToGlobal_->getItem();
  const int* sFrontPointerItem = s.getFrontPointer()->getItem();
  const int* sRowIndexItem = s.getRowIndex()->getItem();

  int order = order_,
      originalFrontSize = originalFrontSize_,
      modifiedFrontSize = modifiedFrontSize_;

  int sp = sFrontPointerItem[front];

  {for (int fj = 0;
       fj < originalFrontSize;
       fj++, sp++)
    localToGlobalItem[fj] = sRowIndexItem[sp];}

  {for (int fj = modifiedFrontSize;
       fj < order;
       fj++, sp++)
    localToGlobalItem[fj] = sRowIndexItem[sp];}

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadIndices"
template<class T>
void FrontalMatrix<T>::loadIndices(const UpdateMatrix<T>& um)
{
  BEGIN_FUNCTION();

  int* localToGlobalItem = localToGlobal_->getItem();
  const int* umLocalToGlobalItem = um.getLocalToGlobal()->getItem();

  int originalFrontSize = originalFrontSize_,
      delayedFrontSize = delayedFrontSize_,
      umDelayedFrontSize = um.getDelayedFrontSize();

  for (int uj = 0, fj = originalFrontSize + delayedFrontSize;
       uj < umDelayedFrontSize;
       uj++, fj++)
    localToGlobalItem[fj] = umLocalToGlobalItem[uj];

  delayedFrontSize_ += um.getDelayedFrontSize();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "orderIndices"
template<class T>
void FrontalMatrix<T>::orderIndices(const SparseFactors<T>& l,
                                    Permutation& p) const
{
  BEGIN_FUNCTION();

  const int* localToGlobalItem = localToGlobal_->getItem();
  int* pOldToNewItem = p.getOldToNew()->getItem();
  int* pNewToOldItem = p.getNewToOld()->getItem();

  int modifiedFrontSize = modifiedFrontSize_,
      lNumberOfEliminatedColumns = l.getNumberOfEliminatedColumns();

  for (int fj = 0; fj < modifiedFrontSize; fj++)
  {
    pOldToNewItem[localToGlobalItem[fj]] = lNumberOfEliminatedColumns + fj;
    pNewToOldItem[lNumberOfEliminatedColumns + fj] = localToGlobalItem[fj];
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadEntries"
template<class T>
void FrontalMatrix<T>::loadEntries(const SparseMatrix<T>& a,
                                   const Permutation& p,
                                   const Array<int>& globalToLocal)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  T* entryItem = entry_->getItem();
  int* localToGlobalItem = localToGlobal_->getItem();
  const int* aColumnPointerItem = a.getColumnPointer()->getItem();
  const int* aRowIndexItem = a.getRowIndex()->getItem();
  const T* aEntryItem = a.getEntry()->getItem();
  const int* pOldToNewItem = p.getOldToNew()->getItem();
  const int* pNewToOldItem = p.getNewToOld()->getItem();
  const int* globalToLocalItem = globalToLocal.getItem();

  int originalFrontSize = originalFrontSize_;

  for (int fj = 0; fj < originalFrontSize; fj++)
  {
    int lj = localToGlobalItem[fj],
        aj = pNewToOldItem[lj],
        fp = columnPointerItem[fj];
    for (int ap = aColumnPointerItem[aj];
         ap < aColumnPointerItem[aj + 1];
         ap++)
    {
      int li = pOldToNewItem[aRowIndexItem[ap]];
      if (li >= lj)
      /* Select lower triangle row indices. */
        entryItem[fp + globalToLocalItem[li] - fj] = aEntryItem[ap];
    }
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "loadEntries"
template<class T>
void FrontalMatrix<T>::loadEntries(const UpdateMatrix<T>& um,
                                   const Array<int>& globalToLocal)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  T* entryItem = entry_->getItem();
  const int* umColumnPointerItem = um.getColumnPointer()->getItem();
  const T* umEntryItem = um.getEntry()->getItem();
  const int* umLocalToGlobalItem = um.getLocalToGlobal()->getItem();
  const int* globalToLocalItem = globalToLocal.getItem();

  int order = order_,
      umOrder = um.getOrder(),
      umDelayedFrontSize = um.getDelayedFrontSize();

  /* Original update entries. */
  {for (int uj = umDelayedFrontSize; uj < umOrder; uj++)
  {
    int fp = columnPointerItem[globalToLocalItem[umLocalToGlobalItem[uj]]],
        up = umColumnPointerItem[uj];
    for (int ui = uj, uq = up; ui < umOrder; ui++, uq++)
    {
      int fq = fp + globalToLocalItem[umLocalToGlobalItem[ui]] -
               globalToLocalItem[umLocalToGlobalItem[uj]];
      entryItem[fq] += umEntryItem[uq];
    }
  }}

  /* Delayed update entries. */
  for (int uj = 0, up = 0; uj < umDelayedFrontSize; uj++, up += umOrder + 1)
    for (int ui = uj, uq = up; ui < umOrder; ui++, uq++)
      /* Check if this goes to a row or to a column of the frontal matrix. */
      if (globalToLocalItem[umLocalToGlobalItem[uj]] <=
          globalToLocalItem[umLocalToGlobalItem[ui]])
      {
        int fq = globalToLocalItem[umLocalToGlobalItem[uj]] * order +
                 globalToLocalItem[umLocalToGlobalItem[ui]];
        entryItem[fq] += umEntryItem[uq];
      }
      else
      {
        int fq = globalToLocalItem[umLocalToGlobalItem[ui]] * order +
                 globalToLocalItem[umLocalToGlobalItem[uj]];
#ifndef HERMITIAN
        entryItem[fq] += umEntryItem[uq];
#else
        entryItem[fq] += Conj(umEntryItem[uq]);
#endif
      }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveIndices"
template<class T>
void FrontalMatrix<T>::saveIndices(SparseFactors<T>& l) const
{
  BEGIN_FUNCTION();

  const int* localToGlobalItem = localToGlobal_->getItem();
  int* lColumnPointerItem = l.getColumnPointer()->getItem();
  int* lRowIndexItem = l.getRowIndex()->getItem();

  int order = order_,
      finalFrontSize = finalFrontSize_,
      lNumberOfEliminatedColumns = l.getNumberOfEliminatedColumns();

  for (int fj = 0; fj < finalFrontSize; fj++)
  {
    int lp = lColumnPointerItem[lNumberOfEliminatedColumns];
    for (int fi = fj, lq = lp; fi < order; fi++, lq++)
      lRowIndexItem[lq] = localToGlobalItem[fi];
    lColumnPointerItem[lNumberOfEliminatedColumns + 1] =
      lColumnPointerItem[lNumberOfEliminatedColumns] + order - fj;
    lNumberOfEliminatedColumns++;
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveIndices"
template<class T>
void FrontalMatrix<T>::saveIndices(UpdateMatrix<T>& um) const
{
  BEGIN_FUNCTION();

  const int* localToGlobalItem = localToGlobal_->getItem();
  int* umLocalToGlobalItem = um.getLocalToGlobal()->getItem();

  int order = order_,
      finalFrontSize = finalFrontSize_;

  for (int fj = finalFrontSize, uj = 0; fj < order; fj++, uj++)
    umLocalToGlobalItem[uj] = localToGlobalItem[fj];

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveEntries"
template<class T>
void FrontalMatrix<T>::saveEntries(SparseFactors<T>& l) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const T* entryItem = entry_->getItem();
  int *lColumnPointerItem = l.getColumnPointer()->getItem();
  T* lEntryItem = l.getEntry()->getItem();

  int order = order_,
      finalFrontSize = finalFrontSize_,
      lNumberOfEliminatedColumns = l.getNumberOfEliminatedColumns();

  /* Save the values of the eliminated columns into the factor. */
  for (int fj = 0; fj < finalFrontSize; fj++)
  {
    int numberOfRows = order - fj,
        fp = columnPointerItem[fj],
        lp = lColumnPointerItem[lNumberOfEliminatedColumns];

    Copy(&numberOfRows, &entryItem[fp], &lEntryItem[lp]);
    lNumberOfEliminatedColumns++;
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "saveEntries"
template<class T>
void FrontalMatrix<T>::saveEntries(UpdateMatrix<T>& um) const
{
  BEGIN_FUNCTION();

  const int* columnPointerItem = columnPointer_->getItem();
  const T* entryItem = entry_->getItem();
  int* umColumnPointerItem = um.getColumnPointer()->getItem();
  T* umEntryItem = um.getEntry()->getItem();

  int order = order_,
      finalFrontSize = finalFrontSize_;

  for (int fj = finalFrontSize, uj = 0; fj < order; fj++, uj++)
  {
    int numberOfRows = order - fj,
        fp = columnPointerItem[fj],
        up = umColumnPointerItem[uj];

    Copy(&numberOfRows, &entryItem[fp], &umEntryItem[up]);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "factor"
template<class T>
void FrontalMatrix<T>::factor(SparseFactors<T>& l)
{
  BEGIN_FUNCTION();

  int* localToGlobalItem = localToGlobal_->getItem();
  int* lPivotTypeItem = l.getPivotType()->getItem();

  int modifiedFrontSize = modifiedFrontSize_,
      finalFrontSize = finalFrontSize_,
      numberOf1x1Pivots = 0;

  for (int fj = 0; fj < modifiedFrontSize; fj++)
  {
    eliminate1x1(fj);
    CHECK_ERROR1();
    update1x1(fj);
    lPivotTypeItem[localToGlobalItem[fj]] = 1;
    numberOf1x1Pivots++;
    finalFrontSize++;
  }

  l.setNumberOf1x1Pivots(l.getNumberOf1x1Pivots() + numberOf1x1Pivots);

  finalFrontSize_ = finalFrontSize;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "factor"
template<class T>
void FrontalMatrix<T>::factor(SparseFactors<T>& l,
                              Array<int>& globalToLocal,
                              Permutation& p,
                              double threshold)
{
  BEGIN_FUNCTION();

  Queue q(modifiedFrontSize_);
  CHECK_ERROR1();

  int* localToGlobalItem = localToGlobal_->getItem();
  T* entryItem = entry_->getItem();
  int* lPivotTypeItem = l.getPivotType()->getItem();
  const int* globalToLocalItem = globalToLocal.getItem();

  int order = order_,
      modifiedFrontSize = modifiedFrontSize_,
      finalFrontSize = finalFrontSize_,
      numberOfSwappedColumns = 0,
      numberOf1x1Pivots = 0,
      numberOf2x2Pivots = 0,
      lNumberOfEliminatedColumns = l.getNumberOfEliminatedColumns();

  /* Enqueue front columns. */
  {for (int fj = 0; fj < modifiedFrontSize; fj++)
    q.enqueue(localToGlobalItem[fj]);}

  int fj = 0;
  bool pivotFound;
  do
  {
    int numberOfTrials = q.getCurrentSize();
    pivotFound = false;
    do
    {
      /* Dequeue a column. */
      int fj1 = globalToLocalItem[q.dequeue()];

      /* Get the diagonal entry in this column. */
      T diagonal11 = entryItem[fj1 + fj1 * order];

      /* Get the maximum off-diagonal magnitude in this column. */
      double lambda = getMaximumOffDiagonal(fj1, finalFrontSize);

      /* Try a 1x1 pivot. */
      if ((Abs(diagonal11) > 0) &&
          (Abs(diagonal11) >= threshold * lambda))
      {
        pivotFound = true;

        /* Swap columns if necessary. */
        if (fj1 != fj)
	{
          swapColumns(fj, fj1, p, globalToLocal);
          numberOfSwappedColumns++;
        }

        eliminate1x1(fj);
        CHECK_ERROR1();
        update1x1(fj);
        lPivotTypeItem[lNumberOfEliminatedColumns + finalFrontSize] = 1;
        numberOf1x1Pivots++;
        finalFrontSize++;
        fj++;
      }
      else
      {
        /* Get the maximum off-diagonal magnitude in the feasible
           region of this column and the row index of the
           first entry having this magnitude. */
        int fj2;
        double miu = getMaximumOffDiagonal(fj1, finalFrontSize, fj2);

        /* Get the corresponding entry. */
        T diagonal12, diagonal21;
        if (fj2 > fj1)
          diagonal21 = entryItem[fj2 + fj1 * order];
        else
#ifndef HERMITIAN
          diagonal21 = entryItem[fj1 + fj2 * order];
#else
          diagonal21 = Conj(entryItem[fj1 + fj2 * order]);
#endif

#ifndef HERMITIAN
        diagonal12 = diagonal21;
#else
        diagonal12 = Conj(diagonal21);
#endif

        /* Get the diagonal entry in the column having the same
           index as the row above. */
        T diagonal22 = entryItem[fj2 + fj2 * order];

        /* Get the maximum off-diagonal magnitude in the new column. */
        double gamma = getMaximumOffDiagonal(fj2, finalFrontSize);

        /* Compute the determinant of the diagonal block. */
        T diagonal = diagonal11 * diagonal22 -
                     diagonal12 * diagonal21;

        /* Try a 2x2 pivot. */
        if (!q.empty() && (Abs(diagonal) > 0) &&
            (Max(Abs(diagonal22) * lambda + miu * gamma,
                 Abs(diagonal11) * gamma + miu * lambda)
             <= Abs(diagonal) / threshold))
        {
          pivotFound = true;

          q.remove(localToGlobalItem[fj2]);

          if (fj1 != fj)
          {
            swapColumns(fj, fj1, p, globalToLocal);
            if (fj2 == fj)
              fj2 = fj1;
            numberOfSwappedColumns++;
          }

          if (fj2 != fj + 1)
	  {
            swapColumns(fj + 1, fj2, p, globalToLocal);
            numberOfSwappedColumns++;
          }

          eliminate2x2(fj);
          CHECK_ERROR1();
          lPivotTypeItem[lNumberOfEliminatedColumns + finalFrontSize] = 2;
          lPivotTypeItem[lNumberOfEliminatedColumns + finalFrontSize + 1] = 3;
          numberOf2x2Pivots++;
          finalFrontSize += 2;
          fj += 2;
        }
        /* No pivot was found, put the currently tested column
           back in the queue. */
        else
          q.enqueue(localToGlobalItem[fj1]);
      }

      numberOfTrials--;
    }
    /* As long as no pivot was found and there are other untested
       columns in the queue. */
    while (!pivotFound && (numberOfTrials > 0));
  }
  /* As long as there are still columns in the queue and pivots
     were found. */
  while (!q.empty() && pivotFound);

  /* Clear the queue. All the columns that are still in the
     queue are delayed to the parent of the supernode this
     frontal matrix is associated with. */
  while (!q.empty())
    q.dequeue();

  l.setNumberOfSwappedColumns(l.getNumberOfSwappedColumns() +
                              numberOfSwappedColumns);
  l.setNumberOfDelayedColumns(l.getNumberOfDelayedColumns() +
                              modifiedFrontSize -
                              finalFrontSize);
  l.setNumberOf1x1Pivots(l.getNumberOf1x1Pivots() + numberOf1x1Pivots);
  l.setNumberOf2x2Pivots(l.getNumberOf2x2Pivots() + numberOf2x2Pivots);

  finalFrontSize_ = finalFrontSize;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "swapColumns"
template<class T>
void FrontalMatrix<T>::swapColumns(int column1, int column2,
                                   Permutation &p,
                                   Array<int>& globalToLocal)
{
  BEGIN_FUNCTION();

  /* Column <column1> must precede column <column2>. */

  int* localToGlobalItem = localToGlobal_->getItem();
  T* entryItem = entry_->getItem();
  int* pOldToNewItem = p.getOldToNew()->getItem();
  int* pNewToOldItem = p.getNewToOld()->getItem();
  int* globalToLocalItem = globalToLocal.getItem();

  int order = order_;

  /* Swap horizontal-horizontal. */
  {for (int fj = 0, fp = column1, fq = column2;
       fj < column1;
       fj++, fp += order, fq += order)
    Swap(entryItem[fp], entryItem[fq]);}

  /* Swap vertical-vertical. */
  {for (int fi = column2 + 1, fp = column1 * order + column2 + 1,
       fq = column2 * order + column2 + 1;
       fi < order;
       fi++, fp++, fq++)
    Swap(entryItem[fp], entryItem[fq]);}

  /* Swap vertical-horizontal. */
  {for (int fi = column1 + 1, fp = column1 * order + column1 + 1,
       fq = (column1 + 1) * order + column2;
       fi < column2;
       fi++, fp++, fq += order)
  {
#ifndef HERMITIAN
    Swap(entryItem[fp], entryItem[fq]);
#else
    T tmp = Conj(entryItem[fp]);
    entryItem[fp] = Conj(entryItem[fq]);
    entryItem[fq] = tmp;
#endif
  }}

  /* Swap diagonal-diagonal. */
  int fp = column1 * order + column1, fq = column2 * order + column2;
  Swap(entryItem[fp], entryItem[fq]);

#ifdef HERMITIAN
  /* Conjugate the intersection. */
  entryItem[column2 + column1 * order] =
    Conj(entryItem[column2 + column1 * order]);
#endif

  /* Update the permutation. */
  Swap(pOldToNewItem[localToGlobalItem[column1]],
    pOldToNewItem[localToGlobalItem[column2]]);
  Swap(pNewToOldItem[localToGlobalItem[column1]],
    pNewToOldItem[localToGlobalItem[column2]]);

  /* Update the global to local map. */
  Swap(globalToLocalItem[localToGlobalItem[column1]],
    globalToLocalItem[localToGlobalItem[column2]]);

  /* Update the local to global map. */
  Swap(localToGlobalItem[column1], localToGlobalItem[column2]);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getMaximumOffDiagonal"
template<class T>
double FrontalMatrix<T>::getMaximumOffDiagonal(int column,
                                               int numberOfEliminatedColumns)
                                              const
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();

  double maximum = 0.0;
  int order = order_;

  {for (int fj = numberOfEliminatedColumns,
            fp = numberOfEliminatedColumns * order_ + column;
        fj < column;
        fj++, fp += order)
    if (maximum < Abs(entryItem[fp]))
      maximum = Abs(entryItem[fp]);}

  for (int fi = column + 1, fp = column * order_ + column + 1;
       fi < order;
       fi++, fp++)
    if (maximum < Abs(entryItem[fp]))
      maximum = Abs(entryItem[fp]);

  END_FUNCTION();

  return maximum;
}

#undef __FUNC__
#define __FUNC__ "getMaximumOffDiagonal"
template<class T>
double FrontalMatrix<T>::getMaximumOffDiagonal(int column,
                                               int numberOfEliminatedColumns,
                                               int &row) const
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();

  double maximum = 0.0;
  int order = order_,
      modifiedFrontSize = modifiedFrontSize_;

  row = numberOfEliminatedColumns;

  {for (int fj = numberOfEliminatedColumns,
            fp = numberOfEliminatedColumns * order + column;
       fj < column;
       fj++, fp += order)
    if (maximum < Abs(entryItem[fp]))
    {
      maximum = Abs(entryItem[fp]);
      row = fj;
    }}

  for (int fi = column + 1, fp = column * order + column + 1;
       fi < modifiedFrontSize;
       fi++, fp++)
    if (maximum < Abs(entryItem[fp]))
    {
      maximum = Abs(entryItem[fp]);
      row = fi;
    }

  END_FUNCTION();

  return maximum;
}

#endif
