#ifndef DENSE_MATRIX_H
#define DENSE_MATRIX_H

#include <iostream.h>
#include <fstream.h>

#include "Array.h"
#include "Error.h"
#include "Kernels.h"

#undef __CLASS__
#define __CLASS__ "DenseMatrix"
template<class T>
class DenseMatrix
{
  protected:
    int order_;
    Array<int>* columnPointer_;
    Array<T>* entry_;

    DenseMatrix(const DenseMatrix<T>&);
    DenseMatrix<T>& operator=(const DenseMatrix<T>&);

    void allocate(void);
    void free(void);

    void initializeColumnPointer(void);

  public:
    DenseMatrix();
    DenseMatrix(int order);
    virtual ~DenseMatrix();

    int getOrder(void) const
      {return order_;}
    Array<int>* getColumnPointer(void)
      {return columnPointer_;}
    const Array<int>* getColumnPointer(void) const
      {return columnPointer_;}
    Array<T>* getEntry(void)
      {return entry_;}
    const Array<T>* getEntry(void) const
      {return entry_;}

    void resize(int order);
    int getHeapSize(void) const;
    bool check(void) const;

    void eliminate1x1(int pivot);
    void update1x1(int pivot);
    void eliminate2x2(int pivot);

    void fill(T value)
      {entry_->fill(value);}
};

#undef __FUNC__
#define __FUNC__ "allocate"
template<class T>
void DenseMatrix<T>::allocate(void)
{
  BEGIN_FUNCTION();

  try
  {
    columnPointer_ = new Array<int>(order_);
    entry_ = new Array<T>(order_ * order_);
  }
  catch (...)
  {
    SET_ERROR1(MemoryAllocation);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "free"
template<class T>
void DenseMatrix<T>::free(void)
{
  BEGIN_FUNCTION();

  delete columnPointer_;
  columnPointer_ = 0;
  delete entry_;
  entry_ = 0;

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "DenseMatrix"
template<class T>
DenseMatrix<T>::DenseMatrix():
  order_(0),
  columnPointer_(0),
  entry_(0)
{
  BEGIN_FUNCTION();

  allocate();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "DenseMatrix"
template<class T>
DenseMatrix<T>::DenseMatrix(int order):
  order_(order),
  columnPointer_(0),
  entry_(0)
{
  BEGIN_FUNCTION();

  allocate();
  initializeColumnPointer();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "~DenseMatrix"
template<class T>
DenseMatrix<T>::~DenseMatrix()
{
  BEGIN_FUNCTION();

  free();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "resize"
template<class T>
void DenseMatrix<T>::resize(int order)
{
  BEGIN_FUNCTION();

  free();
  order_ = order;
  allocate();
  initializeColumnPointer();

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "getHeapSize"
template<class T>
int DenseMatrix<T>::getHeapSize(void) const
{
  BEGIN_FUNCTION();

  int heapSize = columnPointer_->getHeapSize() +
                 entry_->getHeapSize();

  END_FUNCTION();

  return heapSize;
}

#undef __FUNC__
#define __FUNC__ "check"
template<class T>
bool DenseMatrix<T>::check(void) const
{
  BEGIN_FUNCTION();

  END_FUNCTION();

  if (order_ < 0)
    return false;
  return true;
}

#undef __FUNC__
#define __FUNC__ "eliminate1x1"
template<class T>
void DenseMatrix<T>::eliminate1x1(int pivot)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  T* entryItem = entry_->getItem();

  T diagonalEntry = entryItem[columnPointerItem[pivot]];
  if (diagonalEntry == 0)
    SET_ERROR1(Zero1x1Pivot);

  int numberOfRows = order_ - (pivot + 1);
  T a = 1 / diagonalEntry;
  Scal1(&numberOfRows, &a, &entryItem[columnPointerItem[pivot] + 1]);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "update1x1"
template<class T>
void DenseMatrix<T>::update1x1(int pivot)
{
  BEGIN_FUNCTION();

  int* columnPointerItem = columnPointer_->getItem();
  T* entryItem = entry_->getItem();

  T diagonal = entryItem[columnPointerItem[pivot]];
  if (diagonal == 0)
    SET_ERROR1(Zero1x1Pivot);

  int order = order_;

  for (int j = pivot + 1,
           p1 = pivot * order + pivot + 1,
           p2 = (pivot + 1) * order + pivot + 1;
       j < order;
       j++, p1++, p2 += order + 1)
  {
    int numberOfRows = order - j;
#ifndef HERMITIAN
    T a = -(entryItem[p1] * diagonal);
#else
    T a = -(Conj(entryItem[p1]) * diagonal);
#endif
    Axpy(&numberOfRows, &a, &entryItem[p1], &entryItem[p2]);
  }

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "eliminate2x2"
template<class T>
void DenseMatrix<T>::eliminate2x2(int pivot)
{
  BEGIN_FUNCTION();

  T* entryItem = entry_->getItem();

  T diagonal11 = entryItem[pivot + pivot * order_],
    diagonal21 = entryItem[(pivot + 1) + pivot * order_],
#ifndef HERMITIAN
    diagonal12 = diagonal21,
#else
    diagonal12 = Conj(diagonal21),
#endif
    diagonal22 = entryItem[(pivot + 1) + (pivot + 1) * order_],
    diagonal = diagonal11 * diagonal22 - diagonal12 * diagonal21;

  if (diagonal == 0)
    SET_ERROR1(Zero2x2Pivot);

  int order = order_;

  T a1 = diagonal11 / diagonal,
    b1 = diagonal12 / diagonal,
    c1 = diagonal21 / diagonal,
    d1 = diagonal22 / diagonal;

  /* Update the rest of the frontal matrix. */
  {for (int j = pivot + 2, p1 = pivot * order + pivot + 2,
           p2 = (pivot + 1) * order + pivot + 2,
           p3 = (pivot + 2) * order + pivot + 2;
       j < order;
       j++, p1++, p2++, p3 += order + 1)
  {
    int numberOfRows = order - j;
#ifndef HERMITIAN
    T a2 = (b1 * entryItem[p1]) - (a1 * entryItem[p2]),
      b2 = (c1 * entryItem[p2]) - (d1 * entryItem[p1]);
#else
    T a2 = (b1 * Conj(entryItem[p1])) - (a1 * Conj(entryItem[p2])),
      b2 = (c1 * Conj(entryItem[p2])) - (d1 * Conj(entryItem[p1]));
#endif
    Axpy(&numberOfRows, &a2, &entryItem[p2], &entryItem[p3]);
    Axpy(&numberOfRows, &b2, &entryItem[p1], &entryItem[p3]);
  }}

  /* Eliminate the pivot columns. */
  int numberOfRows = order - (pivot + 2),
      p1 = pivot * order + pivot + 2,
      p2 = (pivot + 1) * order + pivot + 2;
  Scal2(&numberOfRows, &a1, &b1, &c1, &d1, &entryItem[p1], &entryItem[p2]);

  END_FUNCTION();
}

#undef __FUNC__
#define __FUNC__ "initializeColumnPointer"
template<class T>
void DenseMatrix<T>::initializeColumnPointer(void)
{
  BEGIN_FUNCTION();

  int *columnPointer = columnPointer_->getItem();
#ifdef TRIANGULAR
  for (int j = 0, p = 0; j < order_; j++, p += order_ - j)
    columnPointer[j] = p;
#else
  for (int j = 0, p = 0; j < order_; j++, p += order_ + 1)
    columnPointer[j] = p;
#endif

  END_FUNCTION();
}

#endif
