/*
   Authors: Florin Dobrian & Alex Pothen
   Date: July, 1998
*/

#ifndef COMPLEX_H
#define COMPLEX_H

#include <iostream.h>
#include <math.h>

class Complex
{
  private:
    double re_;
    double im_;

  public:
    Complex(void)
    {
    }
    Complex(const Complex &complex)
    {
      re_ = complex.re_;
      im_ = complex.im_;
    }
    Complex(double re)
    {
      re_ = re;
      im_ = 0;
    }
    Complex(double re, double im)
    {
      re_ = re;
      im_ = im;
    }
    double getRe(void) {return re_;}
    double getIm(void) {return im_;}
    void setRe(double re) {re_ = re;}
    void setIm(double im) {im_ = im;}
    Complex &operator=(const Complex &operand)
    {
      re_ = operand.re_;
      im_ = operand.im_;
      return *this;
    }
    Complex &operator=(double operand)
    {
      re_ = operand;
      im_ = 0;
      return *this;
    }
    Complex operator+(void) const
    {
      return *this;
    }
    Complex operator-(void) const
    {
      return Complex(-re_, -im_);
    }
    Complex operator+(const Complex &operand) const
    {
      return Complex(re_ + operand.re_, im_ + operand.im_);
    }
    Complex operator+(double operand) const
    {
      return Complex(re_ + operand, im_);
    }
    Complex operator-(const Complex &operand) const
    {
      return Complex(re_ - operand.re_, im_ - operand.im_);
    }
    Complex operator-(double operand) const
    {
      return Complex(re_ - operand, im_);
    }
    Complex operator*(const Complex &operand) const
    {
      return Complex(re_ * operand.re_ - im_ * operand.im_,
                     re_ * operand.im_ + im_ * operand.re_);
    }
    Complex operator*(double operand) const
    {
      return Complex(re_ * operand, im_ * operand);
    }
    Complex operator/(const Complex &operand) const
    {
      double normalization = operand.re_ * operand.re_ +
                             operand.im_ * operand.im_;
      return Complex((re_ * operand.re_ + im_ * operand.im_) / normalization,
                     (im_ * operand.re_ - re_ * operand.im_) / normalization);
    }
    Complex operator/(double operand) const
    {
      return Complex(re_ / operand, im_ / operand);
    }
    Complex &operator+=(const Complex &operand)
    {
      re_ += operand.re_;
      im_ += operand.im_;
      return *this;
    }
    Complex &operator+=(double operand)
    {
      re_ += operand;
      return *this;
    }
    Complex &operator-=(const Complex &operand)
    {
      re_ -= operand.re_;
      im_ -= operand.im_;
      return *this;
    }
    Complex &operator-=(double operand)
    {
      re_ -= operand;
      return *this;
    }
    Complex &operator*=(const Complex &operand)
    {
      double tmpRe = re_, tmpIm = im_;
      re_ = tmpRe * operand.re_ - tmpIm * operand.im_;
      im_ = tmpRe * operand.im_ + tmpIm * operand.re_;
      return *this;
    }
    Complex &operator*=(double operand)
    {
      re_ *= operand;
      im_ *= operand;
      return *this;
    }
    Complex &operator/=(const Complex &operand)
    {
      double tmpRe = re_, tmpIm = im_,
             normalization = operand.re_ * operand.re_ +
                             operand.im_ * operand.im_;
      re_ = (tmpRe * operand.re_ + tmpIm * operand.im_) / normalization;
      im_ = (tmpIm * operand.re_ - tmpRe * operand.im_) / normalization;
      return *this;
    }
    Complex &operator/=(double operand)
    {
      re_ /= operand;
      im_ /= operand;
      return *this;
    }
    int operator==(const Complex &operand) const
    {
      return ((re_ == operand.re_) && (im_ == operand.im_));
    }
    int operator==(double operand) const
    {
      return ((re_ == operand) && (im_ == 0.0));
    }
    int operator!=(const Complex &operand) const
    {
      return ((re_ != operand.re_) || (im_ != operand.im_));
    }
    int operator!=(double operand) const
    {
      return ((re_ != operand) || (im_ != 0.0));
    }

  friend double Re(const Complex &complex);
  friend double Im(const Complex &complex);
  friend Complex Conj(const Complex &complex);
  friend double Abs(const Complex &complex);
  friend Complex operator+(double operand1, const Complex &operand2);
  friend Complex operator-(double operand1, const Complex &operand2);
  friend Complex operator*(double operand1, const Complex &operand2);
  friend Complex operator/(double operand1, const Complex &operand2);
  friend int operator==(double operand1, const Complex &operand2);
  friend ostream &operator<<(ostream &outFile, const Complex &operand);
  friend istream &operator>>(istream &inFile, Complex &operand);
};

inline double Re(const Complex &complex)
{
  return complex.re_;
}

inline double Im(const Complex &complex)
{
  return complex.im_;
}

inline Complex Conj(const Complex &complex)
{
  Complex result(complex.re_, -complex.im_);
  return result;
}

inline double Abs(const Complex &value)
{
  return sqrt(value.re_ * value.re_ + value.im_ * value.im_);
}

inline Complex operator+(double operand1, const Complex &operand2)
{
  return Complex(operand1 + operand2.re_, operand2.im_);
}

inline Complex operator-(double operand1, const Complex &operand2)
{
  return Complex(operand1 - operand2.re_, -operand2.im_);
}

inline Complex operator*(double operand1, const Complex &operand2)
{
  return Complex(operand1 * operand2.re_, operand1 * operand2.im_);
}

inline Complex operator/(double operand1, const Complex &operand2)
{
  double normalization = operand2.re_ * operand2.re_ +
                         operand2.im_ * operand2.im_;
  return Complex(operand1 * operand2.re_ / normalization,
                 operand1 * (-operand2.im_) / normalization);
}

inline int operator==(double operand1, const Complex &operand2)
{
  return ((operand1 == operand2.re_) &&
          (0.0 == operand2.im_));
}

inline ostream &operator<<(ostream &outFile, const Complex &operand)
{
  outFile << operand.re_ << ' ' << operand.im_;
  return outFile;
}

inline istream &operator>>(istream &inFile, Complex &operand)
{
  inFile >> operand.re_ >> operand.im_;
  return inFile;
}

#endif
