#include <iostream.h>
#include <stdlib.h>
#include <new.h>

#include "Complex.h"
#include "SparseMatrix.h"
#include "Permutation.h"
#include "EliminationForest.h"
#include "Symbolic.h"
#include "DenseMatrix.h"
#include "UpdateMatrix.h"
#include "FrontalMatrix.h"
#include "UpdateStack.h"
#include "SparseFactors.h"
#include "MinimumDegreeOrder.h"
#include "MultifrontalFactor.h"
#include "Multiply.h"
#include "Solve.h"
#include "Error.h"
#include "Timer.h"

#define ValueType Complex

const int MaxStrLen = 128;

void oblio_new_handler(void)
{
  int i = 0;
  throw i;
}

int main(int argc, char** argv)
{
  set_new_handler(oblio_new_handler);

  cerr << "Oblio - Version 2.0" << endl;

  if (argc == 1)
  {
    cerr << "usage: oblio matrix"
         << " [-o order_type]"
         << " [-p permutation]"
         << " [-t threshold]"
         << " [-v load_values]"
         << " [-b right_hand_side]"
         << " [-x solution]"
         << " [-i number_of_iterations]"
         << endl;
    return 1;
  }

  char xFileName[MaxStrLen + 1],
       bFileName[MaxStrLen + 1],
       pFileName[MaxStrLen + 1];

  /* Do this to suppress "uninitialized memory" messages from Purify. */
  {for (int i = 0; i <= MaxStrLen; i++)
    xFileName[i] =
    bFileName[i] =
    pFileName[i] = 0;}

  int orderType = 0, loadValues = 0, numberOfIterations = 0;
  double threshold = 0;

  /* Read command line options. */
  {for (int i = 2; i < argc; i++)
  {
    if (!strcmp(argv[i], "-o"))
      orderType = atoi(argv[++i]);
    if (!strcmp(argv[i], "-p"))
      strcpy(pFileName, argv[++i]);
    if (!strcmp(argv[i], "-t"))
      threshold = atof(argv[++i]);
    if (!strcmp(argv[i], "-v"))
      loadValues = atoi(argv[++i]);
    if (!strcmp(argv[i], "-b"))
      strcpy(bFileName, argv[++i]);
    if (!strcmp(argv[i], "-x"))
      strcpy(xFileName, argv[++i]);
    if (!strcmp(argv[i], "-i"))
      numberOfIterations = atoi(argv[++i]);
  }}

  /* Load the coefficeint matrix from a file. */
  SparseMatrix<ValueType> a;
  CHECK_ERROR4(1);
  cerr << "loading coefficient matrix ..." << endl;
  a.setValued(((loadValues == 0) ? false : true));
  a.load(argv[1]);
  CHECK_ERROR4(1);

  /* Prepare the permutation. */
  Permutation p(a.getOrder());
  CHECK_ERROR4(1);

  if (orderType == 1)
  /* Order using multiple minimum degree. */
  {
    MinimumDegreeOrder<ValueType> order;
    order.setA(a);
    order.setP(p);
    cerr << "ordering ..." << endl;
    order.run();
    CHECK_ERROR4(1);

    if (pFileName[0])
    {
      cerr << "saving permutation ..." << endl;
      p.save(pFileName);
      CHECK_ERROR4(1);
      return 0;
    }
  }
  else if (pFileName[0])
  /* Load the permutation from a file. */
  {
    cerr << "loading permutation ..." << endl;
    p.load(pFileName);
    CHECK_ERROR4(1);
  }
  else
  /* Initialize a natural order permutation. */
    p.initialize();

  /* Compute the primary elimination forest (may be changed by pivoting). */
  EliminationForest f1(a.getOrder());
  CHECK_ERROR4(1);
  f1.initialize(a, p);
  CHECK_ERROR4(1);
  f1.compress();
  CHECK_ERROR4(1);

  /* Compute the symbolic object. */
  Symbolic s(f1.getNumberOfNodes(), f1.getNumberOfRowIndices());
  CHECK_ERROR4(1);
  s.initialize(a, p, f1);
  CHECK_ERROR4(1);

  /* Prepare the update stack. */
  UpdateStack<ValueType> u(f1.getNumberOfNodes());
  CHECK_ERROR4(1);

  /* Prepare the factors. */
  SparseFactors<ValueType> l(a.getOrder(), f1.getNumberOfOffDiagonals());
  CHECK_ERROR4(1);

  /* Compute the factors. */
  MultifrontalFactor<ValueType> factor;
  factor.setA(a);
  factor.setP(p);
  factor.setF(f1);
  factor.setS(s);
  factor.setU(u);
  factor.setL(l);
  factor.setThreshold(threshold);
  Timer tmFactor;
  tmFactor.start();
  cerr << "factoring ..." << endl;
  factor.run();
  tmFactor.stop();
  CHECK_ERROR4(1);

  /* Compute the secondary elimination forest. */
  EliminationForest f2(l.getOrder());
  CHECK_ERROR4(1);
  f2.initialize(l);
  CHECK_ERROR4(1);
  f2.compress();
  CHECK_ERROR4(1);

  /* Prepare the solution and the right hand side. */
  Vector<ValueType> x(a.getOrder()), b(a.getOrder());
  CHECK_ERROR4(1);

  if (bFileName[0])
  /* Load the right hand side from a file. */
  {
    b.load(bFileName);
    CHECK_ERROR4(1);
  }
  else
  /* Initialize a simple right hand side. */
    b.initialize();

  /* Compute the solution. */
  Solve<ValueType> solve;
  solve.setP(p);
  solve.setF(f2);
  solve.setL(l);
  solve.setX(x);
  solve.setB(b);
  cerr << "solving ..." << endl;
  Timer tmSolve;
  tmSolve.start();
  solve.run();
  tmSolve.stop();
  CHECK_ERROR4(1);

  /* Compute the 1-norm relative residual. */
  Vector<ValueType> t1(a.getOrder()), r(a.getOrder());
  CHECK_ERROR4(1);

  Multiply<ValueType> multiply;
  multiply.setA(a);
  multiply.setX(x);
  multiply.setB(t1);
  multiply.run();
  CHECK_ERROR4(1);

  r.subtract(b, t1);
  CHECK_ERROR4(1);

  double rNorm1 = r.getNorm1(),
         aNorm1 = a.getNorm1(),
         xNorm1 = x.getNorm1();

  /* Perfrom iterative refinement. */
  if (numberOfIterations > 0)
  {
    cerr << "performing iterative refinement ..." << endl;

    Vector<ValueType> t2(a.getOrder());
    CHECK_ERROR4(1);

    for (int i = 0; i < numberOfIterations; i++)
    {
      solve.setX(t2);
      solve.setB(r);
      solve.run();
      CHECK_ERROR4(1);

      x.add(t2);
      multiply.run();
      CHECK_ERROR4(1);
      r.subtract(b, t1);
    }
  }

  if (xFileName[0])
  /* Save the solution to a file. */
  {
    cerr << "saving solution ..." << endl;
    x.save(xFileName);
    CHECK_ERROR4(1);
  }

  /* Compute the arithmetic work. */
  float eliminationWork1, assemblyWork1, solutionWork1,
        eliminationWork2, assemblyWork2, solutionWork2;
  ValueType dummy = 0;
  f1.getWork(eliminationWork1, assemblyWork1, solutionWork1, dummy);
  f2.getWork(eliminationWork2, assemblyWork2, solutionWork2, dummy);

  /* Print statistics. */
  cout << "Order of the problem: "
       << a.getOrder() << endl;
  cout << "Number of off-diagonal nonzeros (original): "
       << a.getNumberOfOffDiagonals() << endl;
  cout << "Number of off-diagonal nonzeros (filled): "
       << l.getNumberOfOffDiagonals() << endl;
  cout << "Number of trees: "
       << f1.getNumberOfTrees() << endl;
  cout << "Number of fronts without pivoting: "
       << f1.getNumberOfNodes() << endl;
  cout << "Number of fronts with pivoting: "
       << f2.getNumberOfNodes() << endl;
  cout << "Number of swapped columns: "
       << l.getNumberOfSwappedColumns() << endl;
  cout << "Number of delayed columns: "
       << l.getNumberOfDelayedColumns() << endl;
  cout << "Number of 1x1 pivots: "
       << l.getNumberOf1x1Pivots() << endl;
  cout << "Number of 2x2 pivots: "
       << l.getNumberOf2x2Pivots() << endl;

  cout.setf(ios::left | ios::fixed | ios::showpoint);
  cout.precision(3);
  cout << "Dynamic storage required by the coefficient matrix: "
       << (float)a.getHeapSize() / 1000000 << " MB" << endl;
  cout << "Dynamic storage required by the permutation: "
       << (float)p.getHeapSize() / 1000000 << " MB" << endl;
  cout << "Dynamic storage required by the elimination forest: "
       << (float)f1.getHeapSize() / 1000000 << " MB" << endl;
  cout << "Dynamic storage required by the symbolic object: "
       << (float)s.getHeapSize() / 1000000 << " MB" << endl;
  cout << "Dynamic storage required by the factors: "
       << (float)l.getHeapSize() / 1000000 << " MB" << endl;
  cout.unsetf(ios::left | ios::fixed | ios::showpoint);

  cout.setf(ios::left | ios::fixed | ios::showpoint);
  cout.precision(2);
  cout << "Execution time (factor): "
       << tmFactor.getWallTime() << "s (wall), "
       << tmFactor.getCpuTime() << "s (cpu), "
       << tmFactor.getUserCpuTime() << "s (user cpu), "
       << tmFactor.getSystemCpuTime() << "s (system cpu)"
       << endl;
  cout << "Execution time (solve): "
       << tmSolve.getWallTime() << "s (wall), "
       << tmSolve.getCpuTime() << "s (cpu), "
       << tmSolve.getUserCpuTime() << "s (user cpu), "
       << tmSolve.getSystemCpuTime() << "s (system cpu)"
       << endl;
  cout.unsetf(ios::left | ios::fixed | ios::showpoint);

  cout.setf(ios::left | ios::fixed);
  cout.precision(3);
  cout << "Arithmetic work without pivoting (factor): "
       << (eliminationWork1 + assemblyWork1) / 1000000
       << " Mflop" << endl;
  cout << "Arithmetic work without pivoting (solve): "
       << solutionWork1 / 1000000
       << " Mflop" << endl;
  cout << "Arithmetic work with pivoting (factor): "
       << (eliminationWork2 + assemblyWork2) / 1000000
       << " Mflop" << endl;
  cout << "Arithmetic work with pivoting (solve): "
       << solutionWork2 / 1000000
       << " Mflop" << endl;
  cout.precision(0);
  cout << "Performance (factor): "
       << (eliminationWork2 + assemblyWork2) / tmFactor.getWallTime() / 1000000
       << " Mflop/s" << endl;
  cout << "Performance (solve): "
       << solutionWork2 / tmSolve.getWallTime() / 1000000
       << " Mflop/s" << endl;
  cout.unsetf(ios::left | ios::fixed | ios::showpoint);

  cout.setf(ios::left | ios::scientific | ios::showpoint);
  cout.precision(1);
  cout << "Backward error: "
       << rNorm1 / aNorm1 / xNorm1 << endl;
  if (numberOfIterations > 0)
    cout << "Backward error after "
         << numberOfIterations
         << ((numberOfIterations == 1) ? " step" : " steps")
         << " of iterative refinement: "
         << r.getNorm1() / aNorm1 / x.getNorm1() << endl;
  cout.unsetf(ios::left | ios::scientific | ios::showpoint);

  return 0;
}
