/* 
   COPYRIGHT U.S. GOVERNMENT 
   
   This software is distributed without charge and comes with
   no warranty.

   Please feel free to send questions, comments, and problem reports
   to prism@super.org. 
*/

/* PURPOSE
   =======
   This function sets up a data layout on the mesh according to
   the values of parameters. It assumes that the value of
   (i,j)-element is given by a function supplied by the user.
   The present implementation supports torus mapping with
   arbitrary panel-width and panel-spacing. The block mapping
   may be added later.

   check list: 
   add options for different mappings
*/

/* INCLUDE FILES */
#include <math.h>
#include <stdio.h>
#include "mm.h"

void prism_v_layout(int i_mdim, int i_ndim, int i_v_dim, 
		    int i_sbmsh_nw_x, int i_sbmsh_nw_y,
		    int i_sbmsh_rows, int i_sbmsh_cols,
		    int i_r_panelwidth, int i_c_panelwidth,
		    int i_panel_spc,
		    double (* func)(int i, int j), 
		    double *p_dst, int i_lda,  E_LAYOUT e_layout,  
		    int i_2dcomm_rows, int i_2dcomm_cols,
		    int i_2d_row, int i_2d_col)

     /* PARAMETERS
        ==========
	i_mdim --- rows of matrix
	i_ndim --- cols of matrix
	i_vdim --- virtual dim of the mesh
	i_sbmsh_nw_x --- x coordinate of nw node in the submesh
	i_sbmsh_nw_y --- y    ..        ..           ..
	i_sbmsh_rows --- rows of the submesh
	i_sbmsh_cols --- cols of the submesh
	i_r_panelwidth --- panel width for rows
	i_c_panelwidth --- panel width for cols
	i_panel_spc  --- panel space
	*func() --- point to a function for computing (i,j)-element
	*p_dst --- pointer to the buffer for submatrix ob the node
	i_lda --- leading dim of submatrix on the node
	e_layout --- enum (wrap, block, wrap1d, block1d).
	i_2dcomm_rows --- # rows in 2d topology
	i_2dcomm_cols --- # cols in 2d topology
	i_2d_row --- row coordinate in 2d topology
	i_2d_col --- row coordinate in 2d topology
      */
{
  /* --------------- */
  /* Local variables */
  /* --------------- */
  int
    j, i,                  /* current col (row) index */
    c_pnl, r_pnl,          /* current col (row) panel id  */
    j_v_nd, i_v_nd,        /* current col (row) v-node id */
    i_node_row_sb,         /* row idx in the sbmsh */
    i_node_col_sb,         /* col idx in the sbmsh */
    g_c_v_nd, g_r_v_nd,    /* global col (row) id of the v-node*/
    i_v_msh_rows,          /* rows of virtual node per physical node */
    i_v_msh_cols,          /* cols of .. ..           ..    ..       */
    i_v_spc,               /* ratio of v_dim to spc */
    index1, index2,        /* for pointer calculation */
    fst_col, lst_col,      /* fst (lst) col on the current pnl */
    fst_row, lst_row,      /* fst (lst) row on the current pnl */
    fst_c_pnl, fst_r_pnl,  /* fst col (row) panel on the v-node */
    num_c_pnls,            /* num of col pnls */
    num_r_pnls;            /* num of row pnls */

  /* ------------------ */
  /* External functions */
  /* ------------------ */

  prism_v_init_var();

  /* parameter checking */
  if((i_sbmsh_nw_x + i_sbmsh_rows > i_2dcomm_rows) ||
     (i_sbmsh_nw_y + i_sbmsh_cols > i_2dcomm_cols)) {
    prism_v_generror("prism_v_layout: inappropriate submesh size", brief);
  }
  if (i_v_dim % i_panel_spc != 0 ) {
    prism_v_generror("prism_v_layout: i_v_dim not divided by i_panel_spc",
		     brief);
  }

  /* if the node is not in the submesh, return right away */
  if(i_2d_row < i_sbmsh_nw_x || i_2d_row >= i_sbmsh_nw_x + i_sbmsh_rows) { 
    return;
  }
  if(i_2d_col < i_sbmsh_nw_y || i_2d_col >= i_sbmsh_nw_y + i_sbmsh_cols) { 
    return;
  }
  
  /* Initialization */
  i_node_row_sb = i_2d_row - i_sbmsh_nw_x;
  i_node_col_sb = i_2d_col - i_sbmsh_nw_y;
  i_v_msh_rows = i_v_dim / i_sbmsh_rows;
  i_v_msh_cols = i_v_dim / i_sbmsh_cols;
  i_v_spc = i_v_dim / i_panel_spc;
  num_c_pnls = (int)ceil((double)i_ndim / i_c_panelwidth);
  num_r_pnls = (int)ceil((double)i_mdim / i_r_panelwidth);

  /* ------- */
  /* mapping */
  /* ------- */
  switch(e_layout) {
    /* torus wrap mapping on square virtual mesh */
  case wrap: 
    /* loop on cols */
    index1 = 0;

    for(j_v_nd = 0; j_v_nd < i_v_msh_cols; j_v_nd++) {
      /* start loop across virtual nodes */

      /* calculate global col_id of this v-node in the v-mesh */
      g_c_v_nd = j_v_nd + i_v_msh_cols * i_node_col_sb;

      fst_c_pnl = (g_c_v_nd % i_panel_spc ) * i_v_spc +
	g_c_v_nd / i_panel_spc;

      for (c_pnl = fst_c_pnl; c_pnl < num_c_pnls; c_pnl += i_v_dim) {
	/* start loop on v-node (across panels) */
	fst_col = c_pnl * i_c_panelwidth;
	lst_col = fst_col + i_c_panelwidth;
	for(j = fst_col; j < lst_col; j++) {
	  /* start loop on panel (across cols) */
	  if( j >= i_ndim ) {    /* incomplete panel encountered*/
	    break;
	  }
	  
	  /* loop on rows */
	  index2 = index1 * i_lda;
	  for (i_v_nd = 0; i_v_nd < i_v_msh_rows; i_v_nd++) {
	    /* start loop across virtual nodes */

	    /* calculate global row id of this v-node */
	    g_r_v_nd = i_v_nd + i_v_msh_rows * i_node_row_sb;

	    fst_r_pnl = (g_r_v_nd % i_panel_spc ) * i_v_spc +
	      g_r_v_nd / i_panel_spc;

	    for (r_pnl = fst_r_pnl; r_pnl < num_r_pnls; r_pnl += i_v_dim) {
	    /* start loop on v-node (across panles) */
	      fst_row = r_pnl * i_r_panelwidth;
	      lst_row = fst_row + i_r_panelwidth;
	      for (i = fst_row; i < lst_row; i ++) {
		/* start loop on panel (across row) */
		if(i >= i_mdim) {  /* incomplete panel encountered*/
		  break;
		}

		*(p_dst + index2 ) =  (*func) (i, j);  

		index2++;
	      }
	    }
	  }
	  index1++;
	}
      }
    }
    break;
    /* block mapping on square virtual mesh */
  case block:
    prism_v_generror("prism_v_layout: block option not implemented",brief);
    break;
    /* torus wrap mapping on ring */
  case wrap1d:         
    prism_v_generror("prism_v_layout: wrap1d option not implemented",brief);
    break;
    /* block mapping on ring */
  case block1d:         
    prism_v_generror("prism_v_layout: block1d option not implemented",brief);    
    break;
  default:
    prism_v_generror("prism_v_layout: specify the mapping for the matrices",
		     brief);
    break;
  }
}
