/* INCLUDE FILES */
#include "mm.h"

int prism_i_ldim(int i_v_dim, int i_g_dim, int i_blk_nw, int i_first_blk,
		int i_last_blk, int i_panelwidth, int i_offset,
		int i_panel_spc)

     /* 
	COPYRIGHT U.S. GOVERNMENT 
   
	This software is distributed without charge and comes with no
	warranty.

	PRISM routine (version 2.1)

	Please feel free to send questions, comments, and problem
	reports to prism@super.org.
	*/

     /* PURPOSE
	=======
	Function to compute number of rows/columns in node of
	contiguous blocks of a panel torus wrapped matrix

	Some standard cases:





	1) If contiguous blocking or scalar torus wrap is used, set

	i_panelwidth = 1
	i_panel_spc = 1

	Each block contains at least i_g_dim/i_v_dim cols (rows).  If
	i_v_dim does not divide i_g_dim and if there are k leftover
	cols (rows), they are assumed to be spread out so that virtual
	blocks 0, ..., k-1 each have 1 extra col (row).

	2) If physical torus wrap is used, set

	i_panel_spc = i_v_dim/(# nodes in mesh dimension in which
	physical wrapping is done)

	Note that if the mesh is not a square, physical torus wrap can
	only be achieved in one of the two dimensions.  Each virtual
	block contains at least i_panels/i_v_dim complete panels.  If
	i_v_dim does not divide i_panels and if there are k leftover
	panels, they are assumed to be spread out so that panels are
	assigned to blocks in the following way until there are no


	more full panels

	for (j=0; j < i_panel_spc; j++)
	for (i=0; i < i_groups, i++)                                    (*)
	assign a panel to block j+i*i_panelwidth
	      
	If i_panelwidth does not divide i_g_dim, there is an
	incomplete panel left over and it is put in the virtual node
	just after the last node used in (*).
	*/

     /* PARAMETERS
	==========
	i_v_dim:  number of virtual blocks across this dimension of machine
	i_g_dim:  dimension of matrix 
	i_blk_nw:  virtual block in nw corner of mesh
	i_first_blk:  index of first block in node
	i_last_blk:  index of last block in node 
	i_panelwidth:  panel width for torus wrap
	i_offset:  offset to size of first block (has the opposite effect on the 
                   last block)
	i_panel_spc:  spacing between successive panels in virtual blocks
	*/
{
  /* ------------------ */
  /* Internal variables */
  /* ------------------ */
  int
    i,
    i_dim,	      /* returned value */
    i_blk,	      /* block index */
    i_panel_indx,     /* panel index of current block */
    i_panels,	      /* number of full panels */
    i_panels_left,    /* number of leftover panels */
    i_groups,	      /* i_v_dim/i_panel_spc */
    i_v_node_0,
    i_v_node,
    i_spc_0,
    i_spc,
    i_grp_0,
    i_grp,
    i_grp_panel_base
      ;
  
  /* global initializations */
  prism_v_init_var();

  i_panels = (i_g_dim - i_offset)/i_panelwidth;
  if ((i_first_blk >= i_v_dim) || (i_last_blk >= i_v_dim) ||
      (i_blk_nw >= i_v_dim)) {
    prism_v_generror("prism_i_ldim: invalid blocks requested", brief);
  }
  if (i_v_dim%i_panel_spc != 0) {
    prism_v_generror("prism_i_ldim: panel spacing does not divide virtual dim",
		     brief);
  }
    
  /* determine number of complete panels each node has */
  i_dim = ((i_last_blk-i_first_blk+i_v_dim)%i_v_dim + 1)*
    (i_panels/i_v_dim)*i_panelwidth;

  i_panels_left = i_panels%i_v_dim;
  i_groups = i_v_dim/i_panel_spc;

  /* add at most (i_v_dim - 1) extra panels to appropriate blocks */

  for (i = 0; i <= (i_last_blk-i_first_blk+i_v_dim)%i_v_dim; i++) {	
    i_blk = (i_first_blk + i)%i_v_dim;
    /* virtual node containing first panel */
    i_v_node_0 = (i_v_dim - i_blk_nw)%i_v_dim;
    /* virtual node containing current panel */
    i_v_node = (i_blk + i_v_node_0)%i_v_dim;
    /* partition within group containing first panel */
    i_spc_0 = i_v_node_0%i_panel_spc;

    /* partition within group containing current panel */
    i_spc = i_v_node%i_panel_spc;

    /* panel going to corresponding partition of base partition group */
    i_grp_panel_base = (i_v_dim - (i_spc_0 - i_spc)*i_groups)%i_v_dim;

    /* group containing first panel */
    i_grp_0 = i_v_node_0/i_panel_spc;
    /* group containing current panel */
    i_grp = i_v_node/i_panel_spc;
    /* panel index for current block */
    i_panel_indx = (i_grp_panel_base - (i_grp_0 - i_grp) + i_v_dim)%i_v_dim;
    /* add extra panels to first i_panels_left blocks */
    i_dim += ((i_panel_indx < i_panels_left) ? i_panelwidth : 0);

    /* add i_offset to first block */
    if (i_panel_indx == 0) {
      i_dim += i_offset;
    }						    

    /* add any leftover columns/rows to appropriate block */

    if (i_panel_indx == i_panels_left) {
      i_dim += (i_g_dim - i_panels*i_panelwidth - i_offset);
    }
  }
  return i_dim;
}
