/*
 * slave for pvm compile stuff.
 * Keith Moore
 */

#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#ifdef IMA_RIOS
#include <sys/select.h>
#endif
#include "cmd.h"
#define SRVR_ERR(x) (SRC_SRVR | (x))

#ifdef PVM3
#include "pvm3.h"
#endif

char **child_environ;

/***********************************************************************
 *                                                                     *
 *                                errno                                *
 *                                                                     *
 ***********************************************************************/

#include <errno.h>
#ifndef errno
extern int errno;		/* vanilla 4.3 bsd needs this */
#endif

#if defined(IMA_SUN3) || defined(IMA_SUN4) || defined(IMA_SYMM) || \
    defined(IMA_RT)
/*
 * POSIX-style strerror() function
 */

char *
strerror (err)
int err;
{
    extern int sys_nerr;
    extern char *sys_errlist[];
    static char msg[25];

    if (err >= 0 && err < sys_nerr)
	return sys_errlist[err];

    sprintf (msg, "Error %d", err);
    return msg;
}
#endif

/*
 * Given a local errno return a system-independent error code
 * if one exists that corresponds, else return an "unknown error"
 * code.
 */

int
sys_error_code (err)
int err;
{
    switch (err) {
#ifdef EPERM
    case EPERM:		return SRC_SRVR_UNIX |  ERR_EPERM;
#endif
#ifdef ENOENT
    case ENOENT:	return SRC_SRVR_UNIX |  ERR_ENOENT;
#endif
#ifdef ESRCH
    case ESRCH:		return SRC_SRVR_UNIX |  ERR_ESRCH;
#endif
#ifdef EINTR
    case EINTR:		return SRC_SRVR_UNIX |  ERR_EINTR;
#endif
#ifdef EIO
    case EIO:		return SRC_SRVR_UNIX |  ERR_EIO;
#endif
#ifdef ENXIO
    case ENXIO:		return SRC_SRVR_UNIX |  ERR_ENXIO;
#endif
#ifdef E2BIG
    case E2BIG:		return SRC_SRVR_UNIX |  ERR_E2BIG;
#endif
#ifdef ENOEXEC
    case ENOEXEC:	return SRC_SRVR_UNIX |  ERR_ENOEXEC;
#endif
#ifdef EBADF
    case EBADF:		return SRC_SRVR_UNIX |  ERR_EBADF;
#endif
#ifdef ECHILD
    case ECHILD:	return SRC_SRVR_UNIX |  ERR_ECHILD;
#endif
#ifdef EAGAIN
    case EAGAIN:	return SRC_SRVR_UNIX |  ERR_EAGAIN;
#endif
#ifdef ENOMEM
    case ENOMEM:	return SRC_SRVR_UNIX |  ERR_ENOMEM;
#endif
#ifdef EACCES
    case EACCES:	return SRC_SRVR_UNIX |  ERR_EACCES;
#endif
#ifdef EFAULT
    case EFAULT:	return SRC_SRVR_UNIX |  ERR_EFAULT;
#endif
#ifdef ENOTBLK
    case ENOTBLK:	return SRC_SRVR_UNIX |  ERR_ENOTBLK;
#endif
#ifdef EBUSY
    case EBUSY:		return SRC_SRVR_UNIX |  ERR_EBUSY;
#endif
#ifdef EEXIST
    case EEXIST:	return SRC_SRVR_UNIX |  ERR_EEXIST;
#endif
#ifdef EXDEV
    case EXDEV:		return SRC_SRVR_UNIX |  ERR_EXDEV;
#endif
#ifdef ENODEV
    case ENODEV:	return SRC_SRVR_UNIX |  ERR_ENODEV;
#endif
#ifdef ENOTDIR
    case ENOTDIR:	return SRC_SRVR_UNIX |  ERR_ENOTDIR;
#endif
#ifdef EISDIR
    case EISDIR:	return SRC_SRVR_UNIX |  ERR_EISDIR;
#endif
#ifdef EINVAL
    case EINVAL:	return SRC_SRVR_UNIX |  ERR_EINVAL;
#endif
#ifdef ENFILE
    case ENFILE:	return SRC_SRVR_UNIX |  ERR_ENFILE;
#endif
#ifdef EMFILE
    case EMFILE:	return SRC_SRVR_UNIX |  ERR_EMFILE;
#endif
#ifdef ENOTTY
    case ENOTTY:	return SRC_SRVR_UNIX |  ERR_ENOTTY;
#endif
#ifdef ETXTBSY
    case ETXTBSY:	return SRC_SRVR_UNIX |  ERR_ETXTBSY;
#endif
#ifdef EFBIG
    case EFBIG:		return SRC_SRVR_UNIX |  ERR_EFBIG;
#endif
#ifdef ENOSPC
    case ENOSPC:	return SRC_SRVR_UNIX |  ERR_ENOSPC;
#endif
#ifdef ESPIPE
    case ESPIPE:	return SRC_SRVR_UNIX |  ERR_ESPIPE;
#endif
#ifdef EROFS
    case EROFS:		return SRC_SRVR_UNIX |  ERR_EROFS;
#endif
#ifdef EMLINK
    case EMLINK:	return SRC_SRVR_UNIX |  ERR_EMLINK;
#endif
#ifdef EPIPE
    case EPIPE:		return SRC_SRVR_UNIX |  ERR_EPIPE;
#endif
    default:		return SRC_SRVR_UNIX |  ERR_UNKNOWN;
    }
}


/***********************************************************************
 *                                                                     *
 *                                 stat                                *
 *                                                                     *
 ***********************************************************************/

#include <sys/stat.h>

/*
 * some systems don't define the S_ISxxx macros, while others
 * (POSIX compliant) define ONLY these macros.  This code is
 * supposed to define the macros if they are missing.
 * Some systems don't support FIFOs, SOCKETs, or SYMLINKs, so
 * we define a dummy S_ISxxx macro for these which is always false.
 */

#ifndef S_ISBLK
#define S_ISBLK(m) (((m) & S_IFMT) == S_IFBLK)
#endif

#ifndef S_ISCHR
#define S_ISCHR(m) (((m) & S_IFMT) == S_IFCHR)
#endif

#ifndef S_ISDIR
#define S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
#endif

#ifndef S_ISFIFO
#ifdef S_IFIFO
#define S_ISFIFO(m) (((m) & S_IFMT) == S_IFIFO)
#else
#define S_ISFIFO(m) 0			/* never true */
#endif
#endif

#ifndef S_ISLNK
#ifdef S_IFLNK
#define S_ISLNK(m) (((m) & S_IFMT) == S_IFLNK)
#else
#define S_ISLNK(m) 0			/* never true */
#endif
#endif

#ifndef S_ISREG
#define S_ISREG(m) (((m) & S_IFMT) == S_IFREG)
#endif

#ifndef S_ISSOCK
#ifdef S_IFSOCK
#define S_ISSOCK(m) (((m) & S_IFMT) == S_IFSOCK)
#else
#define S_ISSOCK(m) 0			/* never true */
#endif
#endif


/***********************************************************************
 *                                                                     *
 *                                 wait                                *
 *                                                                     *
 ***********************************************************************/

/*
 * Kludge for IBM brain-damage.
 */

#if (defined (IMA_RIOS) || defined(IMA_U370)) && defined(_BSD)
#undef _BSD
#endif

#include <sys/wait.h>

/*
 * Some machines use a union for wait, some use an int.
 * Some take either one, and carefully arrange things so that the bits
 * line up.
 *
 * Older machines don't have the WTERMSIG and WEXITSTATUS macros.
 */

#if defined(IMA_SYMM) || defined(IMA_PMAX) || defined(IMA_UVAX) || \
    defined(IMA_RT) || defined(IMA_NEXT)
typedef union wait waitbuf;
#else
typedef int waitbuf;
#endif

/*
 * magic stuff for machines that don't define WTERMSIG or WEXITSTATUS
 * in <sys/wait.h>.  So far, only the Sequent has needed these, but
 * others might.
 */

#ifndef WTERMSIG
#ifdef w_termsig				/* has union wait */
#define WTERMSIG(status) (((union wait *) &(status))->w_termsig)
#else							/* V7 style */
#define WTERMSIG(status) ((status) & 0x7f)
#endif
#endif

#ifndef WEXITSTATUS
#ifdef w_retcode				/* has union wait */
#define WEXITSTATUS(status) (((union wait *) &(status))->w_retcode)
#else							/* V7 style */
#define WEXITSTATUS(status) (((status) >> 8) & 0xff)
#endif
#endif

/***********************************************************************
 *                                                                     *
 *                               signals                               *
 *                                                                     *
 ***********************************************************************/

#include <signal.h>

#if defined(IMA_SYMM)
#define SIG_RETURN_TYPE int
#else
#define SIG_RETURN_TYPE void
#endif

/***********************************************************************
 *                                                                     *
 *                         interface to utime                          *
 *                                                                     *
 ***********************************************************************/

/*
 * probably everyone still accepts the old-style two-element
 * array as a utimbuf, but on machines that have defined struct utimbuf
 * in <utime.h>, we'll use it just to be sure.
 */

#if defined(IMA_PMAX) || defined(IMA_RIOS) || defined(IMA_SUN3) || \
    defined(IMA_SUN4) || defined(IMA_HP9K)
#include <utime.h>
#define HAVE_STRUCT_UTIMBUF
#endif

int
setfiletime (name, t)
char *name;
time_t t;
{
#ifdef HAVE_STRUCT_UTIMBUF
    struct utimbuf foo;
    memset ((void *) &foo, 0, sizeof (foo));
    foo.actime = t;
    foo.modtime = t;
    return utime (name, &foo);
#else
    time_t foo[2];
    foo[0] = foo[1] = t;
    return utime (name, foo);
#endif
}

/***********************************************************************
 *                                                                     *
 *                           getdtablesize ()                          *
 *                                                                     *
 ***********************************************************************/

#if 0				/* pvm 2.4.1 already supplies this */
#include <unistd.h>

int
getdtablesize ()
{
    return sysconf(_SC_OPEN_MAX);
}
#endif


/***********************************************************************
 *                                                                     *
 *                    end of system-dependent hacks                    *
 *                                                                     *
 ***********************************************************************/

/*
 * Here's some stuff for mucking about with child terminations.
 * Because I'm too lazy to mask signals, I have the SIGCHLD signal
 * handler write a termination event to a pipe.  The main select()
 * loop reads the other end of the pipe and acts accordingly.
 * Thus, child termination events are handled just like child output
 * events.
 */

int signal_pipe[2];

struct death_event {
    int pid;
    int status;
    int signaled;
};

/*
 * this gets tripped when a child dies.
 */

SIG_RETURN_TYPE 
child_sig (sig)
int sig;
{
    struct death_event foo;
    waitbuf w;

#ifdef HAVE_WAITPID
    if ((foo.pid = waitpid (-1, &w, WNOHANG)) > 0) {
#else
    if ((foo.pid = wait3 (&w, WNOHANG, NULL)) > 0) {
#endif
	if (WIFSIGNALED(w)) {
	    foo.signaled = 1;
	    foo.status = WTERMSIG(w);
	}
	else {
	    foo.signaled = 0;
	    foo.status = WEXITSTATUS(w);
	}
	write (signal_pipe[1], &foo, sizeof (foo));
    }
}


/***********************************************************************
 *                                                                     *
 *                               pvm glue                              *
 *                                                                     *
 ***********************************************************************/

/*
 * convenience routine used when sending replies.  Doesn't do much
 * at present, but can be used as a debugging hook.
 */

#ifdef PVM3
void
send_reply (tid)
int tid;
{
    pvm_send (tid, 0);
}
#else
void
send_reply (proc, inum)
char *proc;
int inum;
{
    snd (proc, inum, 0);
}
#endif
/*
 * send a simple yes/no reply with a message
 * a printf-like interface would be better
 */

#ifdef PVM3
void
reply (tid, code, msg)
int tid;
int code;
char *msg;
{
    pvm_initsend (PvmDataDefault);
    putint (code);
    pvm_pkstr (msg);
    send_reply (tid);
}
#else
void
reply (proc, inum, code, msg)
char *proc;
int inum;
int code;
char *msg;
{
    initsend ();
    putint (code);
    putstring (msg);
    send_reply (proc, inum);
}
#endif

int 
putint (i)
int i;
{
#ifdef PVM3
    return pvm_pkint (&i, 1, 1);
#else
    return putnint (&i, 1);
#endif
}

int
putlongint (l)
int l;
{
#ifdef PVM3
    return pvm_pklong (&l, 1, 1);
#else
    return putnlong (&l, 1);
#endif
}

int
getint (ip)
int *ip;
{
#ifdef PVM3
    return pvm_upkint (ip, 1, 1);
#else
    return getnint (ip, 1);
#endif
}

int
getlongint (lp)
long *lp;
{
#ifdef PVM3
    return pvm_upklong (lp, 1, 1);
#else
    return getnlong (lp, 1);
#endif
}

/*
 * given a return code from a pvm function, package it in a universal
 * error code and return it.
 */

int
pvm_error_code (x)
int x;
{
    if (x < 0) {
	x = -x;
	return (SRC_SRVR_PVM | (x & 0xff));
    }
    return x;
}

/***********************************************************************/
/*
 * save a string in heap space
 */

char *
strsave (s)
char *s;
{
    char *p;
    void *malloc ();

    if (s == NULL)
	return NULL;
    if ((p = malloc (strlen (s) + 1)) == NULL)
	return NULL;
    strcpy (p, s);
    return p;
}

/*
 * find a variable name in an environment list, and return its value
 */

char *
find_var (name, length, envp)
char *name;
int length;
char **envp;
{
    int i;
    for (i = 0; envp[i] != NULL; ++i)
	if (strncmp (envp[i], name, length) == 0 && envp[i][length] == '=')
	    return (envp[i] + length + 1);
}

/*
 * expand environment variables in a child process's arguments
 */

char *
expand_vars (arg, env)
char *arg;
char **env;
{
    char buf[1024];
    char *var_name;
    int var_length = -1;
    char *s1, *s2, *d1;

    d1 = buf;
    s1 = arg;
    while (*s1) {
	if (var_length < 0) {
	    if (s1[0] == '$' && s1[1] == '(') {
		s1 += 2;
		var_name = s1;	/* remember start of var_name */
		var_length = 0;
	    }
	    else
		*d1++ = *s1++;
	}
	else {
	    if (*s1 == ')') {
		++s1;
		s2 = find_var (var_name, var_length, env);
		if (s2)
		    while (*d1++ = *s2++);
		var_length = -1;
	    }
	    else {
		++s1;
		var_length++;
	    }
	}
    }
    if (var_length > 0)
	return strsave (arg);
    else
	return strsave (buf);
}


/*
 * spawn a child process.  while it is running, grab its stdout
 * and stderr and send them back as msgs to the caller.
 * Also grab its termination status (when it dies) and send that.
 *
 * XXX need to be able to grab signals from pvm also, in case caller
 * wants us to go away or kill the child, whatever.
 */

void
#ifdef PVM3
do_spawn (tid, child_argv)
int tid;
char **child_argv;
#else
do_spawn (proc, inum, child_argv)
char *proc;
int inum;
char **child_argv;
#endif
{
    int child_stdout[2];	/* pipe for child's stdout */
    int child_stderr[2];	/* pipe for child's stderr */
    int child_pid;		/* child's pid */
    fd_set read_fds;
    fd_set temp_fds;
    int max_fd;
    int nbytes;
    char buf[MSGBUFSIZE];
    struct death_event foo;
    extern char **environ;

    pipe (child_stdout);
    pipe (child_stderr);
    fflush (stderr);		/* make sure stderr buf is empty */
    if ((child_pid = fork ()) == 0) {
	/* child */
	int fd;
	int i;

	close (child_stderr[0]);
	dup2 (child_stderr[1], 2);
	close (child_stderr[1]);

	close (child_stdout[0]);
	dup2 (child_stdout[1], 1);
	close (child_stdout[1]);

	fd = open ("/dev/null", 0); /* stdin gets /dev/null */
	dup2 (fd, 0);
	close (fd);

	for (i = getdtablesize (); i >= 3; --i)
	    close (i);

	environ = child_environ;
	execvp (child_argv[0], child_argv);
	fprintf (stderr, "execvp(%s) failed: %s\n", child_argv[0],
		 strerror (errno));
	fclose (stderr);	/* flush stderr before exiting */
	_exit (127);
    }
    else {
	/* parent */

	close (child_stderr[1]);
	close (child_stdout[1]);
	FD_ZERO (&read_fds);
	FD_SET (child_stderr[0], &read_fds);
	FD_SET (child_stdout[0], &read_fds);
	FD_SET (signal_pipe[0], &read_fds);

#define max(a,b) ((a) > (b) ? (a) : (b))

	max_fd = max (child_stderr[0], child_stdout[0]);
	max_fd = max (max_fd, signal_pipe[0]);
    }
    while (1) {
	temp_fds = read_fds;
	if (select (max_fd + 1, &temp_fds, 0, 0, 0) > 0) {
	    if (FD_ISSET (child_stderr[0], &temp_fds)) {
		nbytes = read (child_stderr[0], buf, sizeof (buf));
		if (nbytes > 0) {
#ifdef PVM3
		    pvm_initsend (PvmDataDefault);
#else
		    initsend ();
#endif
		    /* putf("%i %i %*b", MSG_STDERR, nbytes, nbytes, buf); */
		    putint (MSG_STDERR);
		    putint (nbytes);
#ifdef PVM3
		    pvm_pkbyte (buf, nbytes, 1);
		    send_reply (tid);
#else
		    putbytes (buf, nbytes);
		    send_reply (proc, inum);
#endif
		}
		else if (nbytes == 0) {
		    close (child_stderr[0]);
		    FD_CLR (child_stderr[0], &read_fds);
		}
	    }
	    if (FD_ISSET (child_stdout[0], &temp_fds)) {
		nbytes = read (child_stdout[0], buf, sizeof (buf));
		if (nbytes > 0) {
#ifdef PVM3
		    pvm_initsend (PvmDataDefault);
#else
		    initsend ();
#endif
		    /* putf("%i %i %*b", MSG_STDOUT, nbytes, nbytes, buf); */
		    putint (MSG_STDOUT);
		    putint (nbytes);
#ifdef PVM3
		    pvm_pkbyte (buf, nbytes, 1);
		    send_reply (tid);
#else
		    putbytes (buf, nbytes);
		    send_reply (proc,inum);
#endif
		}
		else if (nbytes == 0) {
		    close (child_stdout[0]);
		    FD_CLR (child_stdout[0], &read_fds);
		}		    
	    }
	    if (FD_ISSET (signal_pipe[0], &temp_fds)) {
		nbytes = read (signal_pipe[0], &foo, sizeof foo);
		if (foo.pid == child_pid) {
#ifdef PVM3
		    pvm_initsend (PvmDataDefault);
#else
		    initsend ();
#endif
		    /* putf ("%i %i", MSG_CMDSTATUS, foo.status); */
		    putint (MSG_PROC_EXIT);
		    putint (foo.status);
#ifdef PVM3
		    send_reply (tid);
#else
		    send_reply (proc,inum);
#endif
		    break;
		}
	    }
	}
    }
    close (child_stdout[0]);
    close (child_stderr[0]);
}


void
work (msgbuf)
char *msgbuf;
{
    int cmd;
    int fd = -1;
    enum {MODE_READ, MODE_WRITE, MODE_NEITHER} fdmode = MODE_NEITHER ;
#ifdef PVM3
    int tid;
#else
    char proc[33];
    int inum;
#endif
    char filename[1024];
    long filesize;
    long modtime;
    struct stat sbuf;
    char buf[FILEBUFSIZE];
    int nbytes;
    int child_argc;
    char *child_argv[100];
    int i, j;
    int x;

    while (1) {
#ifdef PVM3
	if ((x = pvm_recv (-1, 0)) < 0 ||
	    (x = pvm_bufinfo (x, NULL, NULL, &tid) < 0)) {
	    break;
	}
#else
	if ((x = rcv (0)) < 0 || (x = rcvinfo (NULL, NULL, proc, &inum) < 0)) {
	    break;
	}
#endif

	if (msgbuf != NULL && *msgbuf != '\0') {
	    /* putf ("%i %i %*b", MSG_STDERR, strlen(msgbuf),
	             strlen(msgbuf), msgbuf) */
	    putint (MSG_STDERR);
	    putint (strlen (msgbuf));
#ifdef PVM3
	    pvm_pkbyte (msgbuf, strlen (msgbuf), 1);
	    send_reply (tid);
	    reply (tid, MSG_SRVR_EXIT, "");
	    pvm_exit ();
#else
	    putbytes (msgbuf, strlen (msgbuf));
	    send_reply (proc, inum);
	    reply (proc, inum, MSG_SRVR_EXIT, "");
	    leave ();
#endif
	    exit (1);
	}

	getint (&cmd);
	switch (cmd) {
	case CMD_CREATE:
#ifdef PVM3
	    if ((x = pvm_upkstr (filename)) != 0 ||
		(x = getlongint (&filesize)) != 0 ||
		(x = getlongint (&modtime)) != 0) {
		reply (tid, pvm_error_code (x), "invalid argument");
	    }
#else
	    if ((x = getstring (filename)) != 0 ||
		(x = getlongint (&filesize)) != 0 ||
		(x = getlongint (&modtime)) != 0) {
		reply (proc, inum, pvm_error_code (x), "invalid argument");
	    }
#endif
	    if (fd != -1) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "fd already open");
#else
		reply (proc, inum, SRVR_ERR(0), "fd already open");
#endif
	    }
	    if ((fd = open (filename, O_WRONLY|O_CREAT|O_TRUNC, 0600)) < 0) {
#ifdef PVM3
		reply (tid, sys_error_code (errno), strerror (errno));
#else
		reply (proc, inum, sys_error_code (errno), strerror (errno));
#endif
	    }
	    fdmode = MODE_WRITE;
#ifdef PVM3
	    reply (tid, REPL_OK, "");
#else
	    reply (proc, inum, REPL_OK, "");
#endif
	    break;
	case CMD_OPEN:
#ifdef PVM3
	    if ((x = pvm_upkstr (filename)) != 0) {
		reply (tid, pvm_error_code (x), "invalid argument");
	    }
#else
	    if ((x = getstring (filename)) != 0) {
		reply (proc, inum, pvm_error_code (x), "invalid argument");
	    }
#endif
	    if (fd != -1) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "fd already open");
#else
		reply (proc, inum, SRVR_ERR(0), "fd already open");
#endif
	    }
	    if ((fd = open (filename, O_RDONLY, 0)) < 0) {
#ifdef PVM3
		reply (tid, sys_error_code (errno), strerror (errno));
#else
		reply (proc, inum, sys_error_code (errno), strerror (errno));
#endif
	    }
	    fdmode = MODE_READ;
#ifdef PVM3
	    reply (tid, REPL_OK, "");
#else
	    reply (proc, inum, REPL_OK, "");
#endif
	    break;
	case CMD_STAT:
#ifdef PVM3
	    if ((x = pvm_upkstr (filename)) != 0) {
		reply (tid, pvm_error_code (x), "invalid argument");
	    }
#else
	    if ((x = getstring (filename)) != 0) {
		reply (proc, inum, pvm_error_code (x), "invalid argument");
	    }
#endif
	    if (stat (filename, &sbuf) < 0) {
#ifdef PVM3
		reply (tid, sys_error_code (errno), strerror (errno));
#else
		reply (proc, inum, sys_error_code (errno), strerror (errno));
#endif
	    }
#ifdef PVM3
	    pvm_initsend (PvmDataDefault);
#else
	    initsend ();
#endif
	    putint (REPL_STAT);
	    putlongint (sbuf.st_size);
	    putlongint (sbuf.st_mtime);
#ifdef PVM3
	    send_reply (tid);
#else
	    send_reply (proc, inum);
#endif
	    break;
	case CMD_WRITE:
	    if ((x = getint (&nbytes)) != 0) {
#ifdef PVM3
		reply (tid, pvm_error_code (x), "invalid argument");
#else
		reply (proc, inum, pvm_error_code (x), "invalid argument");
#endif
	    }
	    if (nbytes < 0 || nbytes > sizeof (buf)) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "invalid argument");
#else
		reply (proc, inum, SRVR_ERR(0), "invalid argument");
#endif
	    }
	    if (fd < 0 || fdmode != MODE_WRITE) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "fd is not open for writing");
#else
		reply (proc, inum, SRVR_ERR(0), "fd is not open for writing");
#endif
	    }
#ifdef PVM3
	    if ((x = pvm_upkbyte (buf, nbytes, 1)) < 0) {
		reply (tid, pvm_error_code (x), "buffer too small");
	    }
#else
	    if ((x = getbytes (buf, nbytes)) < 0) {
		reply (proc, inum, pvm_error_code (x), "buffer too small");
	    }
#endif
	    if (nbytes == 0) {
		/*
		 * file is complete.  make sure size is correct.
		 *
		 * fstat is cheaper than stat, so check size before
		 * we close the file.
		 *
		 * XXX does this work over NFS, or do we have to close
		 * first to make sure things get flushed?
		 */
		if (fstat (fd, &sbuf) < 0) {
#ifdef PVM3
		    reply (tid, sys_error_code(errno), strerror (errno));
#else
		    reply (proc, inum, sys_error_code(errno),
			   strerror (errno));
#endif
		    continue;
		}
		if (sbuf.st_size != filesize) {
#ifdef PVM3
		    reply (tid, SRVR_ERR(0), "incorrect size");
#else
		    reply (proc, inum, SRVR_ERR(0), "incorrect size");
#endif
		    close (fd);
		    continue;
		}
		if (close (fd) < 0) {
#ifdef PVM3
		    reply (tid, sys_error_code(errno), strerror (errno));
#else
		    reply (proc, inum, sys_error_code(errno),
			   strerror (errno));
#endif
		    fd = -1;
		    continue;
		}
		fd = -1;

		/*
		 * set time of new file.
		 *
		 * XXX this code is for BSD.  Need other code for POSIX,
		 * perhaps for SYSV also.
		 */

		if (setfiletime (filename, modtime) < 0) {
#ifdef PVM3
		    reply (tid, sys_error_code(errno), strerror (errno));
#else
		    reply (proc, inum, sys_error_code(errno),
			   strerror (errno));
#endif
		    continue;
		}
#ifdef PVM3
		reply (tid, REPL_OK, "");
#else
		reply (proc, inum, REPL_OK, "");
#endif
	    }
	    else {
		if (write (fd, buf, nbytes) != nbytes) {
#ifdef PVM3
		    reply (tid, sys_error_code(errno), strerror (errno));
#else
		    reply (proc, inum, sys_error_code(errno),
			   strerror (errno));
#endif
		    close (fd);
		    fd = -1;
		}
		else {
#ifdef PVM3
		    reply (tid, REPL_OK, "");
#else
		    reply (proc, inum, REPL_OK, "");
#endif
		}
	    }
	    break;
	case CMD_READ:
	    if ((x = getint (&nbytes)) != 0) {
#ifdef PVM3
		reply (tid, pvm_error_code (x), "invalid argument");
#else
		reply (proc, inum, pvm_error_code (x), "invalid argument");
#endif
		break;
	    }
	    if (nbytes < 0) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "invalid argument");
#else
		reply (proc, inum, SRVR_ERR(0), "invalid argument");
#endif
		break;
	    }
	    if (fd < 0 || fdmode != MODE_READ) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "fd is not open for reading");
#else
		reply (proc, inum, SRVR_ERR(0), "fd is not open for reading");
#endif
		break;
	    }

	    /* truncate request to fit local buffer */
	    if (nbytes > sizeof (buf))
		nbytes = sizeof (buf);

	    if ((nbytes = read (fd, buf, nbytes)) < 0) {
#ifdef PVM3
		reply (tid, sys_error_code(errno), strerror (errno));
#else
		reply (proc, inum, sys_error_code(errno), strerror (errno));
#endif
		close (fd);
		fd = -1;
		break;
	    }
	    if (nbytes == 0) {
		close (fd);
		fd = -1;
#ifdef PVM3
		pvm_initsend (PvmDataDefault);
#else
		initsend ();
#endif
		putint (REPL_READ);
		putint (0);
#ifdef PVM3
		send_reply (tid);
#else
		send_reply (proc, inum);
#endif
	    }
	    else {
#ifdef PVM3
		pvm_initsend (PvmDataDefault);
#else
		initsend ();
#endif
		putint (REPL_READ);
		putint (nbytes);
#ifdef PVM3
		pvm_pkbyte (buf, nbytes, 1);
		send_reply (tid);
#else
		putbytes (buf, nbytes);
		send_reply (proc, inum);
#endif
	    }
	    break;
	case CMD_SPAWN:
	    if ((x = getint (&child_argc)) != 0) {
#ifdef PVM3
		reply (tid, pvm_error_code (x), "invalid argument");
#else
		reply (proc, inum, pvm_error_code (x), "invalid argument");
#endif
		break;
	    }
	    if (child_argc <= 0) {
#ifdef PVM3
		reply (tid, SRVR_ERR(0), "invalid argument");
#else
		reply (proc, inum, SRVR_ERR(0), "invalid argument");
#endif
		break;
	    }
	    j = 0;
	    for (i = 0; i < child_argc; ++i) {
#ifdef PVM3
		if ((x = pvm_upkstr (buf)) < 0) {
		    reply (tid, pvm_error_code (x), "out of data");
		    break;
		}
#else
		if ((x = getstring (buf)) < 0) {
		    reply (proc, inum, pvm_error_code (x), "out of data");
		    break;
		}
#endif
		child_argv[j++] = expand_vars (buf, child_environ);
	    }
	    if (i == child_argc) {
		child_argv[j] = NULL;
#ifdef PVM3
		do_spawn (tid, child_argv);
#else
		do_spawn (proc, inum, child_argv);
#endif
	    }
	    for (i = 0; i < j; ++i)
		if (child_argv[i])
		    free (child_argv[i]);
	    break;
	case CMD_QUIT:
#ifdef PVM3
	    reply (tid, MSG_SRVR_EXIT, "");
	    reply (tid, REPL_OK, "");
	    pvm_exit ();
#else
	    reply (proc, inum, MSG_SRVR_EXIT, "");
	    reply (proc, inum, REPL_OK, "");
	    leave ();
#endif
	    exit (0);
	default:
	    break;
	}
    }
}


main (argc, argv, envp)
int argc;
char **argv;
char **envp;
{
    char foobar[1024];
#ifdef PVM3
    int tid;
#else
    int inum;
#endif
    char msgbuf[1024];
    int i, j;
    char **new_envp;

    /*
     * set up new environment
     */
    for (i = 0; envp[i]; ++i);	/* count current number of entries  */
    i += 2;			/* leave room for more */
    new_envp = (char **) malloc (i * sizeof (char *)); /* allocate space */

    /* copy existing entries, except for ARCH */
    j = 0;
    for (i = 0, j = 0; envp[i]; ++i)
	if (strncmp (envp[i], "ARCH=", 5) != 0)
	    new_envp[j++] = envp[i];

    sprintf (foobar, "ARCH=%s", ARCHSTR);
    new_envp[j++] = strsave (foobar);
    new_envp[j] = NULL;
    child_environ = new_envp;

#ifdef PVM3
    if ((tid = pvm_mytid ()) < 0) {
	fprintf (stderr, "pvm_mytid() failed .... is pvm 3.x running?\n");
	exit (1);
    }
#else
    if (whoami (foobar, &inum) < 0) {
	fprintf (stderr, "whoami () failed ... is pvm 2.x running?\n");
	exit (1);
    }
#endif
    /*
     * set up the child sig handler stuff
     */
    pipe (signal_pipe);
    signal (SIGCHLD, child_sig);

    /*
     * cd to the right place before starting
     * create directory if necessary
     *
     * XXX for pvm3, is this the right directory?
     */
    sprintf (foobar, "%s/pvm/%s/tmp", getenv("HOME"), ARCHSTR);
    if (chdir (foobar) < 0) {
	mkdir (foobar, 0755);
	errno = 0;
    }
    if (chdir (foobar) < 0) {
	sprintf (msgbuf, "chdir (%s) failed (%s)\n", foobar,
		 strerror (errno));
	fprintf (stderr, "%s", msgbuf);
    }
    else
	*msgbuf = '\0';

    /*
     * okay, do the nitty-gritty
     */
    work (msgbuf);
    exit (0);
}
