/*
 * pvmrsh [ -h host ... ] [ -f file ... ] -c command
 */

#include <stdio.h>
#include <sys/types.h>
#include "cmd.h"

/***********************************************************************
 *                                                                     *
 *                                 stat                                *
 *                                                                     *
 ***********************************************************************/

#include <sys/stat.h>

/*
 * some systems don't define the S_ISxxx macros, while others
 * (POSIX compliant) define ONLY these macros.  This code is
 * supposed to define the macros if they are missing.
 * Some systems don't support FIFOs, SOCKETs, or SYMLINKs, so
 * we define a dummy S_ISxxx macro for these which is always false.
 */

#ifndef S_ISBLK
#define S_ISBLK(m) (((m) & S_IFMT) == S_IFBLK)
#endif

#ifndef S_ISCHR
#define S_ISCHR(m) (((m) & S_IFMT) == S_IFCHR)
#endif

#ifndef S_ISDIR
#define S_ISDIR(m) (((m) & S_IFMT) == S_IFDIR)
#endif

#ifndef S_ISFIFO
#ifdef S_IFIFO
#define S_ISFIFO(m) (((m) & S_IFMT) == S_IFIFO)
#else
#define S_ISFIFO(m) 0			/* never true */
#endif
#endif

#ifndef S_ISLNK
#ifdef S_IFLNK
#define S_ISLNK(m) (((m) & S_IFMT) == S_IFLNK)
#else
#define S_ISLNK(m) 0			/* never true */
#endif
#endif

#ifndef S_ISREG
#define S_ISREG(m) (((m) & S_IFMT) == S_IFREG)
#endif

#ifndef S_ISSOCK
#ifdef S_IFSOCK
#define S_ISSOCK(m) (((m) & S_IFMT) == S_IFSOCK)
#else
#define S_ISSOCK(m) 0			/* never true */
#endif
#endif


/***********************************************************************
 *                                                                     *
 *                               signals                               *
 *                                                                     *
 ***********************************************************************/

#include <signal.h>

#if defined(IMA_SYMM)
#define SIG_RETURN_TYPE int
#else
#define SIG_RETURN_TYPE void
#endif

struct buf {
    struct buf *next;
    int length;
    char *ptr;
};

struct stream {
    struct buf *head;
    struct buf *tail;
};


struct host {
    struct host *next;
    char *name;
    int instno;			/* pvm instance # of this process */
    int flags;
#define H_INIT_FAILED	01	/* initiateM() failed */
#define H_EXITED	02	/* host program completed somehow */
#define H_SEND_FAILED	04	/* send to host failed */
    int exit_status;		/* program completion status */
    int exit_flags;
#define H_EXIT_KILLED 01
#define H_EXIT_COREDUMPED 02
    struct stream errs;		/* storage for error messages */
    struct stream output;	/* storage for output messages */
} *host_list = NULL;

int num_hosts = 0;

struct file {
    struct file *next;
    char *name;
    long size;
    long modtime;
} *file_list = NULL;

char **command_argv = NULL;
int debug = 0;

/*
 * either allocate n bytes of heap space, or complain and exit
 */

static void *
xmalloc (n)
int n;
{
    void *p;
    void *malloc ();

    if ((p = malloc (n)) == NULL) {
	fprintf (stderr, "pvmrsh: malloc(%d) failed...exiting\n");
	exit (1);
    }
    return p;
}

/*
 * either copy string s in managed heap space, or die trying
 */

char *
strsave (s)
char *s;
{
    char *p = (char *) xmalloc (strlen (s) + 1);
    strcpy (p, s);
    return p;
}

/*
 * append a message to the designated stream
 */

static void
append_to_stream (stream, ptr, size)
struct stream *stream;
char *ptr;
int size;
{
    struct buf *newbuf = (struct buf *) xmalloc (sizeof (struct buf));

    newbuf->ptr = xmalloc (size);
    newbuf->length = size;
    newbuf->next = NULL;
    memcpy (newbuf->ptr, ptr, size);
    if (stream->head == NULL) {
	stream->head = newbuf;
	stream->tail = newbuf;
    }
    else {
	stream->tail->next = newbuf;
	stream->tail = newbuf;
    }
}

/*
 * print out the contents of a stream
 */

static void
print_stream (fp, stream)
FILE *fp;
struct stream *stream;
{
    struct buf *ptr;

    for (ptr = stream->head; ptr; ptr = ptr->next)
	fprintf (fp, "%.*s", ptr->length, ptr->ptr);
}

/*
 * add a host to the list of those we want to run this command on
 */

void
add_host (name)
char *name;
{
    struct host *h = (struct host *) xmalloc (sizeof (struct host));

    h->next = host_list;
    h->name = strsave (name);
    h->flags = 0;
    h->instno = 0;
    h->exit_status = 0;
    h->exit_flags = 0;
    host_list = h;
}

struct host *
find_host (inum)
{
    struct host *h;
    for (h = host_list; h != NULL; h = h->next)
	if (h->instno == inum)
	    return h;
    return NULL;
}

/*
 * add a file to the list of those we want to ship to each system
 */

void
add_file (filename)
char *filename;
{
    struct file *f;
    struct stat sbuf;

    if (stat (filename, &sbuf) < 0) {
	fprintf (stderr, "pvmrsh: can't stat %s\n", filename);
	perror ("stat");
	exit (1);
    }
    if (!S_ISREG(sbuf.st_mode)) {
	fprintf (stderr, "pvmrsh: file %s is not a regular file\n", filename);
	exit (1);
    }
    f = (struct file *) xmalloc (sizeof (struct file));

    f->next = file_list;
    f->name = strsave (filename);
    f->size = sbuf.st_size;
    f->modtime = sbuf.st_mtime;
    file_list = f;
}


usage ()
{
    fprintf (stderr, "usage:\n\n");
    fprintf (stderr,
	     "pvmsh -h host [ host ... ] [ -f file ... ] -c command\n");
    exit (1);
}

void
send_file_to_host (f, fd, h)
struct file *f;
int fd;
struct host *h;
{
    char buf[FILEBUFSIZE];
    int nbytes;
    int zero = 0;
    long rsize = 0;
    unsigned long rmodtime;
    int x;

    if (h->flags & (H_INIT_FAILED | H_SEND_FAILED))
	return;

    if (debug)
	fprintf (stderr, "send_file_to_host (%s, %s)\n", f->name, h->name);

    /*
     * decide whether to send file to remote.  Do so if any of:
     * - stat fails for reason ENOENT
     * - file size is different
     * - file mod time is different
     */

    if (((x = cmd_stat (h->instno, f->name, &rsize, &rmodtime)) != 0 &&
	 (x & ERR_CODE_MASK) == ERR_ENOENT) ||
	    (rsize != f->size) || (rmodtime != f->modtime)) {
	if ((x = cmd_create (h->instno, f->name, f->size, f->modtime)) < 0)
	    goto fail;
	while ((nbytes = read (fd, buf, sizeof buf)) > 0)
	    if ((x = cmd_write (h->instno, buf, nbytes)) < 0)
		goto fail;
	if ((x = cmd_write (h->instno, "", 0)) != 0)
	    goto fail;
    }
    else {
	if (debug) 
	    fprintf (stderr, "copy of %s to %s not needed\n", f->name,
		     h->name);
	return;
    }

 succeed:
    if (debug)
	fprintf (stderr, "copy of %s to %s succeeded\n", f->name, h->name);
    return;

 fail:
    fprintf (stderr, "failed to copy file %s to host %s\n",
	     f->name, h->name);
    h->flags |= H_SEND_FAILED;
    return;
}


int
start_command (h, argc, argv)
struct host *h;
char **argv;
{
    int x;
    if ((x = cmd_spawn (h->instno, argc, argv)) < 0) {
	fprintf (stderr, "couldn't spawn %s on %s\n",
		 argv[0], h->name);
	return -1;
    }
    return 0;
}


void
child_exit_hook (proc, instno, status)
char *proc;
int instno;
int status;
{
    struct host *x;
    if (strcmp (proc, "pvmrshd") != 0)
	return;

    for (x = host_list; x != NULL; x = x->next) {
	if (x->instno == instno) {
	    x->exit_status = status;
	    --num_hosts;
	    return;
	}
    }
    return;
}

void
child_stdout (proc, inum, buf, size)
char *proc;
int inum;
char *buf;
int size;
{
    struct host *h = find_host (inum);
#ifdef PVM3
    if (h == NULL) {
	fprintf (stderr, "output from unknown taskid %d\n", inum);
	return;
    }
#endif
    /* fprintf (stdout, "%s: %.*s", h->name, size, buf); */
    append_to_stream (&(h->output), buf, size);
}

void
child_stderr (proc, inum, buf, size)
char *proc;
int inum;
char *buf;
int size;
{
    struct host *h = find_host (inum);
#ifdef PVM3
    if (h == NULL) {
	fprintf (stderr, "errout from unknown taskid %d\n", inum);
	return;
    }
#endif
    /* fprintf (stderr, "%s: %.*s", h->name, size, buf); */
    append_to_stream (&(h->errs), buf, size);
}


void
onintr (sig)
int sig;
{
    struct host *h;
    for (h = host_list ; h ; h=h->next)
	cmd_quit (h->instno);
#ifdef PVM3
    pvm_exit ();
#else
    leave ();
#endif
    exit (1);
}


main (argc, argv)
int argc;
char **argv;
{
    struct host *h;
    struct file *f;
    int command_argc;
    char **command_argv;
    int i;
    int mypvmpid;
    int status;
    int instno;

    if (argc > 1 && strcmp (argv[1], "-debug") == 0) {
	extern int pvmrsh_debug;
	pvmrsh_debug = 1;
	debug = 1;
	--argc;
	++argv;
    }
    if (argc > 1 && strcmp (argv[1], "-h") == 0) {
	--argc;
	++argv;
	while (argc > 1 && *argv[1] != '-') {
	    add_host (argv[1]);
	    --argc;
	    ++argv;
	}
    }
    if (argc > 1 && strcmp (argv[1], "-f") == 0) {
	--argc;
	++argv;
	while (argc > 1 && *argv[1] != '-') {
	    add_file (argv[1]);
	    --argc;
	    ++argv;
	}
    }
    if (argc > 1 && strcmp (argv[1], "-c") == 0) {
	--argc;
	++argv;
	command_argc = argc - 1;
	command_argv = argv + 1;
    }
    if (host_list == NULL) {
	fprintf (stderr, "pvmrsh: no hosts specified\n");
	usage ();
    }
    if (command_argv == NULL) {
	fprintf (stderr, "pvmrsh: no command specified\n");
	usage ();
    }

#if 0
    fprintf (stderr, "argc = %d\n", command_argc);
    for (i = 0; i < command_argc; ++i) {
	fprintf (stderr, "argv[%d] = %s\n", i, command_argv[i]);
    }
    exit (1);
#endif

    /*
     * trap interrupt and clean up
     */

    signal (SIGINT, onintr);

    /*
     * open up a channel to pvmd
     */

#ifdef PVM3
    if (pvm_mytid (&mypvmpid) < 0) {
	fprintf (stderr, "pvmrsh: pvm_mytid() failed.  Is pvmd3 running?\n");
	exit (1);
    }
#else
    if ((mypvmpid = enroll ("pvmrsh")) < 0) {
	fprintf (stderr, "pvmrsh: enroll() failed.  Is pvmd running?\n");
	exit (1);
    }
#endif


    /*
     * set up callbacks for when remote pvmrshd procs send us messages
     */

    cmd_stdout_hook (child_stdout);
    cmd_stderr_hook (child_stderr);
    cmd_exit_hook (child_exit_hook);

    /*
     * start up a process on each host that we care about.
     */

    for (h = host_list; h != NULL; h = h->next) {
	if (cmd_start (h->name, &(h->instno)) != 0) {
#if 0
	    fprintf (stderr, "pvmrsh: couldn't start pvmrshd on %s\n",
		     h->name);
#endif
	    h->flags |= H_INIT_FAILED;
	    h->instno = -1;	/* so we don't get anyone else's output */
	}
#if 0
	fprintf (stderr, "pvmrsh: host %d=%s\n", h->instno, h->name);
#endif	
    }

    for (f = file_list; f != NULL; f = f->next) {
	int fd = open (f->name, 0);
	if (fd < 0) {
	    fprintf (stderr, "can't open %s\n", f->name);
	    exit (1);
	}
	for (h = host_list; h != NULL; h = h->next) {
	    if (h->flags & H_INIT_FAILED)
		continue;
	    lseek (fd, 0L, 0);
	    send_file_to_host (f, fd, h);
	}
	close (fd);
    }

    num_hosts = 0;
    for (h = host_list ; h != NULL; h = h->next) {
	if (h->flags & (H_INIT_FAILED | H_SEND_FAILED))
	    continue;
	start_command (h, command_argc, command_argv);
	++num_hosts;
    }

    while (cmd_wait (&instno) != -1) {
	if (debug)
	    fprintf (stderr, "proc at %d exited\n", instno);
    }

    status = 0;
    for (h = host_list; h != NULL; h = h->next) {

	if (h->output.head) {
	    fprintf (stdout, "\n*** output from %s:\n", h->name);
	    print_stream (stdout, &(h->output));
	    fprintf (stdout, "\n");
	}
	if (h->errs.head || h->flags)
	    fprintf (stderr, "*** error msgs from %s\n", h->name);

	if (h->errs.head)
	    print_stream (stderr, &(h->errs));
	
	if (h->flags & H_INIT_FAILED) {
	    fprintf (stderr, "couldn't start pvmrshd\n", h->name);
	    status = 1;
	}
	else if (h->flags & H_SEND_FAILED) {
	    fprintf (stderr, "couldn't send commands or data\n", h->name);
	    status = 1;
	}
	else if (h->exit_status != 0) {
	    if (h->exit_flags & H_EXIT_KILLED) {
		fprintf (stderr,
			 "remote cmd was killed with signal %d%s\n",
			 h->name, h->exit_status,
			 h->exit_flags & H_EXIT_COREDUMPED ?
			 " (core dumped)" : "");
	    }
	    else {
		fprintf (stderr, "remote cmd exited with status %d\n",
			 h->name, h->exit_status);
	    }
	    status = 1;
	}
#if 0
	else
	    fprintf (stderr, "%s: remote cmd finished ok\n", h->name);
#endif
	if (h->errs.head || h->flags)
	    fprintf (stderr, "\n");
    }
    for (h = host_list; h != NULL; h = h->next) {
	if (h->flags & H_INIT_FAILED)
	    continue;
	cmd_quit (h->instno);
    }
#ifdef PVM3
    pvm_exit ();
#else
    leave ();
#endif
    exit (status);
}
